import 'dart:io';
import 'package:csv/csv.dart';
import 'package:natinfo_flutter/features/pr_locator/data/csv_cache_service.dart';
import 'package:natinfo_flutter/features/pr_locator/domain/entities/pr_point.dart';
import 'package:natinfo_flutter/shared/data_sources/dataset_ids.dart';

class PrRepository {
  final CsvCacheService _cache;
  static Map<String, List<PrPoint>>? _inMemoryCache;

  bool get hasCache => _inMemoryCache != null;

  PrRepository([CsvCacheService? cache]) : _cache = cache ?? CsvCacheService();

  /// Returns whether the CSV data files have been downloaded.
  Future<bool> get isDownloaded async {
    final nat = await _cache.hasDataset(DatasetIds.prNationalAutoroutes);
    final dep = await _cache.hasDataset(DatasetIds.prDepartemental);
    return nat && dep;
  }

  /// Ensures that the CSV files are cached by triggering their download.
  Future<void> cacheData() async {
    await _cache.fetchDataset(DatasetIds.prNationalAutoroutes, force: true);
    await _cache.fetchDataset(DatasetIds.prDepartemental, force: true);
  }

  Future<Map<String, List<PrPoint>>> loadAll([
    void Function(String progress)? onProgress,
  ]) async {
    if (_inMemoryCache != null) {
      return _inMemoryCache!;
    }

    onProgress?.call(
      'Téléchargement du fichier des points de repère sur route national ou autoroute...',
    );
    final natFile = await _cache.fetchDataset(DatasetIds.prNationalAutoroutes);

    onProgress?.call(
      'Téléchargement du fichier des points de repère sur route départementale...',
    );
    final depFile = await _cache.fetchDataset(DatasetIds.prDepartemental);

    onProgress?.call('Traitement des données locales…');
    final national = await _parseCsv(natFile);
    final depart = await _parseCsv(depFile);

    _inMemoryCache = {
      DatasetIds.prNationalAutoroutes: national,
      DatasetIds.prDepartemental: depart,
    };
    return _inMemoryCache!;
  }

  Future<List<PrPoint>> _parseCsv(File file) async {
    final content = await file.readAsString();
    final rows = const CsvToListConverter(
      fieldDelimiter: ',',
      eol: '\n',
      shouldParseNumbers: false,
    ).convert(content);

    if (rows.isEmpty) return [];

    final header = rows.first.cast<String>();
    final typeIndex = header.indexOf('type_de_pr');
    if (typeIndex == -1) return [];

    return rows
        .skip(1)
        .where((r) => r.length > typeIndex && r[typeIndex] == 'PR')
        .map((r) => PrPoint.fromCsv(header, r))
        .toList();
  }
}
