import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/domain/entities/favourite_folder.dart';

/// Nœud de dossier avec enfants et NATINF associés.
class FavouriteFolderNode {
  FavouriteFolderNode({
    required this.folder,
    List<FavouriteFolderNode>? children,
    List<Natinf>? natinfs,
  }) : children = children ?? <FavouriteFolderNode>[],
       natinfs = natinfs ?? <Natinf>[];

  /// Dossier représenté par ce nœud.
  final FavouriteFolder folder;

  /// Sous-dossiers triés.
  final List<FavouriteFolderNode> children;

  /// NATINF contenus dans le dossier, triés par numéro.
  final List<Natinf> natinfs;
}

/// Service utilitaire pour manipuler les dossiers de favoris.
class FavouriteFolderService {
  /// Construit des nœuds hiérarchiques à partir d'une liste plate [folders].
  ///
  /// Les enfants sont triés par nom (ordre alphabétique insensible à la casse).
  /// Les NATINF sont triés par numéro croissant.
  static List<FavouriteFolderNode> buildTree({
    required List<FavouriteFolder> folders,
    required Map<int, List<Natinf>> folderContents,
  }) {
    final childrenMap = <int?, List<FavouriteFolder>>{};
    for (final folder in folders) {
      final parentKey = folder.parentId;
      final list = childrenMap.putIfAbsent(
        parentKey,
        () => <FavouriteFolder>[],
      );
      list.add(folder);
    }

    for (final entry in childrenMap.entries) {
      entry.value.sort(
        (a, b) => a.name.toLowerCase().compareTo(b.name.toLowerCase()),
      );
    }

    List<FavouriteFolderNode> buildLevel(int? parentId) {
      final levelFolders = childrenMap[parentId];
      if (levelFolders == null || levelFolders.isEmpty) {
        return <FavouriteFolderNode>[];
      }

      return levelFolders.map((folder) {
        final natinfs = List<Natinf>.from(
          folderContents[folder.id] ?? const <Natinf>[],
        )..sort(
          (a, b) => (a.numeroNatinf ?? '').compareTo(b.numeroNatinf ?? ''),
        );
        final nodes = buildLevel(folder.id);
        return FavouriteFolderNode(
          folder: folder,
          children: nodes,
          natinfs: natinfs,
        );
      }).toList();
    }

    return buildLevel(null);
  }

  /// Recherche un nœud correspondant à [folderId] dans [nodes].
  static FavouriteFolderNode? findNodeById(
    List<FavouriteFolderNode> nodes,
    int folderId,
  ) {
    for (final node in nodes) {
      if (node.folder.id == folderId) {
        return node;
      }
      final match = findNodeById(node.children, folderId);
      if (match != null) {
        return match;
      }
    }
    return null;
  }
}
