import 'dart:convert';

import 'package:flutter/foundation.dart';
import 'package:http/http.dart' as http;

import '../domain/auth_session.dart';
import 'auth_http_client.dart';
import 'auth_storage.dart';

/// Handles authentication flows (login, refresh, logout) and exposes an
/// authorized [http.Client].
class AuthRepository implements AuthClientDelegate {
  AuthRepository({
    required Uri baseUri,
    required AuthStorage storage,
    http.Client? httpClient,
    Duration accessLeeway = const Duration(seconds: 30),
  }) : _baseUri = baseUri,
       _storage = storage,
       _http = httpClient ?? http.Client(),
       _accessLeeway = accessLeeway;

  final Uri _baseUri;
  final AuthStorage _storage;
  final http.Client _http;
  final Duration _accessLeeway;

  AuthSession? _session;
  bool _loading = false;
  bool _refreshing = false;
  bool _offline = false;
  final List<VoidCallback> _sessionListeners = [];

  /// Current session if any.
  @override
  AuthSession? get currentSession => _session;

  /// Whether auth calls should be bypassed because the app is offline.
  @override
  bool get isOffline => _offline;

  /// Client that injects Authorization and retries after refresh.
  http.Client get authorizedClient => AuthHttpClient(this, _http);

  /// Base URI of the authenticated API.
  Uri get baseUri => _baseUri;

  /// Registers a callback invoked after any session mutation (login/refresh/logout).
  void addSessionListener(VoidCallback listener) {
    _sessionListeners.add(listener);
  }

  /// Loads a persisted session and purges it if expired.
  Future<AuthSession?> loadSession({bool forceOffline = false}) async {
    _offline = forceOffline;
    if (_loading) {
      return _session;
    }
    _loading = true;
    try {
      final stored = await _storage.read();
      if (stored == null) {
        _session = null;
        return null;
      }
      if (stored.isRefreshExpired) {
        await _storage.clear();
        _session = null;
        return null;
      }
      _session = stored;
      return _session;
    } finally {
      _loading = false;
    }
  }

  /// Performs login and stores the resulting session.
  Future<AuthSession> login({
    required String username,
    required String password,
    String? tokenLabel,
  }) async {
    final uri = _endpoint('auth/login/');
    final response = await _http.post(
      uri,
      headers: {'Content-Type': 'application/json'},
      body: jsonEncode({
        'username': username,
        'password': password,
        if (tokenLabel != null) 'token_label': tokenLabel,
      }),
    );
    if (response.statusCode < 200 || response.statusCode >= 300) {
      throw AuthRepositoryException(
        'Login failed',
        statusCode: response.statusCode,
        body: response.body,
      );
    }
    final decoded = jsonDecode(response.body) as Map<String, dynamic>;
    final session = _buildSessionFromPayload(
      decoded,
      fallbackUsername: username,
      fallbackUserId: decoded['user_id'] as int?,
      fallbackHasDocPro: decoded['has_doc_pro'] as bool?,
    );
    final resolved = _offline ? session : await _hydrateProfile(session);
    _session = resolved;
    await _storage.save(resolved);
    _offline = false;
    _notifySessionListeners();
    return resolved;
  }

  /// Refreshes the tokens if possible, returning the new session.
  @override
  Future<AuthSession?> refreshToken() async {
    if (_refreshing) {
      return _session;
    }
    final session = _session;
    if (session == null || session.isRefreshExpired || _offline) {
      return null;
    }
    _refreshing = true;
    try {
      final uri = _endpoint('auth/refresh/');
      final response = await _http.post(
        uri,
        headers: {'Content-Type': 'application/json'},
        body: jsonEncode({'refresh': session.refreshToken}),
      );
      if (response.statusCode < 200 || response.statusCode >= 300) {
        await logout();
        return null;
      }
      final payload = jsonDecode(response.body) as Map<String, dynamic>;
      final merged = _mergeRefreshPayload(session, payload);
      var updated = merged.copyWith(lastRefresh: DateTime.now().toUtc());
      if (!_offline) {
        updated = await _hydrateProfile(updated);
      }
      _session = updated;
      await _storage.save(_session!);
      _notifySessionListeners();
      return _session;
    } finally {
      _refreshing = false;
    }
  }

  /// Clears the local session and attempts server-side revocation when online.
  Future<void> logout() async {
    final uri = _endpoint('auth/logout/');
    try {
      if (!_offline && _session != null) {
        await _http.post(uri);
      }
    } catch (_) {
      // Ignore logout failures to avoid blocking local cleanup.
    }
    _session = null;
    await _storage.clear();
    _notifySessionListeners();
  }

  /// Ensures a valid access token is present, refreshing if needed.
  @override
  Future<void> ensureValidAccessToken() async {
    if (_offline) {
      return;
    }
    _session ??= await _storage.read();
    final session = _session;
    if (session == null) {
      return;
    }
    if (session.isRefreshExpired) {
      await logout();
      return;
    }
    if (session.isAccessExpired ||
        session.willAccessExpireWithin(_accessLeeway)) {
      await refreshToken();
    }
  }

  AuthSession _buildSessionFromPayload(
    Map<String, dynamic> payload, {
    String? fallbackUsername,
    int? fallbackUserId,
    bool? fallbackHasDocPro,
  }) {
    final access = payload['access'] as String?;
    final refresh = payload['refresh'] as String?;
    final refreshExpiresAt = payload['refresh_expires_at'] as String?;
    if (access == null || refresh == null || refreshExpiresAt == null) {
      throw AuthRepositoryException('Missing token fields in response');
    }
    final parsedRefreshExpiry = DateTime.parse(refreshExpiresAt).toUtc();
    return AuthSession.fromTokens(
      access: access,
      refresh: refresh,
      refreshExpiresAt: parsedRefreshExpiry,
      tokenLabel: payload['token_label'] as String?,
      jti: payload['jti'] as String?,
      issuedAt: DateTime.now().toUtc(),
      fallbackUsername:
          payload['username'] as String? ??
          payload['user'] as String? ??
          fallbackUsername,
      fallbackUserId: payload['user_id'] as int? ?? fallbackUserId,
      fallbackHasDocPro: payload['has_doc_pro'] as bool? ?? fallbackHasDocPro,
    );
  }

  Future<AuthSession> _hydrateProfile(AuthSession session) async {
    try {
      final uri = _endpoint('auth/me/');
      final response = await _http.get(
        uri,
        headers: {'Authorization': 'Bearer ${session.accessToken}'},
      );
      if (response.statusCode < 200 || response.statusCode >= 300) {
        return session;
      }
      final decoded = jsonDecode(response.body);
      if (decoded is! Map<String, dynamic>) {
        return session;
      }
      final hasDocPro =
          decoded['has_doc_pro'] as bool? ??
          ((decoded['groups'] is List) &&
              (decoded['groups'] as List).contains('doc_pro'));
      return session.copyWith(
        username:
            decoded['username'] as String? ??
            decoded['user'] as String? ??
            session.username,
        userId:
            decoded['user_id'] as int? ??
            decoded['id'] as int? ??
            session.userId,
        hasDocPro: hasDocPro ?? session.hasDocPro,
      );
    } catch (_) {
      return session;
    }
  }

  AuthSession _mergeRefreshPayload(
    AuthSession current,
    Map<String, dynamic> payload,
  ) {
    final access = payload['access'] as String? ?? current.accessToken;
    final refresh = payload['refresh'] as String? ?? current.refreshToken;
    final refreshExpiresRaw =
        payload['refresh_expires_at'] as String? ??
        current.refreshExpiresAt.toIso8601String();
    final refreshExpiresAt = DateTime.parse(refreshExpiresRaw).toUtc();
    final claims = AuthSession.decodePayload(access);
    return AuthSession.fromTokens(
      access: access,
      refresh: refresh,
      refreshExpiresAt: refreshExpiresAt,
      tokenLabel: payload['token_label'] as String? ?? current.tokenLabel,
      jti: payload['jti'] as String? ?? current.jti,
      issuedAt: DateTime.now().toUtc(),
      fallbackUsername:
          claims['username'] as String? ??
          payload['username'] as String? ??
          current.username,
      fallbackUserId:
          claims['user_id'] as int? ??
          payload['user_id'] as int? ??
          current.userId,
      fallbackHasDocPro:
          claims['has_doc_pro'] as bool? ??
          payload['has_doc_pro'] as bool? ??
          current.hasDocPro,
    );
  }

  Uri _endpoint(String path) {
    final base =
        _baseUri.toString().endsWith('/')
            ? _baseUri.toString().substring(0, _baseUri.toString().length - 1)
            : _baseUri.toString();
    final normalizedPath = path.startsWith('/') ? path.substring(1) : path;
    return Uri.parse('$base/$normalizedPath');
  }

  void _notifySessionListeners() {
    for (final listener in _sessionListeners) {
      listener();
    }
  }
}

/// Describes an authentication failure.
class AuthRepositoryException implements Exception {
  AuthRepositoryException(this.message, {this.statusCode, this.body});

  final String message;
  final int? statusCode;
  final String? body;

  @override
  String toString() =>
      'AuthRepositoryException($message'
      '${statusCode != null ? ', status=$statusCode' : ''}'
      '${body != null ? ', body=$body' : ''})';
}
