import 'dart:io';

import 'package:flutter/material.dart';
import 'package:flutter/gestures.dart';
import 'package:url_launcher/url_launcher_string.dart';
import 'package:package_info_plus/package_info_plus.dart';
import 'package:natinfo_flutter/shared/services/matomo_service.dart';

/// Page affichant les informations sur l'application NATINFo.
class AboutPage extends StatefulWidget {
  const AboutPage({super.key});

  @override
  State<AboutPage> createState() => _AboutPageState();
}

class _AboutPageState extends State<AboutPage> {
  String get appName => Platform.isIOS ? 'NATINFo+' : 'NATINFo+';
  late final Future<PackageInfo> _packageInfoFuture;

  void _launchUrl(String url) async {
    if (await canLaunchUrlString(url)) {
      await launchUrlString(url, mode: LaunchMode.externalApplication);
    }
  }

  String _buildVersionLabel(PackageInfo info) {
    final buildNumber = info.buildNumber.trim();
    if (buildNumber.isEmpty) {
      return info.version;
    }
    return '${info.version} ($buildNumber)';
  }

  @override
  void initState() {
    super.initState();
    _packageInfoFuture = PackageInfo.fromPlatform();
    MatomoService().trackPage(title: 'À propos', path: '/about/');
  }

  @override
  Widget build(BuildContext context) {
    final defaultStyle =
        Theme.of(context).textTheme.bodyLarge?.copyWith(fontSize: 16) ??
        const TextStyle(fontSize: 16);

    return Scaffold(
      appBar: AppBar(title: const Text('À propos')),
      body: Padding(
        padding: const EdgeInsets.all(16.0),
        child: SingleChildScrollView(
          child: Column(
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              // --- Titre principal ------------------------------------------------
              Text(
                'À propos de $appName',
                style: Theme.of(context).textTheme.headlineSmall?.copyWith(
                  fontWeight: FontWeight.bold,
                ),
              ),
              const SizedBox(height: 16),

              // --- Introduction --------------------------------------------------
              RichText(
                text: TextSpan(
                  style: defaultStyle,
                  children: [
                    TextSpan(
                      text:
                          "$appName est une application mobile qui permet de rechercher et de consulter, même hors ligne, les infractions en vigueur de la nomenclature NATINF grâce à l’API de ",
                    ),
                    TextSpan(
                      text: 'NATINFo',
                      style: defaultStyle.copyWith(
                        color: Theme.of(context).colorScheme.secondary,
                        decoration: TextDecoration.underline,
                      ),
                      recognizer:
                          TapGestureRecognizer()
                            ..onTap =
                                () => _launchUrl(
                                  'https://codeberg.org/retiolus/natinfo',
                                ),
                    ),
                    const TextSpan(text: '.'),
                  ],
                ),
              ),

              // --- Que veut dire NATINF ? ---------------------------------------
              const SizedBox(height: 24),
              Text(
                'Que veut dire « NATINF » ?',
                style: Theme.of(
                  context,
                ).textTheme.titleLarge?.copyWith(fontWeight: FontWeight.bold),
              ),
              const SizedBox(height: 8),
              RichText(
                text: TextSpan(
                  style: defaultStyle,
                  children: [
                    const TextSpan(
                      text:
                          '« NATINF » signifie nomenclature des natures d’infraction (NATure d’INFraction). Cette liste officielle, définie par le ministère de la Justice, est utilisée par tous les services judiciaires et plusieurs administrations pour enregistrer les procédures et produire des statistiques.',
                    ),
                    const TextSpan(text: ' '),
                    TextSpan(
                      text: 'En savoir plus',
                      style: defaultStyle.copyWith(
                        color: Theme.of(context).colorScheme.secondary,
                        decoration: TextDecoration.underline,
                      ),
                      recognizer:
                          TapGestureRecognizer()
                            ..onTap =
                                () => _launchUrl(
                                  'https://www.justice.gouv.fr/documentation/ressources/liste-infractions-vigueur-nomenclature-natinf',
                                ),
                    ),
                    const TextSpan(text: '.'),
                  ],
                ),
              ),

              // --- Fonctionnalités principales ----------------------------------
              const SizedBox(height: 24),
              Text(
                'Fonctionnalités principales :',
                style: Theme.of(
                  context,
                ).textTheme.titleLarge?.copyWith(fontWeight: FontWeight.bold),
              ),
              const SizedBox(height: 8),
              Text(
                '• Recherche par numéro NATINF, intitulé, article de loi ou code\n'
                '• Consultation hors ligne grâce au stockage local (données téléchargées lors du premier lancement)\n'
                '• Mise à jour automatique des données à chaque recherche lorsque la connexion est disponible\n'
                '• Historique des NATINF consultées et favoris pour les retrouver plus rapidement\n'
                '• Outils pratiques pour calculer le taux, la vitesse et localiser les points routiers\n'
                '• Thème clair / sombre et couleur personnalisable des badges (contravention / délit / crime)',
                style: defaultStyle,
              ),

              // --- Besoin d'aide ? -----------------------------------------
              const SizedBox(height: 24),
              Text(
                'Besoin d\'aide ?',
                style: Theme.of(
                  context,
                ).textTheme.titleLarge?.copyWith(fontWeight: FontWeight.bold),
              ),
              const SizedBox(height: 8),
              RichText(
                text: TextSpan(
                  style: defaultStyle,
                  children: [
                    const TextSpan(
                      text:
                          'Si vous rencontrez des problèmes ou avez des questions, consultez ',
                    ),
                    TextSpan(
                      text: 'nos issues',
                      style: defaultStyle.copyWith(
                        color: Theme.of(context).colorScheme.secondary,
                        decoration: TextDecoration.underline,
                      ),
                      recognizer:
                          TapGestureRecognizer()
                            ..onTap =
                                () => _launchUrl(
                                  'https://codeberg.org/retiolus/natinfo_flutter/issues',
                                ),
                    ),
                    const TextSpan(text: ' ou envoyez un message à '),
                    TextSpan(
                      text: 'retiolus',
                      style: defaultStyle.copyWith(
                        color: Theme.of(context).colorScheme.secondary,
                        decoration: TextDecoration.underline,
                      ),
                      recognizer:
                          TapGestureRecognizer()
                            ..onTap =
                                () =>
                                    _launchUrl('https://retiolus.net/contact'),
                    ),
                    const TextSpan(text: '.'),
                  ],
                ),
              ),

              // --- Licence et conditions d’utilisation --------------------------
              const SizedBox(height: 24),
              Text(
                'Licences et conditions d’utilisation :',
                style: Theme.of(
                  context,
                ).textTheme.titleLarge?.copyWith(fontWeight: FontWeight.bold),
              ),
              const SizedBox(height: 8),
              RichText(
                text: TextSpan(
                  style: defaultStyle,
                  children: [
                    TextSpan(text: '$appName est développée par '),
                    TextSpan(
                      text: 'retiolus',
                      style: defaultStyle.copyWith(
                        color: Theme.of(context).colorScheme.secondary,
                        decoration: TextDecoration.underline,
                      ),
                      recognizer:
                          TapGestureRecognizer()
                            ..onTap = () => _launchUrl('https://retiolus.net'),
                    ),
                    const TextSpan(text: ' et distribuée sous licence '),
                    TextSpan(
                      text: 'GNU GPL v3 ou ultérieure',
                      style: defaultStyle.copyWith(fontStyle: FontStyle.italic),
                      recognizer:
                          TapGestureRecognizer()
                            ..onTap =
                                () => _launchUrl(
                                  'https://www.gnu.org/licenses/gpl-3.0.html',
                                ),
                    ),
                    TextSpan(
                      text:
                          '.\nVotre utilisation de $appName est soumise aux conditions détaillées sur ',
                    ),
                    TextSpan(
                      text: 'natinfo.app/legal',
                      style: defaultStyle.copyWith(
                        color: Theme.of(context).colorScheme.secondary,
                        decoration: TextDecoration.underline,
                      ),
                      recognizer:
                          TapGestureRecognizer()
                            ..onTap =
                                () => _launchUrl('https://natinfo.app/legal'),
                    ),
                    const TextSpan(text: '.\n\n'),

                    // Sous‑licences -------------------------------------------------
                    const TextSpan(text: 'Autres licences : \n• '),
                    TextSpan(
                      text: 'API de NATINFo',
                      style: defaultStyle.copyWith(
                        color: Theme.of(context).colorScheme.secondary,
                        decoration: TextDecoration.underline,
                      ),
                      recognizer:
                          TapGestureRecognizer()
                            ..onTap =
                                () => _launchUrl('https://natinfo.app/swagger'),
                    ),
                    const TextSpan(
                      text:
                          ' — GNU GPL v3 ou ultérieure\n• Liste des infractions NATINF : ',
                    ),
                    TextSpan(
                      text: 'data.gouv.fr',
                      style: defaultStyle.copyWith(
                        color: Theme.of(context).colorScheme.secondary,
                        decoration: TextDecoration.underline,
                      ),
                      recognizer:
                          TapGestureRecognizer()
                            ..onTap =
                                () => _launchUrl(
                                  'https://www.data.gouv.fr/fr/datasets/liste-des-infractions-en-vigueur-de-la-nomenclature-natinf/',
                                ),
                    ),
                    const TextSpan(
                      text:
                          ' — Licence Ouverte v2.0\n• Données juridiques officielles : ',
                    ),
                    TextSpan(
                      text: 'Legifrance',
                      style: defaultStyle.copyWith(
                        color: Theme.of(context).colorScheme.secondary,
                        decoration: TextDecoration.underline,
                      ),
                      recognizer:
                          TapGestureRecognizer()
                            ..onTap =
                                () => _launchUrl(
                                  'https://www.legifrance.gouv.fr/contenu/pied-de-page/open-data-et-api',
                                ),
                    ),
                    const TextSpan(
                      text: ' — Licence Ouverte v2.0\n• Icônes : ',
                    ),
                    TextSpan(
                      text: 'OpenMoji',
                      style: defaultStyle.copyWith(
                        color: Theme.of(context).colorScheme.secondary,
                        decoration: TextDecoration.underline,
                      ),
                      recognizer:
                          TapGestureRecognizer()
                            ..onTap = () => _launchUrl('https://openmoji.org'),
                    ),
                    const TextSpan(text: ' — CC BY‑SA 4.0.'),
                  ],
                ),
              ),

              // --- Soutenir ou en savoir plus -----------------------------------
              const SizedBox(height: 24),
              Text(
                'Soutenir ou en savoir plus :',
                style: Theme.of(
                  context,
                ).textTheme.titleLarge?.copyWith(fontWeight: FontWeight.bold),
              ),
              const SizedBox(height: 8),
              RichText(
                text: TextSpan(
                  style: defaultStyle,
                  children: [
                    const TextSpan(
                      text:
                          'Vous pouvez en apprendre plus sur mes autres projets sur mon site ',
                    ),
                    TextSpan(
                      text: 'retiolus.net',
                      style: defaultStyle.copyWith(
                        color: Theme.of(context).colorScheme.secondary,
                        decoration: TextDecoration.underline,
                      ),
                      recognizer:
                          TapGestureRecognizer()
                            ..onTap = () => _launchUrl('https://retiolus.net'),
                    ),
                    const TextSpan(text: ', ou me soutenir via '),
                    TextSpan(
                      text: 'BuyMeACoffee',
                      style: defaultStyle.copyWith(
                        color: Theme.of(context).colorScheme.secondary,
                        decoration: TextDecoration.underline,
                      ),
                      recognizer:
                          TapGestureRecognizer()
                            ..onTap =
                                () => _launchUrl(
                                  'https://buymeacoffee.com/retiolus',
                                ),
                    ),
                    const TextSpan(text: '. Merci !'),
                  ],
                ),
              ),

              // --- Version -------------------------------------------------------
              const SizedBox(height: 24),
              FutureBuilder<PackageInfo>(
                future: _packageInfoFuture,
                builder: (context, snapshot) {
                  if (snapshot.connectionState == ConnectionState.waiting) {
                    return const SizedBox();
                  }
                  if (snapshot.hasError) {
                    return const Text(
                      'Version : error',
                      style: TextStyle(
                        fontSize: 16,
                        fontStyle: FontStyle.italic,
                      ),
                    );
                  }
                  return Text(
                    'Version : ${_buildVersionLabel(snapshot.data!)}',
                    style: defaultStyle.copyWith(fontStyle: FontStyle.italic),
                  );
                },
              ),
            ],
          ),
        ),
      ),
    );
  }
}
