import 'dart:convert';
import 'dart:io';

import 'package:flutter_test/flutter_test.dart';
import 'package:natinfo_flutter/shared/data_sources/source_adapter.dart';
import 'package:natinfo_flutter/shared/data_sources/source_loader.dart';
import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';
import 'package:natinfo_flutter/shared/data_sources/update_service.dart';
import 'package:shared_preferences/shared_preferences.dart';

void main() {
  TestWidgetsFlutterBinding.ensureInitialized();

  group('SourceUpdateService', () {
    late Directory tempDir;

    setUp(() async {
      tempDir = await Directory.systemTemp.createTemp('source-cache-test');
      SharedPreferences.setMockInitialValues({});
    });

    tearDown(() async {
      await tempDir.delete(recursive: true);
    });

    test('downloads dataset and stores payload with metadata', () async {
      final loader = _FakeLoader();
      final service = SourceUpdateService(
        loader: loader,
        cacheDirectoryBuilder: () async => tempDir,
        ttl: const Duration(days: 1),
      );

      final result = await service.refreshDataset('dataset');

      expect(result.isUpdated, isTrue);
      expect(result.payloadFile, isNotNull);
      expect(await result.payloadFile!.exists(), isTrue);
      final metadata = File('${result.directory!.path}/metadata.json');
      expect(await metadata.exists(), isTrue);
      final decoded =
          jsonDecode(await metadata.readAsString()) as Map<String, dynamic>;
      expect(decoded['sourceId'], equals('api-source'));
      expect(decoded['schemaVersion'], equals(1));
    });

    test('skips refresh when TTL not expired', () async {
      final loader = _FakeLoader();
      final prefs = await SharedPreferences.getInstance();
      final now = DateTime.now().millisecondsSinceEpoch;
      await prefs.setInt('sources.dataset.lastUpdate', now);

      final service = SourceUpdateService(
        loader: loader,
        cacheDirectoryBuilder: () async => tempDir,
        ttl: const Duration(days: 7),
      );

      final result = await service.refreshDataset('dataset');

      expect(result.status, equals(SourceUpdateStatus.skipped));
      expect(result.reason, SkipReason.ttlFresh);
      expect(loader.loadCount, equals(0));
    });
  });
}

class _FakeLoader implements SourceLoadClient {
  int loadCount = 0;

  @override
  Future<SourceLoadResult> load(
    String dataset, {
    bool allowNetwork = true,
    String? preferredSourceId,
    int? expectedSchemaVersion,
  }) async {
    loadCount += 1;
    final spec = SourceSpec(
      id: 'api-source',
      dataset: dataset,
      name: 'Dataset $dataset',
      type: SourceType.api,
      uri: Uri.parse('https://example.com'),
      scope: dataset,
      priority: 1,
      requiredAtBuild: false,
      requiresNetwork: true,
      schemaVersion: 1,
    );
    return SourceLoadResult(
      spec: spec,
      bytes: const <int>[1, 2, 3],
      integrity: const SourceIntegrity(status: ChecksumStatus.notProvided),
    );
  }
}
