import 'dart:typed_data';

import 'package:flutter/foundation.dart';
import 'package:flutter/services.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:natinfo_flutter/shared/data_sources/source_registry_loader.dart';

void main() {
  TestWidgetsFlutterBinding.ensureInitialized();

  group('loadSourceRegistry', () {
    const registryJson = '''
    {
      "sources": [
        {
          "id": "asset-source",
          "dataset": "demo",
          "name": "Demo dataset",
          "type": "asset",
          "uri": "asset://demo.json",
          "scope": "demo",
          "priority": 1,
          "requiredAtBuild": false,
          "requiresNetwork": false,
          "schemaVersion": 1,
          "checksum": null,
          "checksumAlgo": null
        }
      ]
    }
    ''';

    test('loads registry from bundled asset', () async {
      final registry = await loadSourceRegistry(
        bundle: _FakeAssetBundle({kSourceRegistryAsset: registryJson}),
      );

      expect(registry.all, hasLength(1));
      expect(registry.all.first.id, 'asset-source');
    });

    test('throws when registry asset is missing', () {
      expect(
        () => loadSourceRegistry(bundle: _FakeAssetBundle(const {})),
        throwsA(isA<FlutterError>()),
      );
    });
  });
}

class _FakeAssetBundle extends CachingAssetBundle {
  _FakeAssetBundle(this.assets);

  final Map<String, String> assets;

  @override
  Future<ByteData> load(String key) async {
    final value = _read(key);
    return ByteData.view(Uint8List.fromList(value.codeUnits).buffer);
  }

  @override
  Future<String> loadString(String key, {bool cache = true}) async {
    return _read(key);
  }

  String _read(String key) {
    final value = assets[key];
    if (value == null) throw FlutterError('Asset not found: $key');
    return value;
  }
}
