import 'dart:io';

import 'package:flutter_test/flutter_test.dart';
import 'package:natinfo_flutter/features/settings/data/app_icon_service.dart';

void main() {
  group('AppIconService options', () {
    test('each option has a unique key', () {
      final keys = AppIconService.options.map((option) => option.key).toList();
      expect(keys.toSet().length, keys.length);
    });

    test('each option provides a non-empty preview asset', () {
      for (final option in AppIconService.options) {
        expect(
          option.previewAssetPath,
          isNotEmpty,
          reason: 'Missing asset path for icon ${option.key}',
        );
      }
    });
  });

  group('AppIconService.getOptionByKey', () {
    test('returns matching option when key exists', () {
      final firstOption = AppIconService.options.first;
      final resolved = AppIconService.getOptionByKey(firstOption.key);
      expect(resolved, same(firstOption));
    });

    test('falls back to default option for unknown key', () {
      final resolved = AppIconService.getOptionByKey('unknown-key');
      expect(resolved.key, AppIconService.defaultKey);
    });
  });

  group('iOS alternate icons configuration', () {
    late String infoPlistContents;

    setUpAll(() {
      infoPlistContents = File('ios/Runner/Info.plist').readAsStringSync();
    });

    test('declares each alternate icon key in Info.plist', () {
      final alternateOptions = AppIconService.options.where(
        (option) => option.key != AppIconService.defaultKey,
      );

      for (final option in alternateOptions) {
        expect(
          infoPlistContents.contains('<key>${option.key}</key>'),
          isTrue,
          reason: 'Missing Info.plist entry for icon "${option.key}".',
        );
      }
    });

    test('references matching asset catalog names', () {
      const expectedAssetNames = <String, String>{
        'tricolore_fr': 'AppIconTricoloreFr',
        'blue': 'AppIconBlue',
        'graphite': 'AppIconGraphite',
        'green': 'AppIconGreen',
        'indigo': 'AppIconIndigo',
        'lime': 'AppIconLime',
        'navy': 'AppIconNavy',
        'orange': 'AppIconOrange',
        'pink': 'AppIconPink',
        'purple': 'AppIconPurple',
        'red': 'AppIconRed',
        'teal': 'AppIconTeal',
        'yellow': 'AppIconYellow',
        'pride_ace': 'AppIconPrideAce',
        'pride_bi': 'AppIconPrideBi',
        'pride_pan': 'AppIconPridePan',
        'pride_progress': 'AppIconPrideProgress',
        'pride_rainbow': 'AppIconPrideRainbow',
        'pride_trans': 'AppIconPrideTrans',
      };

      for (final option in AppIconService.options) {
        if (option.key == AppIconService.defaultKey) {
          continue;
        }

        final assetName = expectedAssetNames[option.key];
        expect(
          assetName,
          isNotNull,
          reason: 'Missing asset name mapping for icon "${option.key}".',
        );

        expect(
          infoPlistContents.contains('<string>$assetName</string>'),
          isTrue,
          reason:
              'Info.plist should reference asset "$assetName" for icon "${option.key}".',
        );
      }
    });
  });
}
