import 'package:flutter_test/flutter_test.dart';
import 'package:latlong2/latlong.dart';
import 'package:natinfo_flutter/features/pr_locator/data/location_service.dart';
import 'package:natinfo_flutter/features/pr_locator/data/pr_repository.dart';
import 'package:natinfo_flutter/features/pr_locator/domain/entities/pr_point.dart';
import 'package:natinfo_flutter/features/pr_locator/presentation/viewmodels/pr_locator_viewmodel.dart';
import 'package:natinfo_flutter/shared/data_sources/dataset_ids.dart';

void main() {
  TestWidgetsFlutterBinding.ensureInitialized();

  group('PrLocatorViewModel', () {
    test(
      'recomputes nearest PR after switching dataset even for same location',
      () async {
        final vm = PrLocatorViewModel(
          _FakePrRepository(
            datasets: {
              DatasetIds.prNationalAutoroutes: [
                const PrPoint(route: 'A1', pr: '001', lat: 0, lon: 0),
              ],
              DatasetIds.prDepartemental: [
                const PrPoint(route: 'D1', pr: '100', lat: 0, lon: 0.1),
              ],
            },
          ),
          _FakeLocationService(),
          datasetNamesLoader:
              () async => {
                DatasetIds.prNationalAutoroutes: 'National & Autoroutes',
                DatasetIds.prDepartemental: 'Départemental',
              },
        );

        await _waitUntilReady(vm);
        const tap = LatLng(0, 0);

        expect(
          vm.datasetLabel(vm.selectedDataset),
          equals('National & Autoroutes'),
        );
        vm.handleTap(tap);
        final firstResult = vm.result;
        expect(firstResult, isNotNull);
        expect(vm.markers, hasLength(2));

        vm.selectDataset(DatasetIds.prDepartemental);
        expect(vm.selectedDataset, equals(DatasetIds.prDepartemental));
        expect(vm.markers, isEmpty);
        expect(vm.result, isNull);

        vm.handleTap(tap);
        expect(vm.markers, hasLength(2));
        expect(vm.result, isNotNull);
        expect(vm.result, isNot(equals(firstResult)));
        expect(vm.result, contains('D1'));
      },
    );
  });
}

Future<void> _waitUntilReady(PrLocatorViewModel vm) async {
  final deadline = DateTime.now().add(const Duration(seconds: 1));
  while (vm.state == LoadState.loading) {
    if (DateTime.now().isAfter(deadline)) {
      fail('ViewModel did not finish loading');
    }
    await Future<void>.delayed(const Duration(milliseconds: 10));
  }
  if (vm.state != LoadState.ready) {
    fail('Unexpected view model state: ${vm.state}');
  }
}

class _FakePrRepository extends PrRepository {
  _FakePrRepository({required Map<String, List<PrPoint>> datasets})
    : _datasets = datasets,
      super();

  final Map<String, List<PrPoint>> _datasets;

  @override
  Future<bool> get isDownloaded async => true;

  @override
  bool get hasCache => true;

  @override
  Future<void> cacheData() async {}

  @override
  Future<Map<String, List<PrPoint>>> loadAll([
    void Function(String progress)? onProgress,
  ]) async {
    return _datasets;
  }
}

class _FakeLocationService extends LocationService {
  @override
  Future<void> ensurePermission() async {}

  @override
  Future<LatLng> getCurrentLocation() async => const LatLng(0, 0);

  @override
  Future<LatLng> geocode(String address) async => const LatLng(0, 0);
}
