import 'dart:io';

import 'package:flutter/material.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_download_exception.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_repository.dart';
import 'package:natinfo_flutter/features/natinf/presentation/pages/data_initialization_page.dart';

class _StubNatinfRepository extends NatinfRepository {
  _StubNatinfRepository({
    this.localDataAvailable = false,
    this.throwOnFetch,
    this.throwOnCheck,
  });

  final bool localDataAvailable;
  final Object? throwOnFetch;
  final Object? throwOnCheck;
  int fetchCalls = 0;

  @override
  Future<bool> hasLocalData() async {
    if (throwOnCheck != null) throw throwOnCheck!;
    return localDataAvailable;
  }

  @override
  Future<void> fetchInitialData({
    void Function(String p1)? onStatus,
    void Function(double p1)? onProgress,
    void Function(int p1)? onPhase,
  }) async {
    fetchCalls += 1;
    if (throwOnFetch != null) throw throwOnFetch!;
    onStatus?.call('fetching');
  }
}

void main() {
  TestWidgetsFlutterBinding.ensureInitialized();

  Widget _buildHost(DataInitializationScreen screen) {
    return MaterialApp(home: screen);
  }

  testWidgets('shows prompt and skips when user continues online', (
    WidgetTester tester,
  ) async {
    final repo = _StubNatinfRepository(localDataAvailable: false);
    await tester.pumpWidget(
      _buildHost(
        DataInitializationScreen(
          repository: repo,
          homeBuilder: (_) => const Placeholder(key: Key('home')),
        ),
      ),
    );

    await tester.pumpAndSettle();
    expect(find.textContaining('Télécharger les données'), findsOneWidget);

    final skipFinder = find.text('Continuer avec la recherche en ligne');
    await tester.ensureVisible(skipFinder);
    await tester.tap(skipFinder);
    await tester.pumpAndSettle();

    expect(repo.fetchCalls, 0);
    expect(find.byKey(const Key('home')), findsOneWidget);
  });

  testWidgets('starts download when user accepts', (tester) async {
    final repo = _StubNatinfRepository(localDataAvailable: false);
    await tester.pumpWidget(
      _buildHost(
        DataInitializationScreen(
          repository: repo,
          homeBuilder: (_) => const Placeholder(key: Key('home')),
        ),
      ),
    );

    await tester.pumpAndSettle();
    final downloadFinder = find.text('Télécharger maintenant');
    await tester.ensureVisible(downloadFinder);
    await tester.tap(downloadFinder);
    await tester.pumpAndSettle();

    expect(repo.fetchCalls, 1);
    expect(find.byKey(const Key('home')), findsOneWidget);
  });

  testWidgets('shows error and allows skip when download fails', (
    WidgetTester tester,
  ) async {
    final repo = _StubNatinfRepository(
      localDataAvailable: false,
      throwOnFetch: Exception('fail'),
    );
    await tester.pumpWidget(
      _buildHost(
        DataInitializationScreen(
          repository: repo,
          homeBuilder: (_) => const Placeholder(key: Key('home')),
        ),
      ),
    );

    await tester.pumpAndSettle();
    final downloadFinder = find.text('Télécharger maintenant');
    await tester.ensureVisible(downloadFinder);
    await tester.tap(downloadFinder);
    await tester.pumpAndSettle();

    expect(find.text('Réessayer'), findsOneWidget);
    expect(find.text('Continuer avec la recherche en ligne'), findsWidgets);
  });

  testWidgets('surfaces network diagnostics when download fails', (
    WidgetTester tester,
  ) async {
    final repo = _StubNatinfRepository(
      localDataAvailable: false,
      throwOnFetch: const NatinfDownloadException(
        type: NatinfDownloadIssueType.network,
        message: 'Timeout (60s)',
      ),
    );
    await tester.pumpWidget(
      _buildHost(
        DataInitializationScreen(
          repository: repo,
          homeBuilder: (_) => const Placeholder(key: Key('home')),
        ),
      ),
    );

    await tester.pumpAndSettle();
    final downloadFinder = find.text('Télécharger maintenant');
    await tester.ensureVisible(downloadFinder);
    await tester.tap(downloadFinder);
    await tester.pumpAndSettle();

    expect(find.textContaining('Erreur réseau'), findsOneWidget);
    expect(find.textContaining('Timeout (60s)'), findsOneWidget);
  });

  testWidgets('surfaces storage diagnostics when init check fails', (
    WidgetTester tester,
  ) async {
    final repo = _StubNatinfRepository(
      localDataAvailable: false,
      throwOnCheck: const FileSystemException('permission denied'),
    );
    await tester.pumpWidget(
      _buildHost(
        DataInitializationScreen(
          repository: repo,
          homeBuilder: (_) => const Placeholder(key: Key('home')),
        ),
      ),
    );

    await tester.pumpAndSettle();

    expect(find.textContaining('stockage'), findsOneWidget);
    expect(find.text('Réessayer'), findsOneWidget);
  });
}
