import 'package:flutter/material.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:azlistview/azlistview.dart';
import 'package:natinfo_flutter/features/glossary/data/acronym_service.dart';
import 'package:natinfo_flutter/features/glossary/domain/entities/acronym.dart';
import 'package:natinfo_flutter/features/glossary/presentation/glossaire_page.dart';
import 'package:scrollable_positioned_list/scrollable_positioned_list.dart';
import 'package:shared_preferences/shared_preferences.dart';

class _FakeAcronymService extends AcronymService {
  _FakeAcronymService(this.items);

  final List<Acronym> items;

  @override
  Future<List<Acronym>> search(String query) async => items;
}

void main() {
  TestWidgetsFlutterBinding.ensureInitialized();

  setUp(() {
    SharedPreferences.setMockInitialValues({'matomoTrackingEnabled': false});
  });

  testWidgets('GlossairePage shows alphabet bar and jumps to letter', (
    tester,
  ) async {
    final service = _FakeAcronymService(_buildAcronyms());
    final listener = ItemPositionsListener.create();

    await tester.pumpWidget(
      MaterialApp(
        home: GlossairePage(service: service, positionsListener: listener),
      ),
    );

    await tester.pumpAndSettle();

    final indexBarFinder = find.byType(IndexBar);
    expect(indexBarFinder, findsOneWidget);
    final indexBar = tester.widget<IndexBar>(indexBarFinder);
    expect(indexBar.data, containsAll(<String>['A', 'B', 'C']));
    expect(listener.itemPositions.value, isNotEmpty);

    final initialIndex = _firstVisibleIndex(listener);
    expect(initialIndex, lessThan(20));

    await _tapIndexBarLetter(tester, indexBarFinder, 'C');
    await tester.pumpAndSettle();

    final currentIndex = _firstVisibleIndex(listener);
    expect(
      currentIndex,
      greaterThanOrEqualTo(40),
      reason: 'positions: ${listener.itemPositions.value.toList()}',
    );
  });
}

List<Acronym> _buildAcronyms() {
  final list = <Acronym>[];
  for (var letter in ['A', 'B', 'C']) {
    for (var i = 0; i < 20; i++) {
      list.add(
        Acronym(
          acronym: '$letter item $i',
          meaning: 'Meaning $letter$i',
          source: 'Wikipedia',
        ),
      );
    }
  }
  return list;
}

int _firstVisibleIndex(ItemPositionsListener listener) {
  final positions = listener.itemPositions.value;
  final visible = positions
      .where((p) => p.itemTrailingEdge > 0)
      .toList(growable: false);
  visible.sort((a, b) => a.index.compareTo(b.index));
  return visible.isEmpty ? 0 : visible.first.index;
}

Future<void> _tapIndexBarLetter(
  WidgetTester tester,
  Finder barFinder,
  String letter,
) async {
  final bar = tester.widget<IndexBar>(barFinder);
  final tagIndex = bar.data.indexOf(letter);
  expect(tagIndex, isNonNegative);
  final topLeft = tester.getTopLeft(barFinder);
  final size = tester.getSize(barFinder);
  final itemHeight = size.height / bar.data.length;
  final positionY = topLeft.dy + itemHeight * (tagIndex + 0.5);

  final azListViewState = tester.state(find.byType(AzListView)) as dynamic;
  final dragNotifier = azListViewState.dragListener as IndexBarDragNotifier;

  dragNotifier.dragDetails.value = IndexBarDragDetails(
    action: IndexBarDragDetails.actionDown,
    index: tagIndex,
    tag: letter,
    globalPositionY: positionY,
    localPositionY: positionY - topLeft.dy,
  );
  await tester.pump();
  dragNotifier.dragDetails.value = IndexBarDragDetails(
    action: IndexBarDragDetails.actionEnd,
    index: tagIndex,
    tag: letter,
    globalPositionY: positionY,
    localPositionY: positionY - topLeft.dy,
  );
  await tester.pump();
}
