import 'dart:convert';

import 'package:flutter/material.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:http/http.dart' as http;
import 'package:http/testing.dart';
import 'package:natinfo_flutter/features/auth/data/auth_repository.dart';
import 'package:natinfo_flutter/features/auth/data/auth_storage.dart';
import 'package:natinfo_flutter/features/auth/domain/auth_session.dart';
import 'package:natinfo_flutter/features/auth/presentation/auth_provider.dart';
import 'package:natinfo_flutter/features/auth/presentation/login_page.dart';
import 'package:provider/provider.dart';

void main() {
  testWidgets('pré-remplit le libellé du jeton avec NATINFo+', (tester) async {
    final repository = _FakeAuthRepository();
    final provider = AuthProvider(repository);

    await tester.pumpWidget(
      ChangeNotifierProvider<AuthProvider>.value(
        value: provider,
        child: const MaterialApp(home: LoginPage()),
      ),
    );

    final tokenField = tester.widget<TextFormField>(
      find.byType(TextFormField).at(2),
    );

    expect(tokenField.controller?.text, 'NATINFo+');
  });

  testWidgets('utilise NATINFo+ quand le libellé est laissé vide', (
    tester,
  ) async {
    final repository = _FakeAuthRepository();
    final provider = AuthProvider(repository);

    await tester.pumpWidget(
      ChangeNotifierProvider<AuthProvider>.value(
        value: provider,
        child: const MaterialApp(home: LoginPage()),
      ),
    );

    await tester.enterText(find.bySemanticsLabel('Nom d’utilisateur'), 'alice');
    await tester.enterText(find.bySemanticsLabel('Mot de passe'), 'password');
    await tester.enterText(
      find.bySemanticsLabel('Libellé du jeton (optionnel)'),
      '',
    );

    await tester.tap(find.text('Se connecter'));
    await tester.pumpAndSettle();

    expect(repository.lastTokenLabel, 'NATINFo+');
    expect(repository.lastUsername, 'alice');
    expect(repository.lastPassword, 'password');
  });
}

class _FakeAuthRepository extends AuthRepository {
  _FakeAuthRepository()
    : super(
        baseUri: Uri.parse('https://example.com/api'),
        storage: InMemoryAuthStorage(),
        httpClient: MockClient((request) async => http.Response('{}', 200)),
      );

  String? lastUsername;
  String? lastPassword;
  String? lastTokenLabel;

  @override
  Future<AuthSession> login({
    required String username,
    required String password,
    String? tokenLabel,
  }) async {
    lastUsername = username;
    lastPassword = password;
    lastTokenLabel = tokenLabel;
    return AuthSession.fromTokens(
      access: _token(expMinutes: 60, userId: 1, username: username),
      refresh: 'refresh',
      refreshExpiresAt: DateTime.now().toUtc().add(const Duration(days: 7)),
      fallbackUsername: username,
      fallbackUserId: 1,
      fallbackHasDocPro: false,
    ).copyWith(tokenLabel: tokenLabel);
  }
}

String _token({
  required int expMinutes,
  required int userId,
  required String username,
}) {
  final header = base64UrlEncode(utf8.encode('{"alg":"HS256","typ":"JWT"}'));
  final payload = base64UrlEncode(
    utf8.encode(
      '{"exp":${DateTime.now().toUtc().add(Duration(minutes: expMinutes)).millisecondsSinceEpoch ~/ 1000},"user_id":$userId,"username":"$username"}',
    ),
  );
  return '$header.$payload.signature';
}
