import 'package:flutter/material.dart';
import 'package:flutter_test/flutter_test.dart';
import 'dart:convert';
import 'package:http/http.dart' as http;
import 'package:http/testing.dart';
import 'package:natinfo_flutter/app/navigation/app_drawer.dart';
import 'package:natinfo_flutter/features/auth/data/auth_repository.dart';
import 'package:natinfo_flutter/features/auth/data/auth_storage.dart';
import 'package:natinfo_flutter/features/auth/domain/auth_session.dart';
import 'package:natinfo_flutter/features/auth/presentation/auth_provider.dart';
import 'package:provider/provider.dart';

void main() {
  group('AppDrawer auth states', () {
    late InMemoryAuthStorage storage;
    late AuthRepository repository;

    setUp(() {
      storage = InMemoryAuthStorage();
      repository = AuthRepository(
        baseUri: Uri.parse('https://natinfo.app/api'),
        storage: storage,
        httpClient: MockClient((request) async {
          return http.Response('{}', 200);
        }),
      );
    });

    testWidgets('shows login entry when signed out', (tester) async {
      final provider = AuthProvider(repository);
      await provider.bootstrap();

      await tester.pumpWidget(
        MultiProvider(
          providers: [
            ChangeNotifierProvider<AuthProvider>.value(value: provider),
          ],
          child: const MaterialApp(home: AppDrawer()),
        ),
      );

      expect(find.text('Connexion'), findsOneWidget);
      expect(find.text('Mon compte'), findsNothing);
    });

    testWidgets('shows account and logout when signed in', (tester) async {
      final session = AuthSession.fromTokens(
        access: _token(expMinutes: 60, userId: 1, username: 'alice'),
        refresh: 'refresh',
        refreshExpiresAt: DateTime.now().toUtc().add(const Duration(days: 7)),
      );
      await storage.save(session);
      final provider = AuthProvider(repository);
      await provider.bootstrap();

      await tester.pumpWidget(
        MultiProvider(
          providers: [
            ChangeNotifierProvider<AuthProvider>.value(value: provider),
          ],
          child: const MaterialApp(home: AppDrawer()),
        ),
      );

      expect(find.textContaining('Connecté en tant que'), findsOneWidget);
      expect(find.text('Mon compte'), findsOneWidget);
      expect(find.text('Déconnexion'), findsOneWidget);
      expect(find.text('Connexion'), findsNothing);
    });
  });
}

String _token({
  required int expMinutes,
  required int userId,
  required String username,
}) {
  final header = base64UrlEncode(utf8.encode('{"alg":"HS256","typ":"JWT"}'));
  final payload = base64UrlEncode(
    utf8.encode(
      '{"exp":${DateTime.now().toUtc().add(Duration(minutes: expMinutes)).millisecondsSinceEpoch ~/ 1000},"user_id":$userId,"username":"$username"}',
    ),
  );
  return '$header.$payload.signature';
}
