import 'dart:convert';
import 'dart:io';

import 'package:path_provider/path_provider.dart';
import 'package:shared_preferences/shared_preferences.dart';

import 'package:natinfo_flutter/shared/data_sources/source_adapter.dart';
import 'package:natinfo_flutter/shared/data_sources/source_loader.dart';
import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';

/// Handles manual refresh of external datasets with TTL-based skipping logic.
class SourceUpdateService {
  SourceUpdateService({
    required SourceLoadClient loader,
    Duration ttl = const Duration(days: 7),
    Future<Directory> Function()? cacheDirectoryBuilder,
    SharedPreferences? preferences,
  }) : _loader = loader,
       _ttl = ttl,
       _cacheDirectoryBuilder =
           cacheDirectoryBuilder ?? _defaultCacheDirectoryBuilder,
       _preferences = preferences;

  final SourceLoadClient _loader;
  final Duration _ttl;
  final Future<Directory> Function() _cacheDirectoryBuilder;
  SharedPreferences? _preferences;
  Directory? _cacheDir;

  /// Refreshes [dataset], downloading its payload and storing it in a versioned
  /// directory.
  Future<SourceUpdateResult> refreshDataset(
    String dataset, {
    bool force = false,
    bool allowNetwork = true,
    String? preferredSourceId,
    int? expectedSchemaVersion,
  }) async {
    if (!force && !await _isStale(dataset)) {
      final cachedPayload = await getCachedPayload(dataset);
      return SourceUpdateResult.skipped(
        dataset,
        reason: SkipReason.ttlFresh,
        directory: cachedPayload?.parent,
        payloadFile: cachedPayload,
      );
    }

    final loadResult = await _loader.load(
      dataset,
      allowNetwork: allowNetwork,
      preferredSourceId: preferredSourceId,
      expectedSchemaVersion: expectedSchemaVersion,
    );
    final directory = await _prepareVersionDirectory(
      dataset,
      loadResult.spec.schemaVersion,
    );

    final payloadFile = File('${directory.path}/payload.bin');
    await payloadFile.writeAsBytes(loadResult.bytes, flush: true);
    await _writeMetadata(directory, loadResult);

    final prefs = await _preferencesInstance();
    final now = DateTime.now();
    await prefs.setInt(_lastUpdateKey(dataset), now.millisecondsSinceEpoch);
    await prefs.setInt(
      _schemaVersionKey(dataset),
      loadResult.spec.schemaVersion,
    );
    await prefs.setString(_sourceIdKey(dataset), loadResult.spec.id);

    return SourceUpdateResult.updated(
      dataset: dataset,
      directory: directory,
      payloadFile: payloadFile,
      spec: loadResult.spec,
    );
  }

  /// Ensures the cache directory exists.
  Future<Directory> _ensureCacheRoot() async {
    if (_cacheDir != null) return _cacheDir!;
    final dir = await _cacheDirectoryBuilder();
    if (!await dir.exists()) {
      await dir.create(recursive: true);
    }
    _cacheDir = dir;
    return _cacheDir!;
  }

  Future<bool> _isStale(String dataset) async {
    final prefs = await _preferencesInstance();
    final millis = prefs.getInt(_lastUpdateKey(dataset));
    if (millis == null) return true;
    final lastUpdate = DateTime.fromMillisecondsSinceEpoch(millis);
    final age = DateTime.now().difference(lastUpdate);
    return age >= _ttl;
  }

  Future<Directory> _prepareVersionDirectory(
    String dataset,
    int schemaVersion,
  ) async {
    final root = await _ensureCacheRoot();
    final dir = Directory('${root.path}/$dataset/v$schemaVersion');
    if (await dir.exists()) {
      await dir.delete(recursive: true);
    }
    await dir.create(recursive: true);
    return dir;
  }

  Future<void> _writeMetadata(Directory dir, SourceLoadResult result) async {
    final metadata = {
      'sourceId': result.spec.id,
      'dataset': result.spec.dataset,
      'schemaVersion': result.spec.schemaVersion,
      'uri': result.spec.uri.toString(),
      'fetchedAt': DateTime.now().toIso8601String(),
      'checksumStatus': result.integrity.status.name,
      'checksumExpected': result.integrity.expected,
      'checksumActual': result.integrity.actual,
      'checksumAlgorithm': result.integrity.algorithm,
      // TODO: Enforce checksum verification before persisting to disk.
    };
    final metadataFile = File('${dir.path}/metadata.json');
    await metadataFile.writeAsString(
      const JsonEncoder.withIndent('  ').convert(metadata),
      flush: true,
    );
  }

  Future<SharedPreferences> _preferencesInstance() async {
    if (_preferences != null) return _preferences!;
    _preferences = await SharedPreferences.getInstance();
    return _preferences!;
  }

  String _lastUpdateKey(String dataset) => 'sources.$dataset.lastUpdate';

  String _schemaVersionKey(String dataset) => 'sources.$dataset.schemaVersion';

  String _sourceIdKey(String dataset) => 'sources.$dataset.sourceId';

  static Future<Directory> _defaultCacheDirectoryBuilder() async {
    final dir = await getApplicationSupportDirectory();
    return Directory('${dir.path}/cache/sources');
  }

  Future<File?> getCachedPayload(String dataset) async {
    final directory = await getCachedDirectory(dataset);
    if (directory == null) return null;
    final payload = File('${directory.path}/payload.bin');
    if (await payload.exists()) {
      return payload;
    }
    return null;
  }

  Future<Directory?> getCachedDirectory(String dataset) async {
    return _locateDatasetDirectory(dataset);
  }

  Future<int?> _cachedSchemaVersion(String dataset) async {
    final prefs = await _preferencesInstance();
    return prefs.getInt(_schemaVersionKey(dataset));
  }

  Future<SourceCacheMetadata?> getMetadata(String dataset) async {
    final prefs = await _preferencesInstance();
    final last = prefs.getInt(_lastUpdateKey(dataset));
    final schema = prefs.getInt(_schemaVersionKey(dataset));
    final sourceId = prefs.getString(_sourceIdKey(dataset));
    final payload = await getCachedPayload(dataset);
    if (last == null && schema == null && sourceId == null && payload == null) {
      return null;
    }
    DateTime? computedLast =
        last != null ? DateTime.fromMillisecondsSinceEpoch(last) : null;
    int? size;
    if (payload != null && await payload.exists()) {
      size = await payload.length();
      computedLast ??= await payload.lastModified();
    }
    return SourceCacheMetadata(
      dataset: dataset,
      sourceId: sourceId,
      schemaVersion: schema,
      lastUpdated: computedLast,
      sizeBytes: size,
      payloadPath: payload?.path,
    );
  }

  Future<void> clearDataset(String dataset) async {
    final directory = await getCachedDirectory(dataset);
    if (directory != null && await directory.exists()) {
      await directory.delete(recursive: true);
    }
    final prefs = await _preferencesInstance();
    await prefs.remove(_lastUpdateKey(dataset));
    await prefs.remove(_schemaVersionKey(dataset));
    await prefs.remove(_sourceIdKey(dataset));
  }

  Future<Directory?> _locateDatasetDirectory(String dataset) async {
    final root = await _ensureCacheRoot();
    final schemaVersion = await _cachedSchemaVersion(dataset);
    if (schemaVersion != null) {
      final dir = Directory('${root.path}/$dataset/v$schemaVersion');
      if (await dir.exists()) {
        return dir;
      }
    }

    final datasetDir = Directory('${root.path}/$dataset');
    if (!await datasetDir.exists()) {
      return null;
    }

    final candidates =
        await datasetDir
            .list()
            .where((entity) => entity is Directory)
            .cast<Directory>()
            .toList();
    if (candidates.isEmpty) return null;
    candidates.sort((a, b) => b.path.compareTo(a.path));
    for (final dir in candidates) {
      final payload = File('${dir.path}/payload.bin');
      if (await payload.exists()) {
        return dir;
      }
    }
    return null;
  }
}

/// Result of a dataset refresh attempt.
class SourceUpdateResult {
  SourceUpdateResult.updated({
    required this.dataset,
    required this.directory,
    required this.payloadFile,
    required this.spec,
  }) : status = SourceUpdateStatus.updated,
       reason = null;

  SourceUpdateResult.skipped(
    this.dataset, {
    required this.reason,
    this.directory,
    this.payloadFile,
    this.spec,
  }) : status = SourceUpdateStatus.skipped,
       assert(reason != null);

  final SourceUpdateStatus status;
  final SkipReason? reason;
  final String dataset;
  final Directory? directory;
  final File? payloadFile;
  final SourceSpec? spec;

  bool get isUpdated => status == SourceUpdateStatus.updated;
}

/// Status of a refresh attempt.
enum SourceUpdateStatus { updated, skipped }

/// Why a refresh was skipped.
enum SkipReason { ttlFresh }

/// Metadata about a cached dataset payload.
class SourceCacheMetadata {
  SourceCacheMetadata({
    required this.dataset,
    this.sourceId,
    this.schemaVersion,
    this.lastUpdated,
    this.sizeBytes,
    this.payloadPath,
  });

  final String dataset;
  final String? sourceId;
  final int? schemaVersion;
  final DateTime? lastUpdated;
  final int? sizeBytes;
  final String? payloadPath;
}
