import 'dart:convert';

/// A declared source of data for a specific dataset.
///
/// Instances are created from the JSON spec bundled in assets/source_registry.json.
class SourceSpec {
  SourceSpec({
    required this.id,
    required this.dataset,
    required this.name,
    this.license,
    required this.type,
    required this.uri,
    required this.scope,
    required this.priority,
    required this.requiredAtBuild,
    required this.requiresNetwork,
    required this.schemaVersion,
    this.checksum,
    this.checksumAlgo,
    this.userEditable = true,
  }) {
    if (priority < 1) {
      throw ArgumentError.value(priority, 'priority', 'must be >= 1');
    }
  }

  /// Unique identifier for the source.
  final String id;

  /// Dataset identifier this source provides.
  final String dataset;

  /// Human-readable name for the dataset.
  final String name;

  /// Optional license text describing the dataset terms.
  final String? license;

  /// How the source should be accessed.
  final SourceType type;

  /// Base URI of the source.
  final Uri uri;

  /// Scope of the dataset (ex. national, categories).
  final String scope;

  /// Lower numbers are preferred when multiple sources are available.
  final int priority;

  /// Whether the source must be available for the build to succeed.
  final bool requiredAtBuild;

  /// Whether using this source requires network access.
  final bool requiresNetwork;

  /// Schema version of the dataset provided by this source.
  final int schemaVersion;

  /// Optional checksum for the source payload.
  final String? checksum;

  /// Algorithm used to compute [checksum].
  final String? checksumAlgo;

  /// Whether this source can be replaced by a user-defined URL.
  final bool userEditable;

  factory SourceSpec.fromMap(Map<String, dynamic> map) {
    if (!map.containsKey('id') ||
        !map.containsKey('dataset') ||
        !map.containsKey('name') ||
        !map.containsKey('type') ||
        !map.containsKey('uri') ||
        !map.containsKey('scope') ||
        !map.containsKey('priority') ||
        !map.containsKey('requiredAtBuild') ||
        !map.containsKey('requiresNetwork') ||
        !map.containsKey('schemaVersion')) {
      throw const FormatException('Missing required source fields');
    }

    final typeValue = map['type'];
    final parsedType = SourceTypeExtension.fromString('$typeValue');
    final uriString = map['uri']?.toString();
    final parsedUri = Uri.tryParse(uriString ?? '');
    if (parsedUri == null) {
      throw const FormatException('Invalid or missing URI');
    }

    return SourceSpec(
      id: map['id'] as String,
      dataset: map['dataset'] as String,
      name: map['name'] as String,
      license: map['license'] as String?,
      type: parsedType,
      uri: parsedUri,
      scope: map['scope'] as String,
      priority: map['priority'] as int,
      requiredAtBuild: map['requiredAtBuild'] as bool,
      requiresNetwork: map['requiresNetwork'] as bool,
      schemaVersion: map['schemaVersion'] as int,
      checksum: map['checksum'] as String?,
      checksumAlgo: map['checksumAlgo'] as String?,
      userEditable: map['userEditable'] as bool? ?? true,
    );
  }

  Map<String, dynamic> toMap() {
    return {
      'id': id,
      'dataset': dataset,
      'name': name,
      'license': license,
      'type': type.name,
      'uri': uri.toString(),
      'scope': scope,
      'priority': priority,
      'requiredAtBuild': requiredAtBuild,
      'requiresNetwork': requiresNetwork,
      'schemaVersion': schemaVersion,
      'checksum': checksum,
      'checksumAlgo': checksumAlgo,
      'userEditable': userEditable,
    };
  }

  /// Creates a list of [SourceSpec] from the JSON string bundled in assets.
  static List<SourceSpec> listFromJsonString(String jsonString) {
    final decoded = json.decode(jsonString);
    if (decoded is! Map<String, dynamic>) {
      throw const FormatException('Expected an object at the root of the spec');
    }
    final sources = decoded['sources'];
    if (sources is! List) {
      throw const FormatException('Expected a sources array in the spec');
    }
    return sources
        .map(
          (dynamic entry) => SourceSpec.fromMap(entry as Map<String, dynamic>),
        )
        .toList();
  }
}

/// Supported source access types.
enum SourceType { api, asset, file }

extension SourceTypeExtension on SourceType {
  static SourceType fromString(String value) {
    switch (value) {
      case 'api':
        return SourceType.api;
      case 'asset':
        return SourceType.asset;
      case 'file':
        return SourceType.file;
      default:
        throw FormatException('Unsupported source type: $value');
    }
  }
}
