import 'package:natinfo_flutter/shared/data_sources/source_errors.dart';
import 'package:natinfo_flutter/shared/data_sources/source_registry.dart';
import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';
import 'package:natinfo_flutter/shared/utils/iterable_extensions.dart';

/// Resolves a source to use for a dataset based on priority and policy flags.
class SourceResolver {
  const SourceResolver(this.registry);

  final SourceRegistry registry;

  /// Returns the best matching source for the dataset.
  ///
  /// If [preferredSourceId] is provided and matches an allowed source, it is
  /// selected first. If no allowed source exists, a [StateError] is thrown.
  SourceSpec resolve(
    String dataset, {
    bool allowNetwork = true,
    String? preferredSourceId,
  }) {
    final candidates = resolveCandidates(
      dataset,
      allowNetwork: allowNetwork,
      preferredSourceId: preferredSourceId,
    );
    if (candidates.isEmpty) {
      throw buildUnavailableException(dataset, allowNetwork: allowNetwork);
    }
    return candidates.first;
  }

  /// Returns candidate sources in the order they should be attempted.
  List<SourceSpec> resolveCandidates(
    String dataset, {
    bool allowNetwork = true,
    String? preferredSourceId,
  }) {
    final candidates = registry.sourcesForDataset(
      dataset,
      allowNetwork: allowNetwork,
    );
    if (preferredSourceId == null) {
      return candidates;
    }

    final preferred =
        candidates
            .where((source) => source.id == preferredSourceId)
            .firstOrNull;
    if (preferred == null) {
      return candidates;
    }

    final reordered = <SourceSpec>[preferred, ...candidates];
    // Remove duplicates after reordering to keep the list clean.
    final seen = <String>{};
    return reordered
        .where((source) => seen.add(source.id))
        .toList(growable: false);
  }

  SourceUnavailableException buildUnavailableException(
    String dataset, {
    bool allowNetwork = true,
  }) {
    final allSources = registry.sourcesForDataset(dataset);
    if (allSources.isEmpty) {
      return SourceUnavailableException(
        'No source configured for dataset "$dataset".',
        dataset: dataset,
      );
    }

    final requiredSources =
        allSources.where((source) => source.requiredAtBuild).toList();
    if (requiredSources.isNotEmpty) {
      final ids = requiredSources.map((s) => s.id).join(', ');
      final reason =
          allowNetwork
              ? 'no compatible source available'
              : 'downloads disabled';
      return SourceUnavailableException(
        'Required source(s) $ids for dataset "$dataset" unavailable ($reason).',
        dataset: dataset,
        requiredSourceIds: requiredSources.map((s) => s.id).toList(),
      );
    }

    return SourceUnavailableException(
      'No source available for dataset "$dataset" with current policy.',
      dataset: dataset,
    );
  }
}
