import 'dart:convert';

import 'package:flutter/foundation.dart';
import 'package:flutter/services.dart';
import 'package:natinfo_flutter/shared/data_sources/source_registry.dart';

/// Asset path for the registry spec.
const String kSourceRegistryAsset = 'assets/source_registry.json';

/// Loads the source registry from the bundled asset.
Future<SourceRegistry> loadSourceRegistry({
  AssetBundle? bundle,
  Map<String, Uri>? overrides,
}) async {
  final AssetBundle assetBundle = bundle ?? rootBundle;
  final jsonString = await _loadRegistryJson(assetBundle);
  return SourceRegistry.fromJsonString(jsonString, overrides: overrides);
}

Future<String> _loadRegistryJson(AssetBundle bundle) async {
  const candidates = <String>[
    kSourceRegistryAsset,
    '"$kSourceRegistryAsset"',
    "'$kSourceRegistryAsset'",
  ];

  // Try the common keys first, then fall back to manifest scan.
  for (final candidate in candidates) {
    final sanitized = candidate.replaceAll('"', '').replaceAll("'", '');
    try {
      return await bundle.loadString(sanitized);
    } on FlutterError {
      continue;
    }
  }

  try {
    final manifestRaw = await bundle.loadString('AssetManifest.json');
    final manifest = json.decode(manifestRaw);
    if (manifest is Map<String, dynamic>) {
      final match = manifest.keys.cast<String>().firstWhere(
        (k) => k.endsWith('source_registry.json'),
        orElse: () => '',
      );
      if (match.isNotEmpty) {
        return await bundle.loadString(match);
      }
    }
  } catch (_) {}

  throw FlutterError(
    'Unable to load source registry asset "$kSourceRegistryAsset". '
    'Ensure it is bundled in pubspec.yaml.',
  );
}
