import 'package:flutter/services.dart';
import 'package:natinfo_flutter/shared/data_sources/source_override_store.dart';
import 'package:natinfo_flutter/shared/data_sources/source_preferences.dart';
import 'package:natinfo_flutter/shared/data_sources/source_registry.dart';
import 'package:natinfo_flutter/shared/data_sources/source_registry_loader.dart';
import 'package:natinfo_flutter/shared/data_sources/source_resolver.dart';
import 'package:shared_preferences/shared_preferences.dart';

/// Bundles source-related collaborators resolved at runtime.
class SourceEnvironment {
  SourceEnvironment({
    required this.preferences,
    required this.overrides,
    required this.registry,
    required this.resolver,
    required this.sourcePreferences,
  });

  /// Shared preferences instance used to persist source settings.
  final SharedPreferences preferences;

  /// Overrides loaded from user settings, keyed by dataset.
  final Map<String, Uri> overrides;

  /// Registry of available sources after overrides are applied.
  final SourceRegistry registry;

  /// Resolver built from the registry.
  final SourceResolver resolver;

  /// User-level source preferences (e.g., network allowed).
  final SourcePreferences sourcePreferences;
}

/// Creates [SourceEnvironment] instances so features share consistent setup.
class SourceEnvironmentLoader {
  const SourceEnvironmentLoader._();

  /// Loads the environment using defaults unless collaborators are provided.
  static Future<SourceEnvironment> load({
    SharedPreferences? preferences,
    Map<String, Uri>? overrides,
    SourceRegistry? registry,
    SourceResolver? resolver,
    SourcePreferences? sourcePreferences,
    AssetBundle? bundle,
  }) async {
    final prefs = preferences ?? await SharedPreferences.getInstance();
    final resolvedOverrides =
        overrides ?? SourceOverrideStore(prefs).loadOverrides();
    final resolvedRegistry =
        registry ??
        await loadSourceRegistry(bundle: bundle, overrides: resolvedOverrides);
    final resolvedResolver = resolver ?? SourceResolver(resolvedRegistry);
    final resolvedSourcePrefs = sourcePreferences ?? SourcePreferences(prefs);

    return SourceEnvironment(
      preferences: prefs,
      overrides: resolvedOverrides,
      registry: resolvedRegistry,
      resolver: resolvedResolver,
      sourcePreferences: resolvedSourcePrefs,
    );
  }
}
