import 'dart:convert';
import 'dart:io';
import 'package:geolocator/geolocator.dart';
import 'package:http/http.dart' as http;
import 'package:latlong2/latlong.dart';

/// Service de localisation (GPS & géocodage).
class LocationService {
  /// Vérifie et demande la permission de localisation si nécessaire.
  Future<void> ensurePermission() async {
    var status = await Geolocator.checkPermission();
    if (status == LocationPermission.denied) {
      status = await Geolocator.requestPermission();
    }
    if (status == LocationPermission.denied) {
      throw Exception('Permission de localisation refusée');
    }
    if (status == LocationPermission.deniedForever) {
      throw Exception('Permission de localisation refusée définitivement');
    }
  }

  /// Récupère la position GPS de l'utilisateur.
  Future<LatLng> getCurrentLocation() async {
    // Sur Linux, on lève une exception pour signaler que ce service n'est pas supporté.
    if (Platform.isLinux) {
      throw Exception('La localisation n’est pas prise en charge sur Linux.');
    }
    try {
      if (!await Geolocator.isLocationServiceEnabled()) {
        throw Exception('Service de localisation désactivé');
      }
      final pos = await Geolocator.getCurrentPosition();
      return LatLng(pos.latitude, pos.longitude);
    } catch (e) {
      rethrow;
    }
  }

  /// Géocode une adresse via Nominatim.
  Future<LatLng> geocode(String address) async {
    final uri = Uri.parse(
      'https://nominatim.openstreetmap.org/search'
      '?q=${Uri.encodeComponent(address)}&format=json&limit=1',
    );
    try {
      final res = await http.get(uri, headers: {'User-Agent': 'NATINFo+'});
      final list = jsonDecode(res.body) as List<dynamic>;
      if (list.isEmpty) throw Exception('Adresse introuvable');
      final lat = double.parse(list[0]['lat'].toString());
      final lon = double.parse(list[0]['lon'].toString());
      return LatLng(lat, lon);
    } catch (e) {
      rethrow;
    }
  }
}
