import 'dart:io';

import 'package:flutter/foundation.dart';
import 'package:flutter/services.dart';
import 'package:natinfo_flutter/shared/data_sources/default_source_adapters.dart';
import 'package:natinfo_flutter/shared/data_sources/source_adapter.dart';
import 'package:natinfo_flutter/shared/data_sources/source_environment.dart';
import 'package:natinfo_flutter/shared/data_sources/source_loader.dart';
import 'package:natinfo_flutter/shared/data_sources/source_logger.dart';
import 'package:natinfo_flutter/shared/data_sources/source_preferences.dart';
import 'package:natinfo_flutter/shared/data_sources/source_registry.dart';
import 'package:natinfo_flutter/shared/data_sources/source_resolver.dart';
import 'package:natinfo_flutter/shared/data_sources/update_service.dart';

/// Service de cache pour les CSV : téléchargement via la couche SourceLoader et
/// stockage local versionné avec TTL.
class CsvCacheService {
  CsvCacheService({
    SourceRegistry? sourceRegistry,
    SourceResolver? sourceResolver,
    List<SourceAdapter>? adapters,
    SourceUpdateService? updateService,
    SourcePreferences? sourcePreferences,
    SourceLogger? logger,
  }) : _providedRegistry = sourceRegistry,
       _providedResolver = sourceResolver,
       _providedAdapters = adapters,
       _providedUpdateService = updateService,
       _providedSourcePrefs = sourcePreferences,
       _logger = logger ?? SourceLogger(tag: 'CsvCache');

  static const Duration _staleThreshold = Duration(days: 365);

  final SourceRegistry? _providedRegistry;
  final SourceResolver? _providedResolver;
  final List<SourceAdapter>? _providedAdapters;
  final SourceUpdateService? _providedUpdateService;
  final SourcePreferences? _providedSourcePrefs;
  final SourceLogger _logger;

  SourceUpdateService? _updateService;
  SourcePreferences? _sourcePreferences;
  SourceEnvironment? _environment;
  Map<String, Uri> _activeOverrides = const {};

  Future<void> _ensureInitialized() async {
    final environment = await SourceEnvironmentLoader.load(
      registry: _providedRegistry,
      resolver: _providedResolver,
      sourcePreferences: _providedSourcePrefs,
    );
    _environment = environment;
    _sourcePreferences ??= environment.sourcePreferences;

    final shouldRefresh =
        _updateService == null ||
        (_providedUpdateService == null &&
            !mapEquals(_activeOverrides, environment.overrides));
    if (_updateService == null) {
      _updateService =
          _providedUpdateService ?? await _buildUpdateService(environment);
    } else if (_providedUpdateService == null && shouldRefresh) {
      _updateService = await _buildUpdateService(environment);
    }
    _activeOverrides = environment.overrides;
  }

  Future<SourceUpdateService> _buildUpdateService(
    SourceEnvironment environment,
  ) async {
    final resolver = environment.resolver;
    final adapters = _providedAdapters ?? DefaultSourceAdapters.buildList();
    final loader = SourceLoader(resolver: resolver, adapters: adapters);
    return SourceUpdateService(loader: loader, ttl: _staleThreshold);
  }

  /// Returns whether a dataset already has data cached locally.
  Future<bool> hasDataset(String datasetId) async {
    await _ensureInitialized();
    _logger.info('Checking cached dataset $datasetId');
    final metadata = await _updateService!.getMetadata(datasetId);
    if (metadata != null) {
      final source = metadata.sourceId ?? 'unknown';
      final updated = metadata.lastUpdated?.toIso8601String() ?? 'unknown';
      _logger.info(
        'Dataset $datasetId metadata found (source: $source, updated: $updated)',
      );
      return true;
    }
    final fallback = await _updateService!.getCachedPayload(datasetId);
    if (fallback != null) {
      _logger.info(
        'Dataset $datasetId payload fallback found (${fallback.path})',
      );
      return true;
    }
    _logger.warn('Dataset $datasetId not cached locally');
    return false;
  }

  /// Fetches the dataset, refreshing it if stale beyond TTL.
  Future<File> fetchDataset(String datasetId, {bool force = false}) async {
    await _ensureInitialized();
    final allowNetwork = _sourcePreferences!.isNetworkAllowed(datasetId);
    _logger.info(
      'Fetching dataset $datasetId (force: $force, allowNetwork: $allowNetwork)',
    );
    final result = await _updateService!.refreshDataset(
      datasetId,
      force: force,
      allowNetwork: allowNetwork,
    );
    final payload =
        result.payloadFile ?? await _updateService!.getCachedPayload(datasetId);
    if (payload != null) {
      _logger.info('Dataset $datasetId available at ${payload.path}');
      return payload;
    }
    if (!allowNetwork) {
      _logger.warn(
        'Dataset $datasetId has no local payload and network is disabled',
      );
      throw Exception(
        'Aucune donnée locale pour $datasetId et téléchargements réseau interdits.',
      );
    }
    _logger.warn(
      'Dataset $datasetId had no payload after refresh, forcing download',
    );
    final forced = await _updateService!.refreshDataset(
      datasetId,
      force: true,
      allowNetwork: allowNetwork,
    );
    if (forced.payloadFile != null) {
      _logger.info(
        'Forced refresh returned payload for $datasetId at ${forced.payloadFile!.path}',
      );
      return forced.payloadFile!;
    }
    _logger.warn('Forced refresh failed for $datasetId');
    throw Exception('Impossible de récupérer les données pour $datasetId');
  }

  Future<SourceCacheMetadata?> getDatasetMetadata(String datasetId) async {
    await _ensureInitialized();
    return _updateService!.getMetadata(datasetId);
  }

  Future<void> deleteDataset(String datasetId) async {
    await _ensureInitialized();
    _logger.info('Deleting cached dataset $datasetId');
    await _updateService!.clearDataset(datasetId);
  }
}
