import 'package:flutter/material.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/presentation/widgets/natinf_list.dart';

class NatinfCollectionLayout<T> extends StatelessWidget {
  const NatinfCollectionLayout({
    super.key,
    required this.isLoading,
    required this.sortOption,
    required this.onSortChanged,
    required this.sortItems,
    required this.natinfList,
    this.recordHistory = true,
    this.padding = const EdgeInsets.all(16),
    this.dropdownLabel = 'Trier par',
  });

  final bool isLoading;
  final T sortOption;
  final ValueChanged<T> onSortChanged;
  final List<DropdownMenuItem<T>> sortItems;
  final List<Natinf> natinfList;
  final bool recordHistory;
  final EdgeInsetsGeometry padding;
  final String dropdownLabel;

  @override
  Widget build(BuildContext context) {
    if (isLoading) {
      return const Center(child: CircularProgressIndicator());
    }

    return Padding(
      padding: padding,
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          DropdownButtonFormField<T>(
            value: sortOption,
            decoration: InputDecoration(
              labelText: dropdownLabel,
              border: OutlineInputBorder(
                borderRadius: BorderRadius.circular(8),
              ),
              contentPadding: const EdgeInsets.symmetric(
                horizontal: 12,
                vertical: 4,
              ),
            ),
            onChanged: (value) {
              if (value != null) {
                onSortChanged(value);
              }
            },
            items: sortItems,
          ),
          const SizedBox(height: 8),
          Expanded(
            child: Container(
              margin: const EdgeInsets.only(top: 8.0),
              decoration: BoxDecoration(borderRadius: BorderRadius.circular(8)),
              child: ClipRRect(
                borderRadius: BorderRadius.circular(8),
                child: NatinfList(
                  natinfList: natinfList,
                  recordHistory: recordHistory,
                ),
              ),
            ),
          ),
        ],
      ),
    );
  }
}

Future<bool> showDeletionConfirmationDialog({
  required BuildContext context,
  required String message,
  String title = 'Confirmer la suppression',
  String cancelLabel = 'Annuler',
  String confirmLabel = 'Supprimer',
}) async {
  final shouldDelete = await showDialog<bool>(
    context: context,
    builder:
        (dialogContext) => AlertDialog(
          title: Text(title),
          content: Text(message),
          actions: [
            TextButton(
              onPressed: () => Navigator.pop(dialogContext, false),
              child: Text(cancelLabel),
            ),
            TextButton(
              onPressed: () => Navigator.pop(dialogContext, true),
              child: Text(confirmLabel),
            ),
          ],
        ),
  );

  return shouldDelete ?? false;
}

List<Natinf> sortNatinfsByConsultations({
  required Iterable<Map<String, dynamic>> historyRecords,
  required Iterable<Natinf> natinfList,
  bool descending = true,
  bool includeZeroConsultations = false,
}) {
  final counts = <String, int>{};
  for (final record in historyRecords) {
    final key = record['numero_natinf']?.toString();
    if (key == null) {
      continue;
    }
    counts[key] = (counts[key] ?? 0) + 1;
  }

  final natinfByNumero = <String, Natinf>{};
  for (final natinf in natinfList) {
    final key = natinf.numeroNatinf?.toString();
    if (key == null) {
      continue;
    }
    natinfByNumero.putIfAbsent(key, () => natinf);
  }

  final entries =
      counts.entries
          .where((entry) => natinfByNumero.containsKey(entry.key))
          .map((entry) => MapEntry(entry.key, entry.value))
          .toList();

  if (includeZeroConsultations) {
    final missingKeys = natinfByNumero.keys
        .where((key) => !counts.containsKey(key))
        .map((key) => MapEntry(key, 0));
    entries.addAll(missingKeys);
  }

  entries.sort((a, b) {
    final comparison = a.value.compareTo(b.value);
    if (comparison == 0) {
      final current = natinfByNumero[a.key];
      final other = natinfByNumero[b.key];
      final numeroA = current?.numeroNatinf ?? '';
      final numeroB = other?.numeroNatinf ?? '';
      return numeroA.compareTo(numeroB);
    }
    return descending ? -comparison : comparison;
  });

  return entries
      .map((entry) => natinfByNumero[entry.key])
      .whereType<Natinf>()
      .toList();
}
