import 'package:flutter/material.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/app/config/app_config.dart';
import 'package:natinfo_flutter/app/state/operational_mode_provider.dart';
import 'package:natinfo_flutter/shared/services/matomo_service.dart';
import 'package:natinfo_flutter/app/navigation/app_drawer.dart';
import 'package:natinfo_flutter/features/natinf/data/database_helper.dart';
import 'package:natinfo_flutter/features/natinf/presentation/pages/categories_page.dart';
import 'package:natinfo_flutter/features/natinf/presentation/pages/favourites_page.dart';
import 'package:natinfo_flutter/features/natinf/presentation/pages/natinf_details_page.dart';
import 'package:natinfo_flutter/features/natinf/presentation/pages/natinf_sorter.dart';
import 'package:natinfo_flutter/features/natinf/presentation/widgets/natinf_list.dart';
import 'package:provider/provider.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_repository.dart';

class _SortOption {
  const _SortOption({
    required this.key,
    required this.icon,
    required this.tooltip,
  });

  final NatinfSortKey key;
  final IconData icon;
  final String tooltip;
}

/// Page d'accueil affichant la recherche NATINF, les raccourcis et l'état initial.
class NatinfHomePage extends StatefulWidget {
  /// Crée la page d'accueil NATINF.
  const NatinfHomePage({super.key, this.recentHistoryLoader});

  /// Chargeur personnalisé pour lister les NATINF récemment consultés
  /// (utilisé pour les tests ou des environnements spécifiques).
  final Future<List<Natinf>> Function()? recentHistoryLoader;

  @override
  State<NatinfHomePage> createState() => _NatinfHomePageState();
}

class _NatinfHomePageState extends State<NatinfHomePage> {
  static const _maxRecentHistoryItems = 3;
  static const List<_SortOption> _sortOptions = [
    _SortOption(
      key: NatinfSortKey.apiOrder,
      icon: Icons.replay,
      tooltip: "Par défaut",
    ),
    _SortOption(
      key: NatinfSortKey.natinf,
      icon: Icons.numbers,
      tooltip: 'NATINF',
    ),
    _SortOption(
      key: NatinfSortKey.alphabetical,
      icon: Icons.sort_by_alpha,
      tooltip: 'Qualification',
    ),
    _SortOption(key: NatinfSortKey.nature, icon: Icons.eco, tooltip: "Nature"),
  ];

  final TextEditingController _controller = TextEditingController();
  List<Natinf>? _results;
  List<Natinf> _rawResults = const [];
  bool _isLoading = false;
  String _errorMessage = '';
  String _statusMessage = '';
  bool _isStatusExpanded = false;
  NatinfSortKey _activeSortKey = NatinfSortKey.apiOrder;
  List<Natinf> _recentHistory = const [];
  late final Future<List<Natinf>> Function() _recentHistoryLoader;
  final Map<NatinfSortKey, bool> _ascendingByKey = {
    NatinfSortKey.apiOrder: true,
    NatinfSortKey.natinf: true,
    NatinfSortKey.alphabetical: true,
    NatinfSortKey.nature: true,
  };

  @override
  void initState() {
    super.initState();
    _recentHistoryLoader =
        widget.recentHistoryLoader ?? _loadRecentHistoryFromDb;
    _loadRecentHistory();
    // MatomoService().trackPage(title: 'Accueil', path: '/');
  }

  void _resetSortPreferences() {
    for (final key in NatinfSortKey.values) {
      _ascendingByKey[key] = true;
    }
    _activeSortKey = NatinfSortKey.apiOrder;
  }

  Future<void> _loadRecentHistory() async {
    try {
      final recent = await _recentHistoryLoader();
      if (!mounted) return;
      setState(() {
        _recentHistory = recent;
      });
    } catch (_) {
      if (!mounted) return;
      setState(() {
        _recentHistory = const [];
      });
    }
  }

  Future<List<Natinf>> _loadRecentHistoryFromDb() async {
    final helper = DatabaseHelper();
    final history = await helper.getHistory();
    final seen = <String>{};
    final numeros = <String>[];
    for (final entry in history) {
      final numero = entry['numero_natinf'] as String?;
      if (numero == null) continue;
      if (seen.add(numero)) {
        numeros.add(numero);
      }
      if (numeros.length >= _maxRecentHistoryItems) {
        break;
      }
    }
    if (numeros.isEmpty) {
      return const [];
    }
    final futures = numeros.map(helper.getNatinfByNumero).toList();
    final fetched = await Future.wait(futures);
    return fetched.whereType<Natinf>().toList();
  }

  Future<void> _search() async {
    FocusScope.of(context).unfocus();
    setState(() {
      _isLoading = true;
      _errorMessage = '';
      _results = null;
      _rawResults = [];
      _isStatusExpanded = false;
      _resetSortPreferences();
      _statusMessage = "Démarrage de la recherche...";
    });
    final repository = Provider.of<NatinfRepository>(context, listen: false);
    try {
      final results = await repository.searchNatinf(
        _controller.text,
        onStatus: (msg) {
          setState(() {
            _isStatusExpanded = false;
            _statusMessage = msg;
          });
        },
        forceOffline: AppConfig.forceOffline,
      );
      setState(() {
        _rawResults = List<Natinf>.from(results);
        _resetSortPreferences();
        _applySort();
      });
      await MatomoService().trackSearch(
        keyword: _controller.text,
        count: results.length,
      );
    } catch (e) {
      setState(() {
        _errorMessage = e.toString();
      });
    } finally {
      setState(() {
        _isLoading = false;
      });
    }
  }

  Future<void> _openCategories() async {
    await Navigator.push(
      context,
      MaterialPageRoute<void>(builder: (context) => const CategoriesPage()),
    );
  }

  Future<void> _navigateToFavourites() async {
    await Navigator.push(
      context,
      MaterialPageRoute<void>(builder: (context) => const FavouritesPage()),
    );
  }

  Future<void> _openRecentNatinf(Natinf natinf) async {
    final numero = natinf.numeroNatinf;
    if (numero != null) {
      await context.read<NatinfRepository>().recordHistoryEntry(numero);
    }
    if (!mounted) return;
    await Navigator.push(
      context,
      MaterialPageRoute<void>(builder: (_) => NatinfDetail(natinf: natinf)),
    );
    if (!mounted) return;
    await _loadRecentHistory();
  }

  Widget _buildSearchBar() {
    return Row(
      children: [
        Expanded(
          child: ValueListenableBuilder<TextEditingValue>(
            valueListenable: _controller,
            builder: (context, value, _) {
              return TextField(
                controller: _controller,
                textInputAction: TextInputAction.search,
                onSubmitted: (_) => _isLoading ? null : _search(),
                decoration: InputDecoration(
                  labelText: "Recherche d'infractions",
                  hintText: 'ex: numéro, nature, qualification, articles...',
                  border: const OutlineInputBorder(),
                  suffixIcon:
                      value.text.isEmpty
                          ? null
                          : IconButton(
                            icon: const Icon(Icons.clear),
                            onPressed: () {
                              _controller.clear();
                              setState(() {
                                _isStatusExpanded = false;
                                _statusMessage = '';
                                _errorMessage = '';
                              });
                            },
                          ),
                ),
              );
            },
          ),
        ),
        const SizedBox(width: 8),
        ElevatedButton(
          onPressed: _isLoading ? null : _search,
          style: ElevatedButton.styleFrom(
            padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 20),
          ),
          child:
              _isLoading
                  ? const SizedBox(
                    width: 24,
                    height: 24,
                    child: CircularProgressIndicator(
                      strokeWidth: 2.5,
                      backgroundColor: Colors.white,
                    ),
                  )
                  : const Icon(Icons.search),
        ),
      ],
    );
  }

  Widget _buildWelcomePanel() {
    final theme = Theme.of(context);
    final hasRecentHistory = _recentHistory.isNotEmpty;
    final quickActionStyle = FilledButton.styleFrom(
      minimumSize: const Size(118, 44),
      padding: const EdgeInsets.symmetric(horizontal: 12),
    );

    return KeyedSubtree(
      key: const Key('natinf_home_welcome'),
      child: Center(
        child: SingleChildScrollView(
          child: ConstrainedBox(
            constraints: const BoxConstraints(maxWidth: 520),
            child: Card(
              shape: RoundedRectangleBorder(
                borderRadius: BorderRadius.circular(16),
              ),
              elevation: 2,
              child: Padding(
                padding: const EdgeInsets.all(20),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Text(
                      'Recherchez une NATINF par numéro ou mots-clés, ou accédez directement aux sections clés.',
                      style: theme.textTheme.bodyLarge?.copyWith(
                        fontWeight: FontWeight.w600,
                      ),
                    ),
                    const SizedBox(height: 16),
                    Center(
                      child: Wrap(
                        alignment: WrapAlignment.center,
                        spacing: 12,
                        runSpacing: 12,
                        children: [
                          FilledButton.icon(
                            style: quickActionStyle,
                            key: const Key('natinf_home_open_categories'),
                            onPressed: _openCategories,
                            icon: const Icon(Icons.category_outlined),
                            label: const Text('Catégories'),
                          ),
                          FilledButton.icon(
                            style: quickActionStyle,
                            key: const Key('natinf_home_open_favourites'),
                            onPressed: _navigateToFavourites,
                            icon: const Icon(Icons.star_border),
                            label: const Text('Favoris'),
                          ),
                        ],
                      ),
                    ),
                    if (hasRecentHistory) ...[
                      const SizedBox(height: 20),
                      Divider(color: theme.dividerColor),
                      const SizedBox(height: 12),
                      Text(
                        'Dernières NATINF consultées',
                        style: theme.textTheme.titleMedium,
                      ),
                      const SizedBox(height: 8),
                      ..._recentHistory.map(
                        (natinf) => ListTile(
                          key: ValueKey(
                            'recent_${natinf.numeroNatinf ?? natinf.id}',
                          ),
                          contentPadding: EdgeInsets.zero,
                          leading: const Icon(Icons.history),
                          title: Text(
                            natinf.qualificationInfraction ??
                                (natinf.numeroNatinf != null
                                    ? 'NATINF ${natinf.numeroNatinf}'
                                    : 'NATINF'),
                            maxLines: 2,
                            overflow: TextOverflow.ellipsis,
                          ),
                          subtitle:
                              natinf.numeroNatinf != null
                                  ? Text('NATINF ${natinf.numeroNatinf}')
                                  : null,
                          trailing: const Icon(Icons.chevron_right),
                          onTap: () => _openRecentNatinf(natinf),
                        ),
                      ),
                    ],
                  ],
                ),
              ),
            ),
          ),
        ),
      ),
    );
  }

  void _toggleStatus() {
    setState(() {
      _isStatusExpanded = !_isStatusExpanded;
    });
  }

  void _handleSortSelection(NatinfSortKey key) {
    final hasResults = _rawResults.isNotEmpty;
    if (!hasResults) {
      return;
    }

    setState(() {
      if (_activeSortKey == key) {
        final current = _ascendingByKey[key] ?? true;
        _ascendingByKey[key] = !current;
      } else {
        _activeSortKey = key;
        _ascendingByKey[key] = _ascendingByKey[key] ?? true;
      }
      _applySort();
    });
  }

  void _applySort() {
    final ascending = _ascendingByKey[_activeSortKey] ?? true;
    final sorter = NatinfSorter(_rawResults);
    _results = sorter.sort(_activeSortKey, ascending: ascending);
  }

  int _alphaWithOpacity(Color color, double opacity) {
    final value = (color.a * 255.0 * opacity).round();
    return value.clamp(0, 255).toInt();
  }

  Widget _buildStatusMessage({String? resultsLabel}) {
    final hasStatus = _statusMessage.isNotEmpty;
    final hasResultsLabel = resultsLabel != null && resultsLabel.isNotEmpty;
    final hasAnyResults = _results != null && _results!.isNotEmpty;
    final shouldRender = hasStatus || hasResultsLabel || hasAnyResults;
    final String? resolvedResultsLabel = hasResultsLabel ? resultsLabel : null;

    if (!shouldRender) {
      return const SizedBox.shrink();
    }

    final theme = Theme.of(context);
    final onSurface = theme.colorScheme.onSurface;
    final subtleColor = onSurface.withAlpha(_alphaWithOpacity(onSurface, 0.6));
    final primary = theme.colorScheme.primary;
    final selections = _sortOptions
        .map((option) => option.key == _activeSortKey)
        .toList(growable: false);

    final toggleButtons = ToggleButtons(
      isSelected: selections,
      constraints: const BoxConstraints(minHeight: 32, minWidth: 36),
      borderRadius: BorderRadius.circular(8),
      color: subtleColor,
      selectedColor: primary,
      fillColor: primary.withAlpha(_alphaWithOpacity(primary, 0.12)),
      disabledColor: subtleColor.withAlpha(_alphaWithOpacity(subtleColor, 0.4)),
      borderColor: Colors.transparent,
      selectedBorderColor: Colors.transparent,
      onPressed:
          hasAnyResults
              ? (index) => _handleSortSelection(_sortOptions[index].key)
              : null,
      children: _sortOptions
          .map(
            (option) => Tooltip(
              message: option.tooltip,
              waitDuration: const Duration(milliseconds: 400),
              child: Icon(option.icon, size: 18),
            ),
          )
          .toList(growable: false),
    );

    final resultsStyle = theme.textTheme.bodyMedium?.copyWith(
      fontWeight: FontWeight.w600,
    );

    final Widget? detailsButton =
        hasStatus
            ? Tooltip(
              message: _isStatusExpanded ? 'Masquer' : 'Voir les détails',
              waitDuration: const Duration(milliseconds: 400),
              child: TextButton(
                onPressed: _toggleStatus,
                style: TextButton.styleFrom(
                  padding: const EdgeInsets.symmetric(
                    horizontal: 8,
                    vertical: 4,
                  ),
                  minimumSize: const Size(0, 32),
                  tapTargetSize: MaterialTapTargetSize.shrinkWrap,
                  foregroundColor: subtleColor,
                  textStyle: const TextStyle(fontSize: 13),
                ),
                child: Row(
                  mainAxisSize: MainAxisSize.min,
                  children: [
                    const Icon(Icons.info_outline, size: 18),
                    if (_isStatusExpanded) ...[
                      const SizedBox(width: 4),
                      const Text('Masquer'),
                    ],
                  ],
                ),
              ),
            )
            : null;

    return Padding(
      padding: const EdgeInsets.only(bottom: 8.0),
      child: Column(
        mainAxisSize: MainAxisSize.min,
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Row(
            crossAxisAlignment: CrossAxisAlignment.center,
            children: [
              toggleButtons,
              const SizedBox(width: 8),
              if (resolvedResultsLabel != null)
                Expanded(
                  child: Center(
                    child: Text(
                      resolvedResultsLabel,
                      style: resultsStyle,
                      textAlign: TextAlign.center,
                      overflow: TextOverflow.ellipsis,
                      softWrap: false,
                    ),
                  ),
                )
              else
                const Spacer(),
              if (detailsButton != null) ...[
                const SizedBox(width: 8),
                detailsButton,
              ],
            ],
          ),
          if (hasStatus)
            AnimatedCrossFade(
              firstChild: const SizedBox.shrink(),
              secondChild: Padding(
                padding: const EdgeInsets.only(top: 4.0),
                child: Text(
                  _statusMessage,
                  style: const TextStyle(
                    fontStyle: FontStyle.italic,
                    color: Colors.blueGrey,
                  ),
                ),
              ),
              crossFadeState:
                  _isStatusExpanded
                      ? CrossFadeState.showSecond
                      : CrossFadeState.showFirst,
              duration: const Duration(milliseconds: 200),
              sizeCurve: Curves.easeInOut,
              alignment: Alignment.topLeft,
            ),
        ],
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    final opMode = context.watch<OperationalModeProvider>().opMode;
    final resultsLabel =
        !_isLoading && _results != null
            ? '${_results!.length} ${_results!.length > 1 ? 'résultats' : 'résultat'}'
            : null;

    final disclaimer = Padding(
      padding: const EdgeInsets.symmetric(vertical: 8.0, horizontal: 16.0),
      child: Text(
        "Cette application n'est pas un service officiel de l'État français et n'est ni affiliée, ni soutenue par l'administration publique.",
        textAlign: TextAlign.center,
        style: TextStyle(
          fontSize: 12,
          color: Colors.grey.shade600,
          fontStyle: FontStyle.italic,
        ),
      ),
    );

    return Scaffold(
      appBar: AppBar(title: const Text('NATINFo+')),
      drawer: const AppDrawer(),
      body: Padding(
        padding: const EdgeInsets.all(16),
        child: Column(
          children: [
            if (!opMode) _buildSearchBar(),
            if (!opMode) const SizedBox(height: 16),
            if (!opMode) _buildStatusMessage(resultsLabel: resultsLabel),
            if (_errorMessage.isNotEmpty)
              Text(_errorMessage, style: const TextStyle(color: Colors.red)),
            Expanded(
              child:
                  _results != null
                      ? NatinfList(natinfList: _results!)
                      : _isLoading
                      ? const SizedBox.shrink()
                      : Align(
                        alignment: const Alignment(0, -0.1),
                        child: _buildWelcomePanel(),
                      ),
            ),
          ],
        ),
      ),
      bottomNavigationBar:
          opMode
              ? Padding(
                padding: EdgeInsets.only(
                  bottom: MediaQuery.of(context).viewInsets.bottom,
                ),
                child: SafeArea(
                  child: Column(
                    mainAxisSize: MainAxisSize.min,
                    children: [
                      Padding(
                        padding: const EdgeInsets.symmetric(horizontal: 16.0),
                        child: _buildSearchBar(),
                      ),
                      Padding(
                        padding: const EdgeInsets.symmetric(horizontal: 16.0),
                        child: _buildStatusMessage(resultsLabel: resultsLabel),
                      ),
                      disclaimer,
                    ],
                  ),
                ),
              )
              : disclaimer,
    );
  }
}
