import 'package:flutter/gestures.dart';
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:intl/intl.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.enums.swagger.dart'
    as enums;
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/database_helper.dart';
import 'package:natinfo_flutter/features/natinf/presentation/pages/natinf_legal_texts_page.dart';
import 'package:natinfo_flutter/app/theme/theme_provider.dart';
import 'package:natinfo_flutter/features/natinf/data/favourite_folder_service.dart';
import 'package:natinfo_flutter/shared/services/matomo_service.dart';
import 'package:natinfo_flutter/features/auth/presentation/auth_provider.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_repository.dart';
import 'package:natinfo_flutter/features/natinf/presentation/widgets/amende_popup.dart';
import 'package:natinfo_flutter/features/natinf/presentation/widgets/natinf_share_dialog.dart';
import 'package:natinfo_flutter/features/natinf/presentation/widgets/favourite_folder_dialogs.dart';
import 'package:natinfo_flutter/features/natinf/presentation/widgets/docpro_login_prompt.dart';
import 'package:provider/provider.dart';
import 'package:url_launcher/url_launcher.dart';
import 'package:flutter_linkify/flutter_linkify.dart';

class NatinfDetail extends StatefulWidget {
  final Natinf natinf;
  const NatinfDetail({super.key, required this.natinf});

  @override
  State<NatinfDetail> createState() => _NatinfDetailState();
}

class _CollapsiblePenaltiesTile extends StatefulWidget {
  final String title;
  final Widget child;

  const _CollapsiblePenaltiesTile({required this.title, required this.child});

  @override
  State<_CollapsiblePenaltiesTile> createState() =>
      _CollapsiblePenaltiesTileState();
}

class _CollapsiblePenaltiesTileState extends State<_CollapsiblePenaltiesTile> {
  bool _isExpanded = false;

  @override
  Widget build(BuildContext context) {
    final displayTitle = _isExpanded ? widget.title : '${widget.title}...';
    return ExpansionTile(
      title: Text(
        displayTitle,
        style: const TextStyle(fontWeight: FontWeight.bold),
      ),
      tilePadding: const EdgeInsets.symmetric(horizontal: 16),
      childrenPadding: const EdgeInsets.only(left: 16, right: 16, bottom: 12),
      onExpansionChanged: (expanded) => setState(() => _isExpanded = expanded),
      children: [widget.child],
    );
  }
}

class _NatinfDetailState extends State<NatinfDetail> {
  static final NumberFormat _currencyFormatter = NumberFormat.currency(
    symbol: '€',
    decimalDigits: 0,
  );

  bool _isFavourite = false;
  bool _expandedMore = false;
  bool _loadingDocPro = false;
  String? _docProError;
  List<NatinfDocPro>? _docProDocs;
  AuthProvider? _authProvider;
  bool _docProLoadScheduled = false;

  @override
  void initState() {
    super.initState();
    _checkFavouriteStatus();
    MatomoService().trackPage(
      title: 'NATINF ${widget.natinf.numeroNatinf}',
      path: '/natinf/${widget.natinf.numeroNatinf}/',
    );
    WidgetsBinding.instance.addPostFrameCallback((_) => _maybeLoadDocPro());
  }

  @override
  void didChangeDependencies() {
    super.didChangeDependencies();
    final auth = context.read<AuthProvider>();
    if (_authProvider != auth) {
      _authProvider?.removeListener(_onAuthChanged);
      _authProvider = auth;
      _authProvider?.addListener(_onAuthChanged);
    }
  }

  @override
  void dispose() {
    _authProvider?.removeListener(_onAuthChanged);
    super.dispose();
  }

  Future<void> _checkFavouriteStatus() async {
    if (widget.natinf.numeroNatinf != null) {
      final isFav = await DatabaseHelper().isFavourite(
        widget.natinf.numeroNatinf!,
      );
      setState(() {
        _isFavourite = isFav;
      });
    }
  }

  Future<void> _toggleFavourite() async {
    if (widget.natinf.numeroNatinf == null) return;
    if (_isFavourite) {
      await DatabaseHelper().removeFavourite(widget.natinf.numeroNatinf!);
      ScaffoldMessenger.of(
        context,
      ).showSnackBar(const SnackBar(content: Text("Supprimé des favoris")));
    } else {
      await DatabaseHelper().addFavourites(widget.natinf.numeroNatinf!);
      ScaffoldMessenger.of(
        context,
      ).showSnackBar(const SnackBar(content: Text("Ajouté aux favoris")));
    }
    setState(() {
      _isFavourite = !_isFavourite;
    });
  }

  Future<void> _ensureFavourite() async {
    final numero = widget.natinf.numeroNatinf;
    if (numero == null || _isFavourite) return;
    await DatabaseHelper().addFavourites(numero);
    if (!mounted) return;
    setState(() {
      _isFavourite = true;
    });
    ScaffoldMessenger.of(
      context,
    ).showSnackBar(const SnackBar(content: Text("Ajouté aux favoris")));
  }

  Future<void> _handleFavouriteLongPress() async {
    final numero = widget.natinf.numeroNatinf;
    if (numero == null) return;
    await _ensureFavourite();
    await _showFolderSelectionSheet(numero);
  }

  void _onAuthChanged() {
    if (!mounted) return;
    final hasDocPro = _authProvider?.session?.hasDocPro ?? false;
    if (hasDocPro && !_loadingDocPro && _docProDocs == null) {
      _scheduleDocProLoad();
    }
  }

  void _scheduleDocProLoad() {
    if (_docProLoadScheduled) return;
    _docProLoadScheduled = true;
    WidgetsBinding.instance.addPostFrameCallback((_) {
      _docProLoadScheduled = false;
      if (mounted) {
        _maybeLoadDocPro();
      }
    });
  }

  Future<void> _maybeLoadDocPro() async {
    if (_loadingDocPro || _docProDocs != null) {
      return;
    }
    final numero = widget.natinf.numeroNatinf;
    if (numero == null) return;
    final repository = context.read<NatinfRepository>();
    final auth = context.read<AuthProvider>();
    final hasRights = auth.session?.hasDocPro ?? false;
    final available = repository.isDocsProAvailable(numero);
    final unauthorized = repository.isDocsProUnauthorized(numero);
    if (unauthorized || (!available && !hasRights)) {
      return;
    }
    setState(() {
      _loadingDocPro = true;
      _docProError = null;
    });
    try {
      final docs = await repository.fetchDocPro(numero);
      if (!mounted) return;
      setState(() {
        _docProDocs = docs;
      });
    } catch (error) {
      if (!mounted) return;
      setState(() {
        _docProError = 'Impossible de charger DocPro (${error.runtimeType})';
      });
    } finally {
      if (mounted) {
        setState(() {
          _loadingDocPro = false;
        });
      }
    }
  }

  Future<void> _showFolderSelectionSheet(String numero) async {
    final rootContext = context;
    final helper = DatabaseHelper();
    var folders = await helper.getAllFavouriteFolders();
    if (folders.isEmpty) {
      final name = await showFolderNameInputDialog(
        context: rootContext,
        title: 'Nouveau dossier',
        confirmLabel: 'Créer',
      );
      if (name == null) {
        if (!mounted) return;
        ScaffoldMessenger.of(rootContext).showSnackBar(
          const SnackBar(
            content: Text('Créez un dossier pour organiser vos favoris.'),
          ),
        );
        return;
      }
      await helper.createFavouriteFolder(name: name);
      folders = await helper.getAllFavouriteFolders();
      if (!mounted) return;
      ScaffoldMessenger.of(
        rootContext,
      ).showSnackBar(SnackBar(content: Text('Dossier "$name" créé.')));
    }

    var nodes = FavouriteFolderService.buildTree(
      folders: folders,
      folderContents: const <int, List<Natinf>>{},
    );
    var selectedFolders = await helper.getFolderIdsForNatinf(numero);

    await showModalBottomSheet<void>(
      context: rootContext,
      isScrollControlled: true,
      builder: (sheetContext) {
        return StatefulBuilder(
          builder: (context, setModalState) {
            List<Widget> buildOptions(
              List<FavouriteFolderNode> folderNodes,
              int depth,
            ) {
              final widgets = <Widget>[];
              for (final node in folderNodes) {
                final isSelected = selectedFolders.contains(node.folder.id);
                widgets.add(
                  ListTile(
                    key: ValueKey('picker-${node.folder.id}'),
                    leading: Icon(
                      isSelected
                          ? Icons.check_box
                          : Icons.check_box_outline_blank,
                    ),
                    title: Text(node.folder.name),
                    contentPadding: EdgeInsets.only(left: depth * 16.0 + 16),
                    onTap: () async {
                      final alreadySelected = selectedFolders.contains(
                        node.folder.id,
                      );
                      if (alreadySelected) {
                        await helper.removeNatinfFromFolder(
                          node.folder.id,
                          numero,
                        );
                      } else {
                        await helper.addNatinfToFolder(node.folder.id, numero);
                      }
                      setModalState(() {
                        if (alreadySelected) {
                          selectedFolders.remove(node.folder.id);
                        } else {
                          selectedFolders.add(node.folder.id);
                        }
                      });
                      if (!mounted) return;
                      final message =
                          alreadySelected
                              ? 'NATINF $numero retiré de "${node.folder.name}".'
                              : 'NATINF $numero ajouté à "${node.folder.name}".';
                      ScaffoldMessenger.of(
                        rootContext,
                      ).showSnackBar(SnackBar(content: Text(message)));
                    },
                  ),
                );
                widgets.addAll(buildOptions(node.children, depth + 1));
              }
              return widgets;
            }

            final listChildren =
                nodes.isEmpty
                    ? const [
                      ListTile(
                        leading: Icon(Icons.info_outline),
                        title: Text('Aucun dossier disponible.'),
                      ),
                    ]
                    : buildOptions(nodes, 0);

            return SafeArea(
              child: Padding(
                padding: EdgeInsets.only(
                  bottom: MediaQuery.of(sheetContext).viewInsets.bottom,
                ),
                child: ConstrainedBox(
                  constraints: BoxConstraints(
                    maxHeight: MediaQuery.of(rootContext).size.height * 0.8,
                  ),
                  child: Column(
                    mainAxisSize: MainAxisSize.min,
                    children: [
                      ListTile(
                        title: const Text('Ajouter aux dossiers'),
                        trailing: IconButton(
                          icon: const Icon(Icons.close),
                          onPressed: () => Navigator.of(sheetContext).pop(),
                        ),
                      ),
                      const Divider(height: 1),
                      Expanded(child: ListView(children: listChildren)),
                      Padding(
                        padding: const EdgeInsets.all(16),
                        child: ElevatedButton.icon(
                          onPressed: () async {
                            final name = await showFolderNameInputDialog(
                              context: rootContext,
                              title: 'Nouveau dossier',
                              confirmLabel: 'Créer',
                            );
                            if (name == null) return;
                            await helper.createFavouriteFolder(name: name);
                            folders = await helper.getAllFavouriteFolders();
                            nodes = FavouriteFolderService.buildTree(
                              folders: folders,
                              folderContents: const <int, List<Natinf>>{},
                            );
                            setModalState(() {});
                            if (!mounted) return;
                            ScaffoldMessenger.of(rootContext).showSnackBar(
                              SnackBar(content: Text('Dossier "$name" créé.')),
                            );
                          },
                          icon: const Icon(Icons.create_new_folder),
                          label: const Text('Nouveau dossier'),
                        ),
                      ),
                    ],
                  ),
                ),
              ),
            );
          },
        );
      },
    );
  }

  Future<void> _copyNATINFoUrlToClipboard() async {
    if (widget.natinf.numeroNatinf == null) return;
    await Clipboard.setData(
      ClipboardData(
        text: "https://natinfo.app/plus/${widget.natinf.numeroNatinf}",
      ),
    );
    ScaffoldMessenger.of(context).showSnackBar(
      const SnackBar(content: Text("Lien copié dans le presse-papiers")),
    );
  }

  String? get _shareUrl =>
      widget.natinf.numeroNatinf == null
          ? null
          : 'https://natinfo.app/plus/${widget.natinf.numeroNatinf}';

  Future<void> _showShareDialog() async {
    final numero = widget.natinf.numeroNatinf;
    final url = _shareUrl;
    if (numero == null || url == null) return;

    await showDialog<void>(
      context: context,
      builder: (dialogContext) {
        return NatinfShareDialog(
          numero: numero,
          url: url,
          onCopyLink: _copyNATINFoUrlToClipboard,
        );
      },
    );
  }

  Widget buildBadge(String text) {
    final themeProvider = Provider.of<ThemeProvider>(context, listen: false);
    if (!themeProvider.hasInfractionBadge(text)) return Text(text);
    return IntrinsicWidth(
      child: Container(
        padding: const EdgeInsets.symmetric(horizontal: 8, vertical: 4),
        decoration: BoxDecoration(
          color: themeProvider.colorForInfractionLabel(text),
          borderRadius: BorderRadius.circular(12),
        ),
        child: Text(text),
      ),
    );
  }

  Future<void> _launchURL(String url) async {
    final uri = Uri.parse(url);
    if (await canLaunchUrl(uri)) {
      await launchUrl(uri, mode: LaunchMode.externalApplication);
    } else {
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text("Impossible d'ouvrir le lien")),
      );
    }
  }

  Widget _buildPenaltiesTile(
    BuildContext context,
    String title,
    List<NatinfPenalty> penalties, {
    required bool hideOperator,
    bool collapsed = false,
  }) {
    final theme = Theme.of(context);
    final textStyle = theme.textTheme.bodyMedium!;
    final sorted = List<NatinfPenalty>.from(penalties)..sort((a, b) {
      final orderA = a.ordering ?? 0;
      final orderB = b.ordering ?? 0;
      if (orderA != orderB) return orderA.compareTo(orderB);
      return _penaltyTitle(a).compareTo(_penaltyTitle(b));
    });

    final content = Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      mainAxisSize: MainAxisSize.min,
      children: [
        for (final entry in sorted)
          _buildPenaltyRow(entry, textStyle, hideOperator: hideOperator),
      ],
    );

    if (!collapsed) {
      return ListTile(
        title: Text(title, style: const TextStyle(fontWeight: FontWeight.bold)),
        subtitle: content,
      );
    }

    return _CollapsiblePenaltiesTile(title: title, child: content);
  }

  Widget _buildPenaltyRow(
    NatinfPenalty penalty,
    TextStyle baseStyle, {
    required bool hideOperator,
  }) {
    final detail = _penaltyDetail(penalty, hideOperator: hideOperator);
    return Padding(
      padding: const EdgeInsets.only(bottom: 6),
      child: Row(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text('\u2022 ', style: baseStyle),
          Expanded(
            child: RichText(
              text: TextSpan(
                style: baseStyle,
                children: [
                  TextSpan(text: _penaltyTitle(penalty)),
                  if (detail != null && detail.isNotEmpty)
                    TextSpan(text: ' – $detail'),
                ],
              ),
            ),
          ),
        ],
      ),
    );
  }

  String _penaltyTitle(NatinfPenalty penalty) {
    final candidates = [
      penalty.templateLabel,
      penalty.template,
      penalty.category,
    ];
    for (final candidate in candidates) {
      if (candidate != null && candidate.trim().isNotEmpty) {
        return candidate.trim();
      }
    }
    return 'Peine';
  }

  String? _penaltyDetail(NatinfPenalty penalty, {required bool hideOperator}) {
    final details = <String>[];

    final durationDetail = _penaltyDurationDetail(
      penalty,
      hideOperator: hideOperator,
    );
    if (durationDetail != null && durationDetail.isNotEmpty) {
      details.add(durationDetail);
    }

    final amountDetail = _penaltyAmountDetail(penalty);
    if (amountDetail != null && amountDetail.isNotEmpty) {
      details.add(amountDetail);
    }

    final notes = penalty.notes?.trim();
    if (notes != null && notes.isNotEmpty) {
      details.add(notes);
    }

    if (details.isEmpty) return null;
    return details.join(' – ');
  }

  String? _penaltyDurationDetail(
    NatinfPenalty penalty, {
    required bool hideOperator,
  }) {
    final parts = <String>[];
    final operatorDisplay = penalty.durationOperatorDisplay?.trim();
    final unitDisplay = penalty.durationUnitDisplay?.trim();
    final durationValue = penalty.durationValue;

    if (durationValue != null) {
      final buffer = StringBuffer();
      if (!hideOperator &&
          operatorDisplay != null &&
          operatorDisplay.isNotEmpty) {
        buffer.write(operatorDisplay);
        buffer.write(' ');
      }
      final durationLabel = _formatDurationLabel(
        value: durationValue,
        unit: penalty.durationUnit,
        unitDisplay: unitDisplay,
      );
      buffer.write(durationLabel);
      parts.add(buffer.toString().trim());
    } else if (!hideOperator &&
        operatorDisplay != null &&
        operatorDisplay.isNotEmpty) {
      parts.add(operatorDisplay);
    }

    if (penalty.canDurationDefinitive == true) {
      parts.add('Durée définitive possible');
    }

    if (parts.isEmpty) return null;
    return parts.join(' – ');
  }

  String _formatDurationLabel({
    required int value,
    enums.NatinfPenaltyDurationUnit? unit,
    String? unitDisplay,
  }) {
    final unitText = _resolveDurationUnitText(
      unit: unit,
      unitDisplay: unitDisplay,
      value: value,
    );
    return '$value $unitText'.trim();
  }

  String _resolveDurationUnitText({
    required enums.NatinfPenaltyDurationUnit? unit,
    required String? unitDisplay,
    required int value,
  }) {
    switch (unit) {
      case enums.NatinfPenaltyDurationUnit.h:
        return value > 1 ? 'heures' : 'heure';
      case enums.NatinfPenaltyDurationUnit.d:
        return value > 1 ? 'jours' : 'jour';
      case enums.NatinfPenaltyDurationUnit.m:
        return 'mois';
      case enums.NatinfPenaltyDurationUnit.y:
        return value > 1 ? 'ans' : 'an';
      case enums.NatinfPenaltyDurationUnit.swaggerGeneratedUnknown:
      case null:
        break;
    }

    final fallback = unitDisplay?.trim();
    if (fallback != null && fallback.isNotEmpty) {
      if (fallback.contains('(s)')) {
        return value > 1
            ? fallback.replaceAll('(s)', 's')
            : fallback.replaceAll('(s)', '');
      }
      if (value > 1 && fallback != 'mois' && !fallback.endsWith('s')) {
        return '${fallback}s';
      }
      return fallback;
    }

    return value > 1 ? 'unités' : 'unité';
  }

  String? _penaltyAmountDetail(NatinfPenalty penalty) {
    final raw = penalty.amountValue?.trim();
    if (raw == null || raw.isEmpty) return null;

    final normalized = raw.replaceAll(',', '.');
    final numericValue = double.tryParse(normalized);
    if (numericValue == null) {
      return raw;
    }

    final rounded = numericValue.round();
    return _currencyFormatter.format(rounded);
  }

  @override
  Widget build(BuildContext context) {
    final List<Widget> normalTiles = [];
    final List<Widget> bottomTiles = [];
    final repository = Provider.of<NatinfRepository>(context, listen: false);
    final auth = context.watch<AuthProvider>();
    final bool isAuthenticated = auth.session != null;
    final bool unauthorizedDocsPro =
        widget.natinf.numeroNatinf != null &&
        repository.isDocsProUnauthorized(widget.natinf.numeroNatinf!);
    final bool docsProAvailable =
        widget.natinf.numeroNatinf != null &&
        repository.isDocsProAvailable(widget.natinf.numeroNatinf!);
    final String webUrl =
        "https://natinfo.app/natinf/${widget.natinf.numeroNatinf}";

    void addTile(
      String title,
      String? value, {
      bool useBadge = true,
      bool makeLinks = false,
      bool showInfo = false,
      VoidCallback? onTap,
      bool moveToMoreInfo = false,
      bool inlineValue = false,
    }) {
      if (value == null || value.isEmpty || value == "N/A") return;
      Widget subtitleWidget;
      if (useBadge) {
        subtitleWidget = buildBadge(value);
      } else if (makeLinks) {
        subtitleWidget = Linkify(
          text: value,
          onOpen: (link) => _launchURL(link.url),
          options: const LinkifyOptions(humanize: false),
        );
      } else {
        subtitleWidget = Text(value);
      }
      final titleRow = Row(
        mainAxisSize: MainAxisSize.min,
        children: [
          Text(title, style: const TextStyle(fontWeight: FontWeight.bold)),
          if (showInfo) ...[
            const SizedBox(width: 6),
            Tooltip(
              message: 'Plus d\'infos',
              child: Icon(
                Icons.info_outline,
                size: 18,
                color: Theme.of(context).colorScheme.primary,
              ),
            ),
          ],
          if (inlineValue) ...[
            const SizedBox(width: 8),
            Flexible(child: Text(value, overflow: TextOverflow.visible)),
          ],
        ],
      );

      final tile = ListTile(
        title: titleRow,
        subtitle: inlineValue ? null : subtitleWidget,
        onTap: onTap,
      );
      if (moveToMoreInfo) {
        bottomTiles.add(tile);
      } else if (value.trim().toLowerCase() == 'non') {
        bottomTiles.add(tile);
      } else {
        normalTiles.add(tile);
      }
    }

    void addCustomTile(String title, Widget subtitleWidget) {
      final tile = ListTile(
        title: Text(title, style: const TextStyle(fontWeight: FontWeight.bold)),
        subtitle: subtitleWidget,
      );
      normalTiles.add(tile);
    }

    void addPenaltiesTile(
      String title,
      List<NatinfPenalty>? penalties, {
      required bool hideOperator,
      bool collapsed = false,
    }) {
      if (penalties == null || penalties.isEmpty) return;
      normalTiles.add(
        _buildPenaltiesTile(
          context,
          title,
          penalties,
          hideOperator: hideOperator,
          collapsed: collapsed,
        ),
      );
    }

    // Qualification
    addTile(
      "Qualification de l'infraction",
      widget.natinf.qualificationInfraction,
      useBadge: false,
    );

    // Nature de l'infraction avec icône info si actionnable
    {
      final value = widget.natinf.natureInfraction;
      if (value != null && value.isNotEmpty && value != "N/A") {
        final lowerValue = value.toLowerCase();
        final bool actionable =
            (lowerValue.contains('contravention')) ||
            (lowerValue.contains('délit') &&
                widget.natinf.amendeCategory == NatinfAmendeCategory.afd);

        normalTiles.add(
          ListTile(
            title: Row(
              mainAxisSize: MainAxisSize.min,
              children: [
                const Text(
                  "Nature de l'infraction",
                  style: TextStyle(fontWeight: FontWeight.bold),
                ),
                if (actionable) ...[
                  const SizedBox(width: 6),
                  Tooltip(
                    message: "Plus d’infos",
                    child: Icon(
                      Icons.info_outline,
                      size: 18,
                      color: Theme.of(context).colorScheme.primary,
                    ),
                  ),
                ],
              ],
            ),
            subtitle: buildBadge(value),
            onTap:
                actionable
                    ? () => showAmendePopup(context, widget.natinf)
                    : null,
          ),
        );
      }
    }

    addPenaltiesTile(
      'Peines principales',
      widget.natinf.peinesPrincipales,
      hideOperator: true,
    );
    if (widget.natinf.tentative != null) {
      final isPunishable = widget.natinf.tentative == true;
      addTile(
        'Tentative',
        isPunishable ? 'Oui' : 'Non',
        useBadge: false,
        moveToMoreInfo: !isPunishable,
        inlineValue: true,
      );
    }
    addPenaltiesTile(
      'Peines complémentaires',
      widget.natinf.peinesComplementaires,
      hideOperator: false,
      collapsed: true,
    );

    // Textes juridiques simples
    addTile(
      "Définie par",
      widget.natinf.definiePar,
      showInfo:
          widget.natinf.definiePar != null &&
          widget.natinf.definiePar!.trim().isNotEmpty,
      onTap: () {
        Navigator.push(
          context,
          MaterialPageRoute(
            builder: (_) => NatinfLegalTextsPage(natinf: widget.natinf),
          ),
        );
      },
    );
    addTile(
      "Réprimée par",
      widget.natinf.reprimeePar,
      showInfo:
          widget.natinf.reprimeePar != null &&
          widget.natinf.reprimeePar!.trim().isNotEmpty,
      onTap: () {
        Navigator.push(
          context,
          MaterialPageRoute(
            builder: (_) => NatinfLegalTextsPage(natinf: widget.natinf),
          ),
        );
      },
    );

    // Mode de rédaction
    String? modesProcesVerbalSubtitle;
    if (widget.natinf.modesProcesVerbal != null) {
      if (widget.natinf.modesProcesVerbal is List) {
        modesProcesVerbalSubtitle = (widget.natinf.modesProcesVerbal
                as List<dynamic>)
            .map((mode) {
              switch (mode) {
                case 0:
                  return 'CRI';
                case 1:
                  return 'PVE';
                case 2:
                  return 'PVO';
                default:
                  return mode.toString();
              }
            })
            .join(' ou ');
      } else {
        modesProcesVerbalSubtitle = widget.natinf.modesProcesVerbal.toString();
      }
    }
    addTile("Mode de rédaction", modesProcesVerbalSubtitle);

    // Autres champs
    addTile(
      "Retrait points",
      widget.natinf.retraitPoints?.toString(),
      moveToMoreInfo: widget.natinf.retraitPoints == 0,
    );
    addTile("Contrôle alcoolémie", widget.natinf.controleAlcoolemie?.value);
    addTile(
      "Dépistage imprégnation alcoolique",
      widget.natinf.depistageImpregnationAlcoolique?.value,
    );
    addTile("Dépistage stupéfiants", widget.natinf.depistageStupefiants?.value);
    addTile(
      "Rétention permis de conduire",
      widget.natinf.retentionPermisConduire?.value,
    );
    addTile(
      "Immobilisation du véhicule",
      widget.natinf.immobilisationVehicule?.value,
    );
    addTile("Mise en fourrière", widget.natinf.miseEnFourriere?.value);
    addTile(
      "Élément matériel",
      widget.natinf.elementMateriel,
      useBadge: false,
      makeLinks: true,
    );
    addTile(
      "Élément moral",
      widget.natinf.elementMoral,
      useBadge: false,
      makeLinks: true,
    );
    addTile(
      "Observations",
      widget.natinf.observations,
      useBadge: false,
      makeLinks: true,
    );

    // Références
    final refs = widget.natinf.references;
    if (refs != null && refs.isNotEmpty) {
      final bodyStyle = Theme.of(context).textTheme.bodyMedium!;
      addCustomTile(
        'Références',
        DefaultTextStyle(
          style: bodyStyle,
          child: Column(
            crossAxisAlignment: CrossAxisAlignment.start,
            children:
                refs.map((ref) {
                  final title = ref.title ?? ref.url;
                  final date =
                      ref.publicationDate != null
                          ? ' (${ref.publicationDate!.toIso8601String().split('T').first})'
                          : '';
                  final desc =
                      (ref.description?.isNotEmpty ?? false)
                          ? ' – ${ref.description}'
                          : '';
                  return Padding(
                    padding: const EdgeInsets.only(bottom: 4),
                    child: RichText(
                      text: TextSpan(
                        style: bodyStyle,
                        children: [
                          const TextSpan(text: '• '),
                          TextSpan(
                            text: title,
                            style: bodyStyle.copyWith(
                              color: Theme.of(context).colorScheme.primary,
                              decoration: TextDecoration.underline,
                            ),
                            recognizer:
                                TapGestureRecognizer()
                                  ..onTap = () => _launchURL(ref.url),
                          ),
                          TextSpan(text: date),
                          TextSpan(text: desc),
                        ],
                      ),
                    ),
                  );
                }).toList(),
          ),
        ),
      );
    }

    if (unauthorizedDocsPro) {
      addCustomTile(
        'DocPro',
        isAuthenticated
            ? _buildDocProRestrictedMessage(context, webUrl)
            : DocProLoginPrompt(
              onLoggedIn:
                  () => ScaffoldMessenger.of(context).showSnackBar(
                    const SnackBar(
                      content: Text(
                        'Connecté, relancez la recherche pour DocPro',
                      ),
                    ),
                  ),
            ),
      );
    } else if ((docsProAvailable || (auth.session?.hasDocPro ?? false)) &&
        widget.natinf.numeroNatinf != null) {
      addCustomTile('DocPro', _buildDocProSection(context));
    }

    // Lien web
    final linkTile = ListTile(
      title: const Text(
        "Accéder à plus de détails en ligne",
        style: TextStyle(
          color: Colors.blue,
          decoration: TextDecoration.underline,
        ),
      ),
      onTap: () => _launchURL(webUrl),
    );

    // Construction finale de la liste
    final List<Widget> allTiles = [
      ...normalTiles,
      if (bottomTiles.isNotEmpty)
        ExpansionTile(
          key: const PageStorageKey('moreInfo'),
          title: Text(
            _expandedMore
                ? "Masquer les informations"
                : "Afficher plus d'informations…",
            style: const TextStyle(fontWeight: FontWeight.bold),
          ),
          trailing: Icon(_expandedMore ? Icons.expand_less : Icons.expand_more),
          onExpansionChanged: (open) => setState(() => _expandedMore = open),
          children: [...bottomTiles, linkTile],
        )
      else
        linkTile,
    ];

    return Scaffold(
      appBar: AppBar(
        title: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text('NATINF ${widget.natinf.numeroNatinf ?? "N/A"}'),
            if (widget.natinf.obsoleteDate != null)
              Text(
                'Obsolète depuis ${widget.natinf.obsoleteDate!.toIso8601String().split('T').first}',
                style: const TextStyle(
                  color: Colors.red,
                  fontSize: 12,
                  fontWeight: FontWeight.bold,
                ),
              ),
          ],
        ),
        actions: [
          Tooltip(
            message: 'Partager',
            child: InkResponse(
              onTap: _copyNATINFoUrlToClipboard,
              onLongPress: _showShareDialog,
              radius: 24,
              child: const Padding(
                padding: EdgeInsets.all(12),
                child: Icon(Icons.share),
              ),
            ),
          ),
          Tooltip(
            message:
                _isFavourite ? 'Supprimer des favoris' : 'Ajouter aux favoris',
            child: InkResponse(
              radius: 24,
              onTap: _toggleFavourite,
              onLongPress: _handleFavouriteLongPress,
              child: Padding(
                padding: const EdgeInsets.all(12),
                child: Icon(_isFavourite ? Icons.star : Icons.star_border),
              ),
            ),
          ),
        ],
      ),
      body: Padding(
        padding: const EdgeInsets.all(16.0),
        child: ListView(children: allTiles),
      ),
    );
  }

  Widget _buildDocProRestrictedMessage(BuildContext context, String webUrl) {
    final bodyStyle = Theme.of(context).textTheme.bodyMedium!;
    final linkStyle = bodyStyle.copyWith(
      color: Theme.of(context).colorScheme.primary,
      decoration: TextDecoration.underline,
    );
    return RichText(
      text: TextSpan(
        style: bodyStyle,
        children: [
          const TextSpan(
            text:
                "Des documents professionnels existent pour cette infraction, "
                "mais votre compte n'y a pas accès pour l'instant. Vérifiez vos droits DocPro ou consultez-les sur ",
          ),
          TextSpan(
            text: 'le site',
            style: linkStyle,
            recognizer:
                TapGestureRecognizer()..onTap = () => _launchURL(webUrl),
          ),
          const TextSpan(text: '.'),
        ],
      ),
    );
  }

  Widget _buildDocProSection(BuildContext context) {
    if (_loadingDocPro) {
      return const Padding(
        padding: EdgeInsets.symmetric(vertical: 8),
        child: Center(child: CircularProgressIndicator()),
      );
    }
    if (_docProError != null) {
      return Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Text(_docProError!, style: const TextStyle(color: Colors.red)),
          TextButton(
            onPressed: _maybeLoadDocPro,
            child: const Text('Réessayer'),
          ),
        ],
      );
    }
    final docs = _docProDocs;
    if (docs == null) {
      return const Text('Chargement de DocPro…');
    }
    if (docs.isEmpty) {
      return const Text('Aucun document professionnel fourni pour ce NATINF.');
    }
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children:
          docs.map((doc) {
            final subtitleParts = <String>[];
            if (doc.publicationDate != null) {
              subtitleParts.add(
                'Publi\u00e9 le ${doc.publicationDate!.toLocal().toIso8601String().split('T').first}',
              );
            }
            if (doc.description != null && doc.description!.isNotEmpty) {
              subtitleParts.add(doc.description!);
            }
            final subtitle =
                subtitleParts.isEmpty ? null : subtitleParts.join(' \u2013 ');
            return _DocProLinkTile(
              doc: doc,
              subtitle: subtitle,
              onOpen: () => _launchURL(doc.url),
            );
          }).toList(),
    );
  }
}

class _DocProLinkTile extends StatefulWidget {
  const _DocProLinkTile({
    required this.doc,
    required this.onOpen,
    this.subtitle,
  });

  final NatinfDocPro doc;
  final VoidCallback onOpen;
  final String? subtitle;

  @override
  State<_DocProLinkTile> createState() => _DocProLinkTileState();
}

class _DocProLinkTileState extends State<_DocProLinkTile> {
  bool _expanded = false;

  @override
  Widget build(BuildContext context) {
    final title = widget.doc.title ?? widget.doc.url;
    final display = _expanded ? title : _summarizeDocProTitle(title);

    return ListTile(
      contentPadding: EdgeInsets.zero,
      title: GestureDetector(
        behavior: HitTestBehavior.opaque,
        onTap: () => setState(() => _expanded = !_expanded),
        child: Text(display, style: Theme.of(context).textTheme.bodyLarge),
      ),
      subtitle: widget.subtitle != null ? Text(widget.subtitle!) : null,
      trailing: IconButton(
        icon: const Icon(Icons.launch, size: 18),
        tooltip: 'Ouvrir dans le navigateur',
        onPressed: widget.onOpen,
      ),
      onTap: widget.onOpen,
    );
  }
}

String _ellipsizeMiddle(String text, {int maxLength = 36}) {
  return _summarizeDocProTitle(text, maxSide: (maxLength / 2).floor());
}

String _summarizeDocProTitle(String text, {int maxSide = 20}) {
  final parts = text.split('>');
  final arrowCount = parts.length - 1;
  if (arrowCount >= 2) {
    final prefix = parts.first.trim();
    final suffix = parts.last.trim();
    final condensed = '$prefix > ... > $suffix';
    return condensed.length < text.length ? condensed : text;
  }
  if (text.length <= (maxSide * 2) + 7) {
    return text;
  }
  final start = text.substring(0, maxSide);
  final end = text.substring(text.length - maxSide);
  return '$start > ... > $end';
}
