import 'package:flutter/material.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/database_helper.dart';
import 'package:natinfo_flutter/features/natinf/presentation/widgets/natinf_collection_layout.dart';
import 'package:natinfo_flutter/shared/services/matomo_service.dart';

enum SortOption { defaultOrder, oldestFirst, mostConsulted, leastConsulted }

class HistoriquePage extends StatefulWidget {
  const HistoriquePage({super.key});

  @override
  _HistoriquePageState createState() => _HistoriquePageState();
}

class _HistoriquePageState extends State<HistoriquePage> {
  List<Map<String, dynamic>> _historyRecords = [];
  List<Natinf> _natinfHistory = [];
  SortOption _sortOption = SortOption.defaultOrder;
  bool _isLoading = true;

  @override
  void initState() {
    super.initState();
    _loadHistory();
    MatomoService().trackPage(title: 'Historique', path: '/history/');
  }

  Future<void> _loadHistory() async {
    final records = await DatabaseHelper().getHistory();
    final futures =
        records.map((rec) async {
          return await DatabaseHelper().getNatinfByNumero(rec['numero_natinf']);
        }).toList();
    final results = await Future.wait(futures);
    setState(() {
      _historyRecords = records;
      _natinfHistory = results.whereType<Natinf>().toList();
      _isLoading = false;
    });
  }

  Future<void> _confirmClearHistory() async {
    final shouldClear = await showDeletionConfirmationDialog(
      context: context,
      message: "Voulez-vous vraiment supprimer l'historique ?",
    );
    if (shouldClear) {
      await DatabaseHelper().clearHistory();
      setState(() {
        _historyRecords.clear();
        _natinfHistory.clear();
      });
    }
  }

  List<Natinf> _applySort() {
    switch (_sortOption) {
      case SortOption.oldestFirst:
        return List<Natinf>.from(_natinfHistory.reversed);
      case SortOption.mostConsulted:
        return sortNatinfsByConsultations(
          historyRecords: _historyRecords,
          natinfList: _natinfHistory,
          descending: true,
        );
      case SortOption.leastConsulted:
        return sortNatinfsByConsultations(
          historyRecords: _historyRecords,
          natinfList: _natinfHistory,
          descending: false,
        );
      case SortOption.defaultOrder:
      default:
        return List<Natinf>.from(_natinfHistory);
    }
  }

  @override
  Widget build(BuildContext context) {
    final displayList = _applySort();
    return Scaffold(
      appBar: AppBar(
        title: const Text("Historique"),
        actions: [
          IconButton(
            icon: const Icon(Icons.delete),
            onPressed: _confirmClearHistory,
          ),
        ],
      ),
      body: NatinfCollectionLayout<SortOption>(
        isLoading: _isLoading,
        sortOption: _sortOption,
        onSortChanged: (value) => setState(() => _sortOption = value),
        sortItems: [
          DropdownMenuItem(
            value: SortOption.defaultOrder,
            child: Text('Du plus récent au plus ancien'),
          ),
          DropdownMenuItem(
            value: SortOption.oldestFirst,
            child: Text('Du plus ancien au plus récent'),
          ),
          DropdownMenuItem(
            value: SortOption.mostConsulted,
            child: Text('Les plus consultés'),
          ),
          DropdownMenuItem(
            value: SortOption.leastConsulted,
            child: Text('Les moins consultés'),
          ),
        ],
        natinfList: displayList,
        recordHistory: false,
      ),
    );
  }
}
