import 'dart:io';

import 'package:flutter/material.dart';
import 'package:natinfo_flutter/features/downloads/presentation/widgets/download_screen.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_download_exception.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_repository.dart';
import 'package:natinfo_flutter/features/natinf/presentation/pages/natinf_home_page.dart';
import 'package:natinfo_flutter/shared/data_sources/source_logger.dart';
import 'package:sembast/sembast.dart';
import 'package:url_launcher/url_launcher.dart';

class DataInitializationScreen extends StatefulWidget {
  final NatinfRepository repository;
  final WidgetBuilder? homeBuilder;

  const DataInitializationScreen({
    super.key,
    required this.repository,
    this.homeBuilder,
  });

  @override
  _DataInitializationScreenState createState() =>
      _DataInitializationScreenState();
}

enum _InitPhase { checking, prompt, downloading, error }

class _DataInitializationScreenState extends State<DataInitializationScreen> {
  _InitPhase _phase = _InitPhase.checking;
  String _status = "Vérification des données locales…";
  double _progress = 0.0;
  final SourceLogger _logger = SourceLogger(tag: 'Init');

  @override
  void initState() {
    super.initState();
    _ensureInitialized();
  }

  Future<void> _ensureInitialized() async {
    try {
      final hasData = await widget.repository.hasLocalData();
      if (hasData) {
        await _navigateToHome();
        return;
      }
      setState(() {
        _phase = _InitPhase.prompt;
      });
    } catch (error, stackTrace) {
      _logger.warn('Local data check failed: $error\n$stackTrace');
      setState(() {
        _phase = _InitPhase.error;
        _status = _initErrorMessage(error);
      });
    }
  }

  Future<void> _startDownload() async {
    setState(() {
      _phase = _InitPhase.downloading;
      _status = "Préparation du téléchargement…";
      _progress = 0.0;
    });
    try {
      await widget.repository.fetchInitialData(
        onStatus: (msg) {
          setState(() {
            _status = msg;
          });
        },
        onProgress: (progressFraction) {
          setState(() {
            _progress = progressFraction;
          });
        },
        onPhase: (phase) {
          setState(() {
            _phase = _InitPhase.downloading;
          });
        },
      );
      await _navigateToHome();
    } on NatinfDownloadException catch (error, stackTrace) {
      _logger.warn(
        'Initial download failed (${error.type.name}): ${error.message}'
        '${error.cause != null ? ' (cause: ${error.cause})' : ''}\n'
        '$stackTrace',
      );
      setState(() {
        _phase = _InitPhase.error;
        _status = _statusForDownloadError(error);
      });
    } catch (error, stackTrace) {
      _logger.warn('Initial download failed: $error\n$stackTrace');
      setState(() {
        _phase = _InitPhase.error;
        _status =
            "Erreur inattendue lors du téléchargement.\n"
            "Détail: ${error.toString()}\n"
            "Réessayez ou ouvrez une issue.";
      });
    }
  }

  Future<void> _navigateToHome() async {
    if (!mounted) return;
    final builder = widget.homeBuilder ?? (_) => const NatinfHomePage();
    await Navigator.pushReplacement(
      context,
      MaterialPageRoute(builder: builder),
    );
  }

  void _retry() {
    if (_phase == _InitPhase.prompt) {
      _startDownload();
      return;
    }
    _ensureInitialized();
  }

  Future<void> _launchIssueUrl() async {
    final Uri url = Uri.parse(
      "https://codeberg.org/retiolus/natinfo_flutter/issues",
    );
    if (await canLaunchUrl(url)) {
      await launchUrl(url, mode: LaunchMode.externalApplication);
    }
  }

  @override
  Widget build(BuildContext context) {
    switch (_phase) {
      case _InitPhase.checking:
        return DownloadScreen(
          status: _status,
          progress: 0,
          phase: 0,
          onRetry: _retry,
          onIssueTap: _launchIssueUrl,
        );
      case _InitPhase.prompt:
        return _PromptPane(onDownload: _startDownload, onSkip: _navigateToHome);
      case _InitPhase.downloading:
        return DownloadScreen(
          status: _status,
          progress: _progress,
          phase: 2,
          onRetry: _retry,
          onIssueTap: _launchIssueUrl,
        );
      case _InitPhase.error:
        return DownloadScreen(
          status: _status,
          progress: _progress,
          phase: 0,
          onRetry: _retry,
          onIssueTap: _launchIssueUrl,
          onSkip: _navigateToHome,
        );
    }
  }

  String _initErrorMessage(Object error) {
    if (error is DatabaseException || error is FileSystemException) {
      return "Erreur lors de l'initialisation locale.\n"
          "Impossible de lire le stockage. Vérifiez l'espace libre puis réessayez.";
    }
    return "Erreur lors de la vérification des données locales.\n"
        "Réessayez ou ouvrez une issue si le problème persiste.";
  }

  String _statusForDownloadError(NatinfDownloadException error) {
    switch (error.type) {
      case NatinfDownloadIssueType.network:
        return "Erreur réseau lors du téléchargement.\n"
            "${error.message}\n"
            "Vérifiez votre connexion Internet.";
      case NatinfDownloadIssueType.parsing:
        return "Erreur lors du téléchargement: données invalides.\n"
            "${error.message}\n"
            "Réessayez plus tard ou ouvrez une issue.";
      case NatinfDownloadIssueType.storage:
        return "Erreur de stockage pendant l'import.\n"
            "${error.message}\n"
            "Vérifiez l'espace libre puis réessayez.";
      case NatinfDownloadIssueType.unknown:
        return "Erreur inconnue lors du téléchargement.\n"
            "${error.message}\n"
            "Réessayez ou ouvrez une issue.";
    }
  }
}

class _PromptPane extends StatelessWidget {
  const _PromptPane({required this.onDownload, required this.onSkip});

  final VoidCallback onDownload;
  final Future<void> Function() onSkip;

  @override
  Widget build(BuildContext context) {
    final theme = Theme.of(context);
    return Scaffold(
      backgroundColor: theme.colorScheme.surface,
      body: SafeArea(
        child: Center(
          child: SingleChildScrollView(
            padding: const EdgeInsets.symmetric(horizontal: 24, vertical: 16),
            child: ConstrainedBox(
              constraints: const BoxConstraints(maxWidth: 520),
              child: DecoratedBox(
                decoration: BoxDecoration(
                  color: theme.cardColor,
                  borderRadius: BorderRadius.circular(16),
                  boxShadow: kElevationToShadow[3],
                ),
                child: Padding(
                  padding: const EdgeInsets.symmetric(
                    horizontal: 24,
                    vertical: 28,
                  ),
                  child: Column(
                    mainAxisSize: MainAxisSize.min,
                    crossAxisAlignment: CrossAxisAlignment.center,
                    children: [
                      CircleAvatar(
                        radius: 32,
                        backgroundColor: theme.colorScheme.primary.withOpacity(
                          0.1,
                        ),
                        child: Icon(
                          Icons.cloud_download,
                          size: 36,
                          color: theme.colorScheme.primary,
                        ),
                      ),
                      const SizedBox(height: 16),
                      Text(
                        "Télécharger les données NATINF ?",
                        style: theme.textTheme.titleLarge?.copyWith(
                          fontWeight: FontWeight.bold,
                        ),
                        textAlign: TextAlign.center,
                      ),
                      const SizedBox(height: 12),
                      Text(
                        "Vous pouvez récupérer tout le jeu de données pour un usage hors ligne complet.",
                        style: theme.textTheme.bodyMedium,
                        textAlign: TextAlign.center,
                      ),
                      const SizedBox(height: 12),
                      const _BenefitRow(
                        icon: Icons.offline_pin,
                        text: "Tout consulter hors ligne après téléchargement.",
                      ),
                      const _BenefitRow(
                        icon: Icons.sync_alt,
                        text:
                            "Sans téléchargement, les recherches en ligne rempliront progressivement la base locale.",
                      ),
                      const _BenefitRow(
                        icon: Icons.warning_amber_rounded,
                        text:
                            "Certaines références resteront indisponibles hors connexion tant que le téléchargement complet n'est pas lancé.",
                      ),
                      const _BenefitRow(
                        icon: Icons.settings,
                        text:
                            "Vous pourrez déclencher le téléchargement complet plus tard depuis les réglages.",
                      ),
                      const SizedBox(height: 24),
                      SizedBox(
                        width: double.infinity,
                        child: ElevatedButton.icon(
                          icon: const Icon(Icons.cloud_download),
                          onPressed: onDownload,
                          label: const Text("Télécharger maintenant"),
                        ),
                      ),
                      const SizedBox(height: 8),
                      SizedBox(
                        width: double.infinity,
                        child: OutlinedButton(
                          onPressed: onSkip,
                          child: const Text(
                            "Continuer avec la recherche en ligne",
                            textAlign: TextAlign.center,
                          ),
                        ),
                      ),
                      const SizedBox(height: 8),
                      TextButton(
                        onPressed:
                            () => launchUrl(
                              Uri.parse(
                                "https://codeberg.org/retiolus/natinfo_flutter/issues",
                              ),
                              mode: LaunchMode.externalApplication,
                            ),
                        child: const Text("Ouvrir une issue sur Codeberg"),
                      ),
                      const SizedBox(height: 16),
                      Text(
                        "Cette application n'est pas un service officiel de l'État français "
                        "et n'est ni affiliée, ni soutenue par l'administration publique.",
                        textAlign: TextAlign.center,
                        style: theme.textTheme.bodySmall?.copyWith(
                          color: theme.textTheme.bodySmall?.color?.withOpacity(
                            0.7,
                          ),
                          fontStyle: FontStyle.italic,
                        ),
                      ),
                    ],
                  ),
                ),
              ),
            ),
          ),
        ),
      ),
    );
  }
}

class _BenefitRow extends StatelessWidget {
  const _BenefitRow({required this.icon, required this.text});

  final IconData icon;
  final String text;

  @override
  Widget build(BuildContext context) {
    final theme = Theme.of(context);
    return Padding(
      padding: const EdgeInsets.symmetric(vertical: 6),
      child: Row(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Icon(icon, size: 18, color: theme.colorScheme.primary),
          const SizedBox(width: 8),
          Expanded(child: Text(text, style: theme.textTheme.bodyMedium)),
        ],
      ),
    );
  }
}
