import 'package:flutter/material.dart';
import 'package:provider/provider.dart';

import 'package:natinfo_flutter/features/natinf/data/database_helper.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_repository.dart';

class NatDatasetStatus {
  const NatDatasetStatus({required this.hasData, required this.description});

  final bool hasData;
  final String description;
}

/// Service pour les données NATINF (API + base locale).
class NatDataService {
  static const String defaultApiBase = 'https://natinfo.app/api';

  static Future<void> downloadNatData({
    required BuildContext context,
    Function(double)? onProgress,
    Function(String)? onStatus,
  }) async {
    try {
      final repo = Provider.of<NatinfRepository>(context, listen: false);
      await repo.fetchInitialData(
        onStatus: onStatus ?? (_) {},
        onProgress: onProgress ?? (_) {},
        onPhase: (_) {},
      );
      onStatus?.call('Téléchargement terminé.');
    } catch (e) {
      onStatus?.call('Erreur : ${e.toString()}');
      rethrow;
    }
  }

  static Future<void> downloadNatCategories({
    required BuildContext context,
    Function(String)? onStatus,
  }) async {
    try {
      final repo = Provider.of<NatinfRepository>(context, listen: false);
      onStatus?.call('Téléchargement des catégories...');
      final categories = await repo.fetchRootCategories();
      onStatus?.call('${categories.length} catégories synchronisées.');
    } catch (e) {
      onStatus?.call('Erreur : ${e.toString()}');
      rethrow;
    }
  }

  static Future<void> deleteNatData() async {
    await DatabaseHelper().clearNatinfData();
  }

  static Future<void> deleteNatCategories() async {
    await DatabaseHelper().clearCategories();
  }

  static Future<NatDatasetStatus> getNatinfStatus() async {
    try {
      final db = DatabaseHelper();
      await db.initDb();
      final count = await db.countItems();
      if (count == 0) {
        return const NatDatasetStatus(
          hasData: false,
          description: 'Non téléchargé',
        );
      }
      return NatDatasetStatus(hasData: true, description: '$count infractions');
    } catch (e) {
      return NatDatasetStatus(
        hasData: false,
        description: 'Erreur : ${e.toString()}',
      );
    }
  }

  static Future<NatDatasetStatus> getCategoriesStatus() async {
    try {
      final db = DatabaseHelper();
      await db.initDb();
      final count = await db.countCategories();
      if (count == 0) {
        return const NatDatasetStatus(
          hasData: false,
          description: 'Non téléchargé',
        );
      }
      return NatDatasetStatus(hasData: true, description: '$count catégories');
    } catch (e) {
      return NatDatasetStatus(
        hasData: false,
        description: 'Erreur : ${e.toString()}',
      );
    }
  }
}
