import 'package:flutter/foundation.dart';

import '../data/auth_repository.dart';
import '../domain/auth_session.dart';

/// Simple notifier exposing authentication state to the UI.
class AuthProvider extends ChangeNotifier {
  AuthProvider(this._repository);

  final AuthRepository _repository;
  AuthSession? _session;
  bool _loading = false;

  /// Latest session.
  AuthSession? get session => _session;

  /// Whether a login/refresh action is in progress.
  bool get isLoading => _loading;

  /// Initializes the provider by loading any persisted session.
  Future<void> bootstrap({bool forceOffline = false}) async {
    _session = await _repository.loadSession(forceOffline: forceOffline);
    notifyListeners();
  }

  /// Attempts to log in with the provided credentials.
  Future<void> login({
    required String username,
    required String password,
    String? tokenLabel,
  }) async {
    _setLoading(true);
    try {
      _session = await _repository.login(
        username: username,
        password: password,
        tokenLabel: tokenLabel,
      );
    } finally {
      _setLoading(false);
    }
    notifyListeners();
  }

  /// Explicitly refreshes the current session.
  Future<void> refresh() async {
    _setLoading(true);
    try {
      _session = await _repository.refreshToken();
    } finally {
      _setLoading(false);
    }
    notifyListeners();
  }

  /// Logs out and clears persisted tokens.
  Future<void> logout() async {
    _setLoading(true);
    try {
      await _repository.logout();
      _session = null;
    } finally {
      _setLoading(false);
    }
    notifyListeners();
  }

  void _setLoading(bool value) {
    if (_loading != value) {
      _loading = value;
      notifyListeners();
    }
  }
}
