import 'dart:async';

import 'package:app_links/app_links.dart';
import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:natinfo_flutter/features/natinf/data/database_helper.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_repository.dart';
import 'package:natinfo_flutter/features/natinf/presentation/pages/natinf_details_page.dart';

class DeepLinkService {
  DeepLinkService({required NatinfRepository repository})
    : _repository = repository;

  final NatinfRepository _repository;
  final GlobalKey<NavigatorState> navigatorKey = GlobalKey<NavigatorState>();
  final AppLinks _appLinks = AppLinks();
  StreamSubscription<Uri>? _uriSubscription;
  String? _lastOpenedNumero;
  String? _processingNumero;
  bool _isDisposed = false;

  bool get _supportsAppLinks {
    if (kIsWeb) return false;
    switch (defaultTargetPlatform) {
      case TargetPlatform.android:
      case TargetPlatform.iOS:
        return true;
      default:
        return false;
    }
  }

  Future<void> initialize() async {
    if (!_supportsAppLinks) {
      debugPrint('Deep links not supported on $defaultTargetPlatform.');
      return;
    }
    await _handleInitialUri();
    try {
      _uriSubscription = _appLinks.uriLinkStream.listen(
        _scheduleUriHandling,
        onError: (Object error) {
          debugPrint('app_links stream error: $error');
          _scheduleSnackBar("Impossible d'ouvrir le lien reçu.");
        },
      );
    } on MissingPluginException catch (error) {
      debugPrint('app_links listen missing plugin: $error');
    }
  }

  Future<void> _handleInitialUri() async {
    if (!_supportsAppLinks) return;
    try {
      final initialUri = await _appLinks.getInitialLink();
      if (initialUri != null) {
        _scheduleUriHandling(initialUri);
      }
    } on PlatformException catch (err) {
      debugPrint('Failed to get initial uri: $err');
      _scheduleSnackBar("Lien initial invalide.");
    } on FormatException catch (err) {
      debugPrint('Malformed initial uri: $err');
      _scheduleSnackBar("Lien initial invalide.");
    } on MissingPluginException catch (err) {
      debugPrint('app_links initial link missing plugin: $err');
    }
  }

  void _scheduleUriHandling(Uri uri) {
    WidgetsBinding.instance.addPostFrameCallback((_) {
      if (_isDisposed) return;
      _handleUri(uri);
    });
  }

  Future<void> _handleUri(Uri uri) async {
    if (!_isSupportedNatinfoUri(uri)) {
      _scheduleSnackBar("Lien NATINF invalide.");
      return;
    }

    final numero = _extractNumero(uri);
    final shouldFavourite = _shouldAddToFavourites(uri);
    if (numero == null || numero.isEmpty) {
      _scheduleSnackBar("Numéro NATINF manquant.");
      return;
    }

    final alreadyOpen = _lastOpenedNumero == numero;
    if (alreadyOpen && !shouldFavourite) {
      return;
    }

    if (_processingNumero == numero) {
      return;
    }

    final previousNumero = _lastOpenedNumero;
    _processingNumero = numero;
    if (!alreadyOpen) {
      _lastOpenedNumero = numero;
    }

    try {
      final natinf = await _repository.getNatinfByNumero(numero);
      if (natinf == null) {
        if (!alreadyOpen && _lastOpenedNumero == numero) {
          _lastOpenedNumero = previousNumero;
        }
        _scheduleSnackBar("NATINF $numero introuvable.");
        return;
      }
      if (shouldFavourite) {
        await _addToFavourites(numero);
      }
      if (alreadyOpen || _lastOpenedNumero != numero) {
        return;
      }
      navigatorKey.currentState
          ?.push(
            MaterialPageRoute<void>(
              builder: (_) => NatinfDetail(natinf: natinf),
            ),
          )
          .whenComplete(() {
            if (!_isDisposed && _lastOpenedNumero == numero) {
              _lastOpenedNumero = null;
            }
          });
    } catch (err) {
      if (!alreadyOpen && _lastOpenedNumero == numero) {
        _lastOpenedNumero = previousNumero;
      }
      debugPrint('Failed to open NATINF $numero from uri $uri: $err');
      _scheduleSnackBar("Erreur lors de l'ouverture du lien.");
    } finally {
      if (_processingNumero == numero) {
        _processingNumero = null;
      }
    }
  }

  bool _isSupportedNatinfoUri(Uri uri) {
    final scheme = uri.scheme.toLowerCase();
    if (scheme == 'natinf') {
      return true;
    }
    if (scheme == 'https') {
      final host = uri.host.toLowerCase();
      if (host == 'natinfo.app' || host == 'www.natinfo.app') {
        final segments = uri.pathSegments;
        if (segments.isNotEmpty && segments.first.toLowerCase() == 'plus') {
          return true;
        }
      }
    }
    return false;
  }

  String? _extractNumero(Uri uri) {
    final numeroParam = uri.queryParameters['numero']?.trim();
    if (numeroParam != null && RegExp(r'^\d+$').hasMatch(numeroParam)) {
      return numeroParam;
    }

    final candidates = <String>[];
    if (uri.host.isNotEmpty) {
      candidates.add(uri.host);
    }
    for (final segment in uri.pathSegments) {
      if (segment.isNotEmpty) {
        candidates.add(segment);
      }
    }

    for (final candidate in candidates) {
      final value = candidate.trim();
      if (value.isEmpty) continue;
      if (RegExp(r'^\d+$').hasMatch(value)) {
        return value;
      }
    }
    return null;
  }

  bool _shouldAddToFavourites(Uri uri) {
    const favTokens = {'fav', 'favorite', 'favourite', 'favoris'};
    final hostToken = uri.host.toLowerCase();
    if (hostToken.isNotEmpty && favTokens.contains(hostToken)) {
      return true;
    }
    for (final segment in uri.pathSegments) {
      if (favTokens.contains(segment.toLowerCase())) {
        return true;
      }
    }
    final action = uri.queryParameters['action']?.toLowerCase();
    if (action != null && favTokens.contains(action)) {
      return true;
    }
    final favParam = uri.queryParameters['fav']?.toLowerCase();
    if (favParam == 'true' || favParam == '1') {
      return true;
    }
    return false;
  }

  Future<void> _addToFavourites(String numero) async {
    try {
      final db = DatabaseHelper();
      if (await db.isFavourite(numero)) {
        _scheduleSnackBar("NATINF $numero est déjà dans les favoris.");
        return;
      }
      await db.addFavourites(numero);
      _scheduleSnackBar("NATINF $numero ajouté aux favoris.");
    } catch (error) {
      debugPrint('Failed to add NATINF $numero to favourites: $error');
      _scheduleSnackBar("Impossible d'ajouter le NATINF aux favoris.");
    }
  }

  void _scheduleSnackBar(String message) {
    WidgetsBinding.instance.addPostFrameCallback((_) {
      if (_isDisposed) return;
      final context = navigatorKey.currentContext;
      if (context == null) {
        return;
      }
      ScaffoldMessenger.of(
        context,
      ).showSnackBar(SnackBar(content: Text(message)));
    });
  }

  void dispose() {
    _isDisposed = true;
    _uriSubscription?.cancel();
  }
}
