package net.retiolus.natinfo

import android.content.ComponentName
import android.os.Build
import io.flutter.embedding.android.FlutterActivity
import io.flutter.embedding.engine.FlutterEngine
import io.flutter.plugin.common.MethodChannel

class MainActivity : FlutterActivity() {
    override fun configureFlutterEngine(flutterEngine: FlutterEngine) {
        super.configureFlutterEngine(flutterEngine)

        MethodChannel(flutterEngine.dartExecutor.binaryMessenger, CHANNEL).setMethodCallHandler { call, result ->
            when (call.method) {
                "isSupported" -> result.success(isDynamicIconsSupported())
                "getCurrentIcon" -> result.success(getCurrentIconKey())
                "setIcon" -> {
                    val key = call.argument<String>("iconKey")
                    if (key == null) {
                        result.error("invalid_arguments", "Paramètre 'iconKey' manquant.", null)
                    } else {
                        try {
                            setIcon(key)
                            result.success(null)
                        } catch (error: IllegalArgumentException) {
                            result.error("invalid_icon", error.message, null)
                        }
                    }
                }
                else -> result.notImplemented()
            }
        }
    }

    private fun isDynamicIconsSupported(): Boolean {
        return Build.VERSION.SDK_INT >= Build.VERSION_CODES.O
    }

    private fun getCurrentIconKey(): String {
        val packageManager = packageManager
        iconAliasMapping.forEach { (key, alias) ->
            val componentName = ComponentName(this, "$packageName.$alias")
            val state = packageManager.getComponentEnabledSetting(componentName)
            if (state == android.content.pm.PackageManager.COMPONENT_ENABLED_STATE_ENABLED) {
                return key
            }
            if (key == DEFAULT_ICON_KEY && state == android.content.pm.PackageManager.COMPONENT_ENABLED_STATE_DEFAULT) {
                return key
            }
        }
        return DEFAULT_ICON_KEY
    }

    private fun setIcon(key: String) {
        if (!isDynamicIconsSupported()) {
            throw IllegalArgumentException("Changement d'icône non supporté sur cette version d'Android.")
        }
        val alias = iconAliasMapping[key] ?: throw IllegalArgumentException("Icône inconnue: $key")
        val packageManager = packageManager

        iconAliasMapping.forEach { (entryKey, entryAlias) ->
            val componentName = ComponentName(this, "$packageName.$entryAlias")
            val newState =
                if (entryKey == key) android.content.pm.PackageManager.COMPONENT_ENABLED_STATE_ENABLED
                else android.content.pm.PackageManager.COMPONENT_ENABLED_STATE_DISABLED
            packageManager.setComponentEnabledSetting(
                componentName,
                newState,
                android.content.pm.PackageManager.DONT_KILL_APP
            )
        }

        // Ensure the selected alias is enabled (redundant but explicit).
        val selectedComponent = ComponentName(this, "$packageName.$alias")
        packageManager.setComponentEnabledSetting(
            selectedComponent,
            android.content.pm.PackageManager.COMPONENT_ENABLED_STATE_ENABLED,
            android.content.pm.PackageManager.DONT_KILL_APP
        )
    }

    companion object {
        private const val CHANNEL = "net.retiolus.natinfo/app_icon"
        private const val DEFAULT_ICON_KEY = "classic"

        private val iconAliasMapping = mapOf(
            "classic" to "MainActivityAliasClassic",
            "tricolore_fr" to "MainActivityAliasTricoloreFr",
            "blue" to "MainActivityAliasBlue",
            "graphite" to "MainActivityAliasGraphite",
            "green" to "MainActivityAliasGreen",
            "indigo" to "MainActivityAliasIndigo",
            "lime" to "MainActivityAliasLime",
            "navy" to "MainActivityAliasNavy",
            "orange" to "MainActivityAliasOrange",
            "pink" to "MainActivityAliasPink",
            "purple" to "MainActivityAliasPurple",
            "red" to "MainActivityAliasRed",
            "teal" to "MainActivityAliasTeal",
            "yellow" to "MainActivityAliasYellow",
            "pride_ace" to "MainActivityAliasPrideAce",
            "pride_bi" to "MainActivityAliasPrideBi",
            "pride_pan" to "MainActivityAliasPridePan",
            "pride_progress" to "MainActivityAliasPrideProgress",
            "pride_rainbow" to "MainActivityAliasPrideRainbow",
            "pride_trans" to "MainActivityAliasPrideTrans",
        )
    }
}
