import 'dart:io';

import 'package:analyzer/dart/analysis/utilities.dart';
import 'package:analyzer/dart/ast/ast.dart' hide Declaration;
import 'package:dart_style/dart_style.dart';
import 'package:ffigen/ffigen.dart';
import 'package:ffigen/src/config_provider/config_types.dart';
import 'package:ffigen/src/config_provider/spec_utils.dart';
import 'package:ffigen/src/code_generator.dart';
import 'package:ffigen/src/header_parser.dart' as ffigen;
import 'package:pub_semver/pub_semver.dart';

import 'wasm_symbols.dart';

void main() {
  final generator = FfiGen();
  generator.run(_GenerationMode.shared.resolveConfig());
  generator.run(_GenerationMode.native.resolveConfig());
  generator.run(_GenerationMode.dynamicLibrary.resolveConfig());

  postProcess();
  writeWasmDefinitions();
}

void postProcess() {
  final formatter = DartFormatter(languageVersion: Version(3, 6, 0));

  // Generate a shared interface for the `@Native` and `DynamicLibrary.lookup`
  // bindings.
  final sharedLibrary = StringBuffer()
    ..writeln()
    ..writeln(r'''
abstract interface class SharedSymbolAddresses {
  ffi.Pointer<
          ffi.NativeFunction<
              ffi.Int Function(ffi.Pointer<sqlite3_changeset_iter>)>>
      get sqlite3changeset_finalize;
  ffi.Pointer<
          ffi.NativeFunction<
              ffi.Void Function(ffi.Pointer<sqlite3_session>)>>
      get sqlite3session_delete;
  ffi.Pointer<ffi.NativeFunction<ffi.Void Function(ffi.Pointer<ffi.Void>)>>
      get sqlite3_free;
}
''')
    ..writeln('abstract interface class SqliteLibrary {')
    ..writeln('  SharedSymbolAddresses get addresses;');
  final nativeImplementation = StringBuffer()
    ..writeln('// Generated by tool/generate_bindings.dart')
    ..writeln('// ignore_for_file: type=lint')
    ..writeln("import 'dart:ffi' as ffi;")
    ..writeln("import 'native.dart' as native;")
    ..writeln("import 'shared.dart';")
    ..writeln('final class NativeAssetsLibrary implements SqliteLibrary {')
    ..writeln('@override')
    ..writeln('SharedSymbolAddresses get addresses => native.addresses;');

  final dynamicBindingsFile =
      File('lib/src/ffi/generated/dynamic_library.dart');
  final nativeBindingsFile = File('lib/src/ffi/generated/native.dart');
  final dynamicBindingsContents = dynamicBindingsFile.readAsStringSync();

  String patchSource(String source) {
    // The generated bindings use `imp$1.` as a prefix to reference members in
    // the shared Dart file. We're appending to that file though, so no need to
    // include that.
    return source.replaceAll(r'imp$1.', '');
  }

  final generatedBindings = parseString(content: dynamicBindingsContents).unit;
  // Generated `class nativeLibrary
  final nativeLibrary =
      generatedBindings.childEntities.whereType<ClassDeclaration>().first;
  for (final entry in nativeLibrary.members) {
    if (entry is MethodDeclaration) {
      void writeDefinition(StringBuffer buffer) {
        if (entry.isGetter) {
          buffer
            ..write(patchSource(entry.returnType!.toSource()))
            ..write(' get ')
            ..write(entry.name.lexeme);
        } else if (entry.isSetter) {
          buffer
            ..write('set ')
            ..write(entry.name.lexeme)
            ..write(patchSource(entry.parameters!.toSource()));
        } else {
          buffer
            ..write(patchSource(entry.returnType!.toSource()))
            ..write(' ')
            ..write(entry.name.lexeme)
            ..write(patchSource(entry.parameters!.toSource()));
        }
      }

      writeDefinition(sharedLibrary);
      sharedLibrary.writeln(';');

      nativeImplementation.writeln('@override');
      writeDefinition(nativeImplementation);
      nativeImplementation.writeln('{');
      if (!entry.isSetter) {
        nativeImplementation.write('return ');
      }
      nativeImplementation
        ..write('native.')
        ..write(entry.name.lexeme);

      if (entry.isSetter) {
        nativeImplementation
            .write(' = ${entry.parameters!.parameters.single.name!.lexeme}');
      } else if (!entry.isGetter) {
        final forwardedArgs =
            entry.parameters!.parameters.map((e) => e.name!.lexeme).join(', ');
        nativeImplementation.write('($forwardedArgs)');
      }

      nativeImplementation.writeln(';}');
    }
  }

  sharedLibrary.writeln('}');
  nativeImplementation.writeln('}');

  final sharedFile = File('lib/src/ffi/generated/shared.dart');
  final originalSharedBindings = sharedFile.readAsStringSync();
  sharedFile.writeAsStringSync(
      formatter.format('$originalSharedBindings$sharedLibrary'));

  dynamicBindingsFile.writeAsStringSync(dynamicBindingsContents
      .replaceFirst('class NativeLibrary',
          r'final class NativeLibrary implements imp$1.SqliteLibrary')
      .replaceFirst(
        'class _SymbolAddresses',
        r'final class _SymbolAddresses implements imp$1.SharedSymbolAddresses',
      ));

  nativeBindingsFile
      .writeAsStringSync(nativeBindingsFile.readAsStringSync().replaceFirst(
            'class _SymbolAddresses',
            r'final class _SymbolAddresses implements imp$1.SharedSymbolAddresses',
          ));

  File('lib/src/ffi/generated/native_library.dart')
      .writeAsStringSync(formatter.format(nativeImplementation.toString()));
}

enum _GenerationMode {
  dynamicLibrary('dynamic_library.dart'),
  native('native.dart'),
  shared('shared.dart');

  final String output;

  const _GenerationMode(this.output);

  Uri get outputDartUri => Uri.parse('lib/src/ffi/generated/').resolve(output);

  Config resolveConfig() {
    final libraryImports = <String, LibraryImport>{};
    Map<String, ImportedType> typeMappings = <String, ImportedType>{};

    if (this != shared) {
      typeMappings = symbolFileImportExtractor(
          [symbolUri.toString()], libraryImports, null, null);
    }

    return Config(
      preamble: '// ignore_for_file: type=lint',
      ffiNativeConfig: FfiNativeConfig(
          enabled: this == native,
          assetId: 'package:sqlite3_native_assets/sqlite3_native_assets.dart'),
      output: outputDartUri,
      entryPoints: [Uri.parse('assets/sqlite3.h')],
      symbolFile: this == shared
          ? SymbolFile(Uri.parse('shared.dart'), symbolUri)
          : null,
      structDecl: _includeSqlite3Only,
      functionDecl: this == shared ? _includeNothing : _includeSqlite3Only,
      globals: this == shared ? _includeNothing : _includeSqlite3Only,
      varArgFunctions: makeVarArgFunctionsMapping({
        'sqlite3_db_config': [
          RawVarArgFunction('', ['int', 'int*'])
        ],
      }, libraryImports),
      libraryImports: libraryImports.values.toList(),
      usrTypeMappings: typeMappings,
    );
  }

  static Uri symbolUri = Uri.parse('lib/src/ffi/generated/shared_symbols.yml');

  static DeclarationFilters _includeSqlite3Only = DeclarationFilters(
      shouldInclude: (d) => d.isSqlite3Symbol,
      shouldIncludeSymbolAddress: (decl) {
        return switch (decl.originalName) {
          'sqlite3changeset_finalize' => true,
          'sqlite3session_delete' => true,
          'sqlite3_free' => true,
          _ => false,
        };
      });

  static DeclarationFilters _includeNothing =
      DeclarationFilters(shouldInclude: (d) => false);
}

extension on Declaration {
  bool get isSqlite3Symbol => originalName.startsWith('sqlite3');
}

void writeWasmDefinitions() {
  final filter = DeclarationFilters(
      shouldInclude: (d) =>
          stableFunctions.contains(d.originalName) ||
          unstable.contains(d.originalName));

  final library = ffigen.parse(Config(
    output: Uri.parse('unused'),
    entryPoints: [
      Uri.parse('assets/sqlite3.h'),
      Uri.parse('assets/sqlite3_dart_wasm.h'),
    ],
    functionDecl: filter,
    globals: filter,
  ));
  final buffer = StringBuffer('''
import 'dart:js_interop';

import 'js_interop.dart';

// ignore_for_file: non_constant_identifier_names
typedef Pointer = int;

/// Typed wrappers around the symbols exported by SQLite.
///
/// Generated by `tool/generate_bindings.dart`.
extension type SqliteExports(JSObject raw) implements JSObject {
''');

  void writeType(Type type) {
    const bigIntInJs = ['Int64', 'Uint64', 'Long'];

    switch (type) {
      case PointerType():
        buffer.write('Pointer /*<${type.getNativeType().trim()}>*/');
        return;
      case NativeType():
        if (bigIntInJs.contains(type.toString())) {
          buffer.write('JSBigInt');
          return;
        }
      case ImportedType():
        if (bigIntInJs.contains(type.cType)) {
          buffer.write('JSBigInt');
          return;
        }
    }

    buffer.write(type.getDartType(library.writer));
  }

  for (final binding in library.bindings) {
    final mayBeAbsent = !stableFunctions.contains(binding.originalName);

    switch (binding) {
      case Func():
        buffer.write('external ');
        if (mayBeAbsent) {
          buffer.writeln('JSFunction? get ${binding.originalName};');
        } else {
          final type = binding.functionType;
          writeType(type.returnType);
          buffer.write(' ${binding.originalName}(');

          for (final (i, arg) in type.parameters.indexed) {
            if (i != 0) {
              buffer.write(', ');
            }

            writeType(arg.type);
            buffer.write(' ${arg.name}');
          }

          buffer.writeln(');');
        }

      case Global():
        var jsType = 'Global';
        if (mayBeAbsent) {
          jsType += '?';
        }

        buffer
          ..write('external $jsType get ')
          ..write(binding.originalName)
          ..writeln(';');
    }
  }

  buffer.writeln('}');

  final formatter = DartFormatter(languageVersion: Version(3, 6, 0));
  File('lib/src/wasm/sqlite3_wasm.g.dart')
      .writeAsStringSync(formatter.format(buffer.toString()));
}
