import 'dart:async';

import 'package:flutter/material.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:natinfo_flutter/app/config/app_config.dart';
import 'package:natinfo_flutter/app/state/operational_mode_provider.dart';
import 'package:natinfo_flutter/app/theme/theme_provider.dart';
import 'package:natinfo_flutter/features/auth/data/auth_repository.dart';
import 'package:natinfo_flutter/features/auth/data/auth_storage.dart';
import 'package:natinfo_flutter/features/auth/presentation/auth_provider.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_repository.dart';
import 'package:natinfo_flutter/features/natinf/presentation/pages/natinf_home_page.dart';
import 'package:http/http.dart' as http;
import 'package:http/testing.dart';
import 'package:provider/provider.dart';
import 'package:shared_preferences/shared_preferences.dart';

class _StubNatinfRepository extends NatinfRepository {
  _StubNatinfRepository({this.searchCompleter});

  final Completer<List<Natinf>>? searchCompleter;

  @override
  Future<List<Natinf>> searchNatinf(
    String query, {
    void Function(String p1)? onStatus,
    bool forceOffline = false,
  }) {
    if (searchCompleter != null) {
      return searchCompleter!.future;
    }
    return Future.value(const []);
  }

  @override
  Future<void> recordHistoryEntry(String numero) async {}

  @override
  bool isDocsProUnauthorized(String numero) => false;
}

void main() {
  TestWidgetsFlutterBinding.ensureInitialized();
  late AuthProvider authProvider;

  setUp(() async {
    SharedPreferences.setMockInitialValues({});
    AppConfig.forceOffline = false;
    authProvider = await _buildAuthProvider();
  });

  Widget buildHome(
    NatinfRepository repository, {
    Future<List<Natinf>> Function()? recentHistoryLoader,
  }) {
    return MultiProvider(
      providers: [
        Provider<NatinfRepository>.value(value: repository),
        ChangeNotifierProvider<AuthProvider>.value(value: authProvider),
        ChangeNotifierProvider(create: (_) => ThemeProvider()),
        ChangeNotifierProvider(create: (_) => OperationalModeProvider()),
      ],
      child: MaterialApp(
        home: NatinfHomePage(recentHistoryLoader: recentHistoryLoader),
      ),
    );
  }

  Future<void> pumpFor(WidgetTester tester, {int ticks = 8}) async {
    for (var i = 0; i < ticks; i++) {
      await tester.pump(const Duration(milliseconds: 50));
    }
  }

  testWidgets('shows welcome with quick actions and recent history', (
    WidgetTester tester,
  ) async {
    const recentHistory = [
      Natinf(
        id: 1,
        numeroNatinf: '1001',
        qualificationInfraction: 'Infraction test',
        dacgSource: 0,
      ),
    ];

    await tester.pumpWidget(
      buildHome(
        _StubNatinfRepository(),
        recentHistoryLoader: () async => recentHistory,
      ),
    );
    await pumpFor(tester);

    expect(find.byKey(const Key('natinf_home_welcome')), findsOneWidget);
    expect(
      find.text(
        'Recherchez une NATINF par numéro ou mots-clés, ou accédez directement aux sections clés.',
      ),
      findsOneWidget,
    );
    expect(find.text('Catégories'), findsOneWidget);
    expect(find.text('Favoris'), findsOneWidget);
    expect(find.text('Historique'), findsNothing);
    expect(find.byKey(const ValueKey('recent_1001')), findsOneWidget);
  });

  testWidgets('hides welcome state while loading and after results', (
    WidgetTester tester,
  ) async {
    AppConfig.forceOffline = true;
    final completer = Completer<List<Natinf>>();
    final repository = _StubNatinfRepository(searchCompleter: completer);

    await tester.pumpWidget(
      buildHome(repository, recentHistoryLoader: () async => const []),
    );
    await pumpFor(tester);

    await tester.enterText(find.byType(TextField), 'offline query');
    await tester.tap(find.byIcon(Icons.search));
    await tester.pump();

    expect(find.byKey(const Key('natinf_home_welcome')), findsNothing);

    completer.complete(const [
      Natinf(
        id: 2,
        numeroNatinf: '2002',
        qualificationInfraction: 'Résultat en attente',
        dacgSource: 0,
      ),
    ]);
    await pumpFor(tester);

    expect(find.byKey(const Key('natinf_home_welcome')), findsNothing);
    expect(find.text('Résultat en attente'), findsOneWidget);
  });
}

Future<AuthProvider> _buildAuthProvider() async {
  final repository = AuthRepository(
    baseUri: Uri.parse('https://natinfo.app/api'),
    storage: InMemoryAuthStorage(),
    httpClient: MockClient((request) async => http.Response('{}', 200)),
  );
  final provider = AuthProvider(repository);
  await provider.bootstrap();
  return provider;
}
