import 'package:flutter/material.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/database_helper.dart';
import 'package:natinfo_flutter/app/navigation/app_route_observer.dart';
import 'package:natinfo_flutter/features/natinf/presentation/widgets/favourite_folder_content_view.dart';
import 'package:natinfo_flutter/features/natinf/presentation/widgets/favourite_folder_dialogs.dart';
import 'package:natinfo_flutter/features/natinf/presentation/widgets/natinf_collection_layout.dart';
import 'package:natinfo_flutter/features/natinf/data/favourite_folder_service.dart';
import 'package:natinfo_flutter/shared/services/matomo_service.dart';
import 'package:natinfo_flutter/features/natinf/domain/entities/favourite_folder.dart';

enum SortOption {
  ascendingNum,
  descendingNum,
  newestFirst,
  oldestFirst,
  mostConsulted,
  leastConsulted,
}

class FavouritesPage extends StatefulWidget {
  const FavouritesPage({super.key});

  @override
  _FavouritesPageState createState() => _FavouritesPageState();
}

class _FavouritesPageState extends State<FavouritesPage>
    with RouteAware, SingleTickerProviderStateMixin {
  List<Map<String, dynamic>> _favouritesRecords = [];
  List<Map<String, dynamic>> _historyRecords = [];
  List<Natinf> _natinfFavourites = [];
  SortOption _sortOption = SortOption.mostConsulted;
  bool _isLoading = true;
  bool _isLoadingFolders = true;
  List<FavouriteFolderNode> _folderNodes = [];
  List<int> _folderPath = [];
  late final TabController _tabController;

  @override
  void initState() {
    super.initState();
    _tabController = TabController(length: 2, vsync: this)..addListener(() {
      setState(() {});
    });
    _loadFavourites();
    MatomoService().trackPage(title: 'Favoris', path: '/favourites/');
  }

  @override
  void didChangeDependencies() {
    super.didChangeDependencies();
    final route = ModalRoute.of(context);
    if (route != null) {
      appRouteObserver.unsubscribe(this);
      appRouteObserver.subscribe(this, route);
    }
  }

  @override
  void dispose() {
    appRouteObserver.unsubscribe(this);
    _tabController.dispose();
    super.dispose();
  }

  Future<void> _loadFavourites() async {
    setState(() {
      _isLoading = true;
      _isLoadingFolders = true;
    });
    final records = await DatabaseHelper().getFavourites();
    final historyRecords = await DatabaseHelper().getHistory();
    final futures =
        records.map((rec) async {
          return await DatabaseHelper().getNatinfByNumero(rec['numero_natinf']);
        }).toList();
    final results = await Future.wait(futures);
    final folders = await DatabaseHelper().getAllFavouriteFolders();
    final folderMap = await DatabaseHelper().getFolderNatinfMap();
    final folderContents = await _loadFolderContents(folderMap);
    final nodes = FavouriteFolderService.buildTree(
      folders: folders,
      folderContents: folderContents,
    );
    final normalizedPath = _normalizePath(_folderPath, nodes);
    if (!mounted) return;
    setState(() {
      _favouritesRecords = records;
      _historyRecords = historyRecords;
      _natinfFavourites = results.whereType<Natinf>().toList();
      _folderNodes = nodes;
      _folderPath = normalizedPath;
      _isLoading = false;
      _isLoadingFolders = false;
    });
  }

  @override
  void didPopNext() {
    _loadFavourites();
  }

  Future<void> _confirmClearFavourites() async {
    final shouldClear = await showDeletionConfirmationDialog(
      context: context,
      message: "Voulez-vous vraiment supprimer les favoris ?",
    );
    if (shouldClear) {
      await DatabaseHelper().clearFavourites();
      setState(() {
        _favouritesRecords.clear();
        _natinfFavourites.clear();
        _isLoadingFolders = true;
      });
      await _reloadFolders();
    }
  }

  Future<Map<int, List<Natinf>>> _loadFolderContents(
    Map<int, List<String>> folderMap,
  ) async {
    final helper = DatabaseHelper();
    final result = <int, List<Natinf>>{};
    for (final entry in folderMap.entries) {
      final futures =
          entry.value
              .where((numero) => numero.isNotEmpty)
              .map(helper.getNatinfByNumero)
              .toList();
      final fetched = await Future.wait(futures);
      result[entry.key] =
          fetched.whereType<Natinf>().toList()..sort(
            (a, b) => (a.numeroNatinf ?? '').compareTo(b.numeroNatinf ?? ''),
          );
    }
    return result;
  }

  Future<void> _reloadFolders() async {
    final folders = await DatabaseHelper().getAllFavouriteFolders();
    final folderMap = await DatabaseHelper().getFolderNatinfMap();
    final folderContents = await _loadFolderContents(folderMap);
    final nodes = FavouriteFolderService.buildTree(
      folders: folders,
      folderContents: folderContents,
    );
    final normalizedPath = _normalizePath(_folderPath, nodes);
    if (!mounted) return;
    setState(() {
      _folderNodes = nodes;
      _folderPath = normalizedPath;
      _isLoadingFolders = false;
    });
  }

  Future<void> _createFolder({
    required BuildContext context,
    FavouriteFolder? parent,
  }) async {
    final name = await showFolderNameInputDialog(
      context: context,
      title: parent == null ? 'Nouveau dossier' : 'Nouveau sous-dossier',
      confirmLabel: 'Créer',
    );
    if (name == null) return;
    await DatabaseHelper().createFavouriteFolder(
      name: name,
      parentId: parent?.id,
    );
    if (!mounted) return;
    setState(() {
      _isLoadingFolders = true;
    });
    await _reloadFolders();
    if (!context.mounted) return;
    ScaffoldMessenger.of(
      context,
    ).showSnackBar(SnackBar(content: Text('Dossier "$name" créé.')));
  }

  Future<void> _renameFolder({
    required BuildContext context,
    required FavouriteFolder folder,
  }) async {
    final name = await showFolderNameInputDialog(
      context: context,
      title: 'Renommer le dossier',
      confirmLabel: 'Renommer',
      initialValue: folder.name,
    );
    if (name == null) return;
    await DatabaseHelper().renameFavouriteFolder(folder.id, name);
    if (!mounted) return;
    setState(() {
      _isLoadingFolders = true;
    });
    await _reloadFolders();
    if (!context.mounted) return;
    ScaffoldMessenger.of(
      context,
    ).showSnackBar(SnackBar(content: Text('Dossier renommé en "$name".')));
  }

  Future<bool> _deleteFolder({
    required BuildContext context,
    required FavouriteFolder folder,
  }) async {
    final shouldDelete = await showDeletionConfirmationDialog(
      context: context,
      message: 'Supprimer le dossier "${folder.name}" ?',
    );
    if (!shouldDelete) {
      return false;
    }
    await DatabaseHelper().deleteFavouriteFolder(folder.id);
    final newPath = _truncatePathAfter(folder.id);
    if (!mounted) {
      return true;
    }
    setState(() {
      _isLoadingFolders = true;
      _folderPath = newPath;
    });
    await _reloadFolders();
    if (!context.mounted) {
      return true;
    }
    ScaffoldMessenger.of(context).showSnackBar(
      SnackBar(content: Text('Dossier "${folder.name}" supprimé.')),
    );
    return true;
  }

  Future<void> _removeNatinfFromFolder({
    required BuildContext context,
    required FavouriteFolder folder,
    required Natinf natinf,
  }) async {
    final numero = natinf.numeroNatinf;
    if (numero == null) return;
    await DatabaseHelper().removeNatinfFromFolder(folder.id, numero);
    if (!mounted) return;
    setState(() {
      _isLoadingFolders = true;
    });
    await _reloadFolders();
    if (!context.mounted) return;
    ScaffoldMessenger.of(context).showSnackBar(
      SnackBar(content: Text('NATINF $numero retiré de "${folder.name}".')),
    );
  }

  String _formatFolderSubtitle(FavouriteFolderNode node) {
    final parts = <String>[];
    final subCount = node.children.length;
    if (subCount > 0) {
      parts.add('$subCount sous-dossier${subCount > 1 ? 's' : ''}');
    }
    final natCount = node.natinfs.length;
    if (natCount > 0) {
      parts.add('$natCount NATINF');
    }
    if (parts.isEmpty) {
      return 'Dossier vide';
    }
    return parts.join(' · ');
  }

  List<int> _normalizePath(List<int> path, List<FavouriteFolderNode> roots) {
    final normalized = <int>[];
    for (final id in path) {
      final node = FavouriteFolderService.findNodeById(roots, id);
      if (node == null) {
        break;
      }
      normalized.add(id);
    }
    return normalized;
  }

  FavouriteFolderNode? get _currentFolderNode {
    if (_folderPath.isEmpty) {
      return null;
    }
    return FavouriteFolderService.findNodeById(_folderNodes, _folderPath.last);
  }

  void _openFolder(FavouriteFolderNode node) {
    setState(() {
      _folderPath = [..._folderPath, node.folder.id];
    });
  }

  void _exitCurrentFolder() {
    if (_folderPath.isEmpty) {
      return;
    }
    setState(() {
      _folderPath = _folderPath.sublist(0, _folderPath.length - 1);
    });
  }

  List<int> _truncatePathAfter(int folderId) {
    final index = _folderPath.indexOf(folderId);
    if (index == -1) {
      return List<int>.from(_folderPath);
    }
    return _folderPath.sublist(0, index);
  }

  Widget _buildFolderCard(FavouriteFolderNode node) {
    return Card(
      margin: const EdgeInsets.symmetric(horizontal: 12, vertical: 6),
      child: ListTile(
        leading: const Icon(Icons.folder),
        title: Text(node.folder.name),
        subtitle: Text(_formatFolderSubtitle(node)),
        onTap: () => _openFolder(node),
        trailing: PopupMenuButton<String>(
          onSelected: (value) async {
            switch (value) {
              case 'rename':
                await _renameFolder(context: context, folder: node.folder);
                break;
              case 'delete':
                await _deleteFolder(context: context, folder: node.folder);
                break;
            }
          },
          itemBuilder:
              (context) => const [
                PopupMenuItem(value: 'rename', child: Text('Renommer')),
                PopupMenuItem(value: 'delete', child: Text('Supprimer')),
              ],
        ),
      ),
    );
  }

  Widget _buildRootFoldersView() {
    if (_folderNodes.isEmpty) {
      return Center(
        child: Column(
          mainAxisSize: MainAxisSize.min,
          children: const [
            Icon(Icons.create_new_folder_outlined, size: 48),
            SizedBox(height: 16),
            Text(
              'Utilisez le bouton en haut pour créer votre premier dossier.',
              textAlign: TextAlign.center,
            ),
            SizedBox(height: 8),
            Text(
              "Ajoutez ensuite des NATINF depuis leur fiche en maintenant l'étoile du bouton favoris.",
              textAlign: TextAlign.center,
            ),
          ],
        ),
      );
    }
    return ListView(
      padding: const EdgeInsets.only(top: 8, bottom: 80),
      children: _folderNodes.map(_buildFolderCard).toList(),
    );
  }

  Widget _buildFoldersTab() {
    if (_isLoadingFolders) {
      return const Center(child: CircularProgressIndicator());
    }
    final currentNode = _currentFolderNode;
    final rootView = _buildRootFoldersView();
    if (currentNode == null) {
      return AnimatedSwitcher(
        duration: const Duration(milliseconds: 250),
        child: KeyedSubtree(
          key: const ValueKey<String>('root-folders'),
          child: rootView,
        ),
      );
    }
    return AnimatedSwitcher(
      duration: const Duration(milliseconds: 250),
      child: KeyedSubtree(
        key: ValueKey<int>(currentNode.folder.id),
        child: FavouriteFolderContentView(
          node: currentNode,
          subtitle: _formatFolderSubtitle(currentNode),
          onBack: _exitCurrentFolder,
          onRenameCurrent:
              () => _renameFolder(context: context, folder: currentNode.folder),
          onDeleteCurrent: () async {
            await _deleteFolder(context: context, folder: currentNode.folder);
          },
          onRemoveNatinf:
              (natinf) => _removeNatinfFromFolder(
                context: context,
                folder: currentNode.folder,
                natinf: natinf,
              ),
          subfolderCards: currentNode.children.map(_buildFolderCard).toList(),
        ),
      ),
    );
  }

  List<Natinf> _applySort() {
    switch (_sortOption) {
      case SortOption.ascendingNum:
        var list = List<Natinf>.from(_natinfFavourites);
        list.sort(
          (a, b) => (a.numeroNatinf ?? '').compareTo(b.numeroNatinf ?? ''),
        );
        return list;
      case SortOption.descendingNum:
        var list = List<Natinf>.from(_natinfFavourites);
        list.sort(
          (a, b) => (b.numeroNatinf ?? '').compareTo(a.numeroNatinf ?? ''),
        );
        return list;
      case SortOption.newestFirst:
        var list = List<Natinf>.from(_natinfFavourites);
        final addedTimes = <String, int>{};
        for (var rec in _favouritesRecords) {
          addedTimes[rec['numero_natinf'].toString()] = rec['added_at'] as int;
        }
        list.sort((a, b) {
          final timeA = addedTimes[a.numeroNatinf ?? ''] ?? 0;
          final timeB = addedTimes[b.numeroNatinf ?? ''] ?? 0;
          return timeB.compareTo(timeA);
        });
        return list;
      case SortOption.oldestFirst:
        var list = List<Natinf>.from(_natinfFavourites);
        final addedTimes = <String, int>{};
        for (var rec in _favouritesRecords) {
          addedTimes[rec['numero_natinf'].toString()] = rec['added_at'] as int;
        }
        list.sort((a, b) {
          final timeA = addedTimes[a.numeroNatinf ?? ''] ?? 0;
          final timeB = addedTimes[b.numeroNatinf ?? ''] ?? 0;
          return timeA.compareTo(timeB);
        });
        return list;
      case SortOption.mostConsulted:
        return sortNatinfsByConsultations(
          historyRecords: _historyRecords,
          natinfList: _natinfFavourites,
          descending: true,
        );
      case SortOption.leastConsulted:
        return sortNatinfsByConsultations(
          historyRecords: _historyRecords,
          natinfList: _natinfFavourites,
          descending: false,
        );
      default:
        return List<Natinf>.from(_natinfFavourites);
    }
  }

  @override
  Widget build(BuildContext context) {
    final displayList = _applySort();
    final currentFolder = _currentFolderNode;
    final theme = Theme.of(context);
    final tabForegroundColor =
        theme.appBarTheme.foregroundColor ?? theme.colorScheme.onPrimary;
    final unselectedTabColor = tabForegroundColor.withValues(
      alpha: (tabForegroundColor.alpha / 255.0) * 0.7,
    );

    return Scaffold(
      appBar: AppBar(
        title: const Text("Favoris"),
        bottom: TabBar(
          controller: _tabController,
          labelColor: tabForegroundColor,
          unselectedLabelColor: unselectedTabColor,
          indicatorColor: tabForegroundColor,
          tabs: const [Tab(text: 'NATINF'), Tab(text: 'Dossiers')],
        ),
        actions: [
          if (_tabController.index == 0)
            IconButton(
              icon: const Icon(Icons.delete),
              tooltip: 'Vider les favoris',
              onPressed: _confirmClearFavourites,
            ),
          if (_tabController.index == 1)
            IconButton(
              icon: const Icon(Icons.create_new_folder_outlined),
              tooltip:
                  currentFolder == null
                      ? 'Nouveau dossier'
                      : 'Nouveau sous-dossier',
              onPressed:
                  _isLoadingFolders
                      ? null
                      : () => _createFolder(
                        context: context,
                        parent: currentFolder?.folder,
                      ),
            ),
        ],
      ),
      body: TabBarView(
        controller: _tabController,
        children: [
          NatinfCollectionLayout<SortOption>(
            isLoading: _isLoading,
            sortOption: _sortOption,
            onSortChanged: (value) => setState(() => _sortOption = value),
            sortItems: const [
              DropdownMenuItem<SortOption>(
                value: SortOption.ascendingNum,
                child: Text('Numéro croissant'),
              ),
              DropdownMenuItem<SortOption>(
                value: SortOption.descendingNum,
                child: Text('Numéro décroissant'),
              ),
              DropdownMenuItem<SortOption>(
                value: SortOption.newestFirst,
                child: Text('Les plus récents en premier'),
              ),
              DropdownMenuItem<SortOption>(
                value: SortOption.oldestFirst,
                child: Text('Les plus anciens en premier'),
              ),
              DropdownMenuItem<SortOption>(
                value: SortOption.mostConsulted,
                child: Text('Les plus consultés'),
              ),
              DropdownMenuItem<SortOption>(
                value: SortOption.leastConsulted,
                child: Text('Les moins consultés'),
              ),
            ],
            natinfList: displayList,
          ),
          _buildFoldersTab(),
        ],
      ),
    );
  }
}
