import 'dart:async';
import 'package:http/http.dart' as http;

import '../domain/auth_session.dart';

/// Minimal contract the HTTP client needs from an auth provider.
abstract class AuthClientDelegate {
  /// Current session if any.
  AuthSession? get currentSession;

  /// True when calls should skip auth (forced offline).
  bool get isOffline;

  /// Ensures the access token is still valid, refreshing when needed.
  Future<void> ensureValidAccessToken();

  /// Refreshes the tokens manually.
  Future<AuthSession?> refreshToken();
}

/// HTTP client that injects Authorization headers and retries once after a
/// successful token refresh when receiving a 401 response.
class AuthHttpClient extends http.BaseClient {
  AuthHttpClient(this._delegate, this._inner);

  final AuthClientDelegate _delegate;
  final http.Client _inner;

  @override
  Future<http.StreamedResponse> send(http.BaseRequest request) async {
    await _delegate.ensureValidAccessToken();
    final session = _delegate.currentSession;
    if (session != null && !_delegate.isOffline) {
      request.headers['Authorization'] = 'Bearer ${session.accessToken}';
    }

    final response = await _inner.send(request);
    if (response.statusCode != 401 || session == null) {
      return response;
    }

    final refreshed = await _delegate.refreshToken();
    if (refreshed == null) {
      return response;
    }

    final retry = await _retryWith(request, refreshed);
    return retry ?? response;
  }

  Future<http.StreamedResponse?> _retryWith(
    http.BaseRequest original,
    AuthSession session,
  ) async {
    if (original is http.Request) {
      final retry =
          http.Request(original.method, original.url)
            ..headers.addAll(original.headers)
            ..bodyBytes = original.bodyBytes;
      retry.headers['Authorization'] = 'Bearer ${session.accessToken}';
      return _inner.send(retry);
    }
    if (original is http.MultipartRequest) {
      final retry =
          http.MultipartRequest(original.method, original.url)
            ..headers.addAll(original.headers)
            ..fields.addAll(original.fields)
            ..files.addAll(original.files);
      retry.headers['Authorization'] = 'Bearer ${session.accessToken}';
      return _inner.send(retry);
    }
    return null;
  }
}
