import 'package:flutter/material.dart';
import 'package:flutter_test/flutter_test.dart';
import 'package:http/http.dart' as http;
import 'package:http/testing.dart';
import 'package:natinfo_flutter/app/theme/theme_provider.dart';
import 'package:natinfo_flutter/features/auth/data/auth_repository.dart';
import 'package:natinfo_flutter/features/auth/data/auth_storage.dart';
import 'package:natinfo_flutter/features/auth/presentation/auth_provider.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/favourite_folder_service.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_repository.dart';
import 'package:natinfo_flutter/features/natinf/domain/entities/favourite_folder.dart';
import 'package:natinfo_flutter/features/natinf/presentation/widgets/favourite_folder_content_view.dart';
import 'package:provider/provider.dart';

void main() {
  late AuthProvider authProvider;

  setUp(() async {
    authProvider = await _buildAuthProvider();
  });

  testWidgets('FavouriteFolderContentView renders and handles callbacks', (
    tester,
  ) async {
    final folder = FavouriteFolder(
      id: 1,
      name: 'Dossier parent',
      parentId: null,
      createdAt: DateTime(2024, 1, 1),
    );
    final natinf = const Natinf(
      numeroNatinf: '101',
      qualificationInfraction: 'Test qualification',
      dacgSource: 0,
    );
    final node = FavouriteFolderNode(
      folder: folder,
      children: const [],
      natinfs: [natinf],
    );

    var backCalled = false;
    var renameCalled = false;
    var deleteCalled = false;
    var removeCalled = false;

    await tester.pumpWidget(
      MultiProvider(
        providers: [
          ChangeNotifierProvider(create: (_) => ThemeProvider()),
          ChangeNotifierProvider<AuthProvider>.value(value: authProvider),
          Provider<NatinfRepository>.value(value: _StubNatinfRepository()),
        ],
        child: MaterialApp(
          home: FavouriteFolderContentView(
            node: node,
            subtitle: '1 NATINF',
            onBack: () => backCalled = true,
            onRenameCurrent: () async {
              renameCalled = true;
            },
            onDeleteCurrent: () async {
              deleteCalled = true;
            },
            onRemoveNatinf: (value) async {
              if (value.numeroNatinf == natinf.numeroNatinf) {
                removeCalled = true;
              }
            },
            subfolderCards: [
              Card(
                margin: const EdgeInsets.symmetric(horizontal: 12, vertical: 6),
                child: ListTile(
                  title: const Text('Sous-dossier'),
                  onTap: () {},
                ),
              ),
            ],
          ),
        ),
      ),
    );

    expect(find.text('Dossier parent'), findsOneWidget);
    expect(find.text('Sous-dossier'), findsOneWidget);
    expect(find.text('NATINF 101'), findsOneWidget);

    await tester.tap(find.byIcon(Icons.arrow_back));
    await tester.pumpAndSettle();
    expect(backCalled, isTrue);

    await tester.tap(find.byIcon(Icons.more_vert));
    await tester.pumpAndSettle();
    await tester.tap(find.text('Renommer'));
    await tester.pumpAndSettle();
    expect(renameCalled, isTrue);

    await tester.tap(find.byIcon(Icons.more_vert));
    await tester.pumpAndSettle();
    await tester.tap(find.text('Supprimer'));
    await tester.pumpAndSettle();
    expect(deleteCalled, isTrue);

    await tester.tap(find.byIcon(Icons.remove_circle_outline));
    await tester.pumpAndSettle();
    expect(removeCalled, isTrue);
  });

  testWidgets('Empty folder state displays guidance', (tester) async {
    final folder = FavouriteFolder(
      id: 2,
      name: 'Vide',
      parentId: null,
      createdAt: DateTime(2024, 2, 1),
    );

    await tester.pumpWidget(
      MultiProvider(
        providers: [
          ChangeNotifierProvider(create: (_) => ThemeProvider()),
          ChangeNotifierProvider<AuthProvider>.value(value: authProvider),
          Provider<NatinfRepository>.value(value: _StubNatinfRepository()),
        ],
        child: MaterialApp(
          home: FavouriteFolderContentView(
            node: FavouriteFolderNode(folder: folder),
            subtitle: 'Dossier vide',
            onBack: () {},
            onRenameCurrent: () async {},
            onDeleteCurrent: () async {},
            onRemoveNatinf: (_) async {},
            subfolderCards: const [],
          ),
        ),
      ),
    );

    expect(find.text('Ce dossier est vide.'), findsOneWidget);
    expect(
      find.text(
        "Créez un sous-dossier avec le bouton en haut, ou effectuez un appui long sur l'étoile des favoris d'une fiche NATINF pour l'ajouter ici.",
      ),
      findsOneWidget,
    );
    expect(find.byIcon(Icons.create_new_folder), findsNothing);
  });
}

Future<AuthProvider> _buildAuthProvider() async {
  final repository = AuthRepository(
    baseUri: Uri.parse('https://natinfo.app/api'),
    storage: InMemoryAuthStorage(),
    httpClient: MockClient((request) async => http.Response('{}', 200)),
  );
  final provider = AuthProvider(repository);
  await provider.bootstrap();
  return provider;
}

class _StubNatinfRepository extends NatinfRepository {
  @override
  bool isDocsProUnauthorized(String numero) => false;

  @override
  Future<void> recordHistoryEntry(String numero) async {}
}
