import 'dart:io';

import 'package:http/http.dart' as http;
import 'package:natinfo_flutter/shared/data_sources/source_adapter.dart';
import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';

/// Loads dataset payloads over HTTP/HTTPS.
class HttpSourceAdapter extends SourceAdapter {
  HttpSourceAdapter({http.Client? client}) : _client = client ?? http.Client();

  final http.Client _client;

  @override
  SourceType get type => SourceType.api;

  @override
  Future<SourceLoadResult> load(SourceSpec spec) async {
    if (spec.type != type) {
      throw ArgumentError.value(
        spec.type,
        'spec.type',
        'HttpSourceAdapter only supports $type',
      );
    }

    final response = await _client.get(spec.uri);
    if (response.statusCode >= 400) {
      throw HttpException(
        'Failed to fetch source "${spec.id}" (${spec.uri}), '
        'status ${response.statusCode}',
        uri: spec.uri,
      );
    }

    final bytes = response.bodyBytes;
    final integrity = verifyIntegrity(spec, bytes);
    return SourceLoadResult(spec: spec, bytes: bytes, integrity: integrity);
  }
}
