import 'package:flutter/material.dart';
import 'package:provider/provider.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/shared/services/matomo_service.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_repository.dart';
import 'package:natinfo_flutter/features/natinf/presentation/widgets/natinf_list.dart';
import 'package:natinfo_flutter/features/natinf/presentation/pages/category_view_helpers.dart';

class CategoriesPage extends StatefulWidget {
  final CategoryBrief? category;
  final int? categoryId;

  const CategoriesPage({super.key, this.category, this.categoryId});

  @override
  State<CategoriesPage> createState() => _CategoriesPageState();
}

class _CategoriesPageState extends State<CategoriesPage> {
  bool _loading = true;
  bool _isFirstLoad = false;
  String? _error;
  List<CategoryBrief>? _subCategories;
  List<Natinf>? _natinfs;
  String? _categoryDescription;
  final Map<int, String> _descriptions = {};

  @override
  void initState() {
    super.initState();
    _loadData();
    MatomoService().trackPage(title: 'Catégories', path: '/categories/');
  }

  Future<void> _loadData() async {
    final repo = Provider.of<NatinfRepository>(context, listen: false);
    setState(() {
      _loading = true;
      _error = null;
      _isFirstLoad = false;
    });
    try {
      if (widget.category == null && widget.categoryId == null) {
        final cached = await repo.getCachedRootCategories();
        if (cached.isNotEmpty) {
          final sortedCached = sortCategoryBriefs(cached);
          if (!mounted) return;
          setState(() {
            _subCategories = sortedCached;
            _loading = false;
          });
          _loadDescriptionsFor(cached);
          repo
              .fetchRootCategories()
              .then((remote) {
                if (!mounted) return;
                final sortedRemote = sortCategoryBriefs(remote);
                setState(() {
                  _subCategories = sortedRemote;
                });
                _loadDescriptionsFor(remote);
              })
              .catchError((_) {
                if (!mounted) return;
                ScaffoldMessenger.of(context).showSnackBar(
                  const SnackBar(
                    content: Text('Impossible de mettre à jour les catégories'),
                  ),
                );
              });
          return;
        }

        // No cached data: this is the initial load of root categories
        if (!mounted) return;
        setState(() {
          _isFirstLoad = true;
        });
        final rootCats = await repo.fetchRootCategories();
        final sortedRoots = sortCategoryBriefs(rootCats);
        if (!mounted) return;
        setState(() {
          _subCategories = sortedRoots;
          _loading = false;
          _isFirstLoad = false;
        });
        _loadDescriptionsFor(rootCats);
      } else {
        final id = widget.category?.id ?? widget.categoryId;
        if (id == null) {
          throw Exception('Catégorie invalide');
        }

        final cached = await repo.getCachedCategory(id);
        if (cached != null) {
          final snapshot = buildCategoryViewSnapshot(cached);
          if (!mounted) return;
          setState(() {
            _subCategories = snapshot.subCategories;
            _natinfs = snapshot.natinfs;
            _categoryDescription = snapshot.description;
            _loading = false;
          });
          _loadDescriptionsFor(cached.children ?? []);
          repo
              .fetchCategory(id)
              .then((cat) {
                if (!mounted) return;
                final newSnapshot = buildCategoryViewSnapshot(cat);
                setState(() {
                  _subCategories = newSnapshot.subCategories;
                  _natinfs = newSnapshot.natinfs;
                  _categoryDescription = newSnapshot.description;
                });
                _loadDescriptionsFor(cat.children ?? []);
              })
              .catchError((_) {
                if (!mounted) return;
                ScaffoldMessenger.of(context).showSnackBar(
                  const SnackBar(
                    content: Text('Impossible de mettre à jour la catégorie'),
                  ),
                );
              });
          return;
        }

        final cat = await repo.fetchCategory(id);
        final snapshot = buildCategoryViewSnapshot(cat);
        if (!mounted) return;
        setState(() {
          _subCategories = snapshot.subCategories;
          _natinfs = snapshot.natinfs;
          _categoryDescription = snapshot.description;
          _loading = false;
        });
        _loadDescriptionsFor(cat.children ?? []);
      }
    } catch (e) {
      final message = e.toString();
      if (!mounted) return;
      setState(() {
        _error = message;
        _loading = false;
      });
      ScaffoldMessenger.of(
        context,
      ).showSnackBar(SnackBar(content: Text(message)));
    }
  }

  @override
  Widget build(BuildContext context) {
    final title = widget.category?.name ?? 'Catégories';
    return Scaffold(
      appBar: AppBar(
        title: Text(title),
        actions: [
          if ((widget.category != null || widget.categoryId != null) &&
              (_categoryDescription ?? '').trim().isNotEmpty)
            IconButton(
              icon: const Icon(Icons.info_outline),
              tooltip: 'Description',
              onPressed: _showDescriptionSheet,
            ),
        ],
      ),
      body:
          _loading
              ? Center(
                child: Column(
                  mainAxisSize: MainAxisSize.min,
                  children: [
                    const CircularProgressIndicator(),
                    if (_isFirstLoad) ...[
                      const SizedBox(height: 12),
                      Text(
                        'Chargement des catégories…\nCela peut prendre quelques secondes.\nMerci de patienter et ne fermez pas cette page.',
                        style: Theme.of(context).textTheme.bodySmall,
                        textAlign: TextAlign.center,
                      ),
                    ],
                  ],
                ),
              )
              : _error != null
              ? Center(child: Text(_error!))
              : _buildContent(),
    );
  }

  Widget _buildContent() {
    final children = <Widget>[];
    // Description affichée via icône d'information dans l'AppBar
    if (_subCategories != null && _subCategories!.isNotEmpty) {
      children.addAll(
        _subCategories!.map(
          (c) => Card(
            margin: const EdgeInsets.symmetric(horizontal: 4, vertical: 8),
            shape: RoundedRectangleBorder(
              borderRadius: BorderRadius.circular(12),
            ),
            child: ListTile(
              title: Text(c.name),
              subtitle:
                  (c.id != null &&
                          (_descriptions[c.id!] ?? '').trim().isNotEmpty)
                      ? Text(_descriptions[c.id!]!.trim())
                      : null,
              trailing: const Icon(Icons.chevron_right),
              onTap: () {
                Navigator.push(
                  context,
                  MaterialPageRoute(
                    builder: (_) => CategoriesPage(category: c),
                  ),
                );
              },
            ),
          ),
        ),
      );
    }

    if (_natinfs != null && _natinfs!.isNotEmpty) {
      children.add(NatinfList(natinfList: _natinfs!, nested: true));
    }

    if (children.isEmpty) {
      return const Center(child: Text('Aucune donnée'));
    }

    return ListView(padding: const EdgeInsets.all(8), children: children);
  }

  void _showDescriptionSheet() {
    final text = _categoryDescription?.trim();
    if (text == null || text.isEmpty) return;
    showModalBottomSheet(
      context: context,
      isScrollControlled: true,
      shape: const RoundedRectangleBorder(
        borderRadius: BorderRadius.vertical(top: Radius.circular(16)),
      ),
      builder: (context) {
        final media = MediaQuery.of(context);
        final maxHeight = media.size.height * 0.8;
        return SafeArea(
          top: false,
          child: ConstrainedBox(
            constraints: BoxConstraints(maxHeight: maxHeight),
            child: Padding(
              padding: const EdgeInsets.fromLTRB(16, 12, 16, 16),
              child: Column(
                mainAxisSize: MainAxisSize.min,
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Center(
                    child: Container(
                      width: 36,
                      height: 4,
                      decoration: BoxDecoration(
                        color: Colors.grey.shade400,
                        borderRadius: BorderRadius.circular(2),
                      ),
                    ),
                  ),
                  const SizedBox(height: 12),
                  Text(
                    'Description',
                    style: Theme.of(context).textTheme.titleMedium?.copyWith(
                      fontWeight: FontWeight.bold,
                    ),
                  ),
                  const SizedBox(height: 8),
                  Flexible(child: SingleChildScrollView(child: Text(text))),
                ],
              ),
            ),
          ),
        );
      },
    );
  }

  Future<void> _loadDescriptionsFor(List<CategoryBrief> categories) async {
    if (!mounted || categories.isEmpty) return;
    final repo = Provider.of<NatinfRepository>(context, listen: false);
    final Map<int, String> map = {};
    for (final c in categories) {
      final id = c.id;
      if (id == null) continue;
      try {
        final cat = await repo.getCachedCategory(id);
        final desc = cat?.description?.trim();
        if (desc != null && desc.isNotEmpty) {
          map[id] = desc;
        }
      } catch (_) {
        // ignore cache errors for subtitle enrichment
      }
    }
    if (!mounted || map.isEmpty) return;
    setState(() {
      _descriptions.addAll(map);
    });
  }
}
