import 'dart:async';
import 'package:flutter/services.dart';
import 'package:natinfo_flutter/app/config/app_config.dart';
import 'package:natinfo_flutter/features/auth/data/auth_repository.dart';
import 'package:natinfo_flutter/features/auth/data/auth_storage.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_cache_service.dart';
import 'package:natinfo_flutter/features/natinf/data/remote/natinf_remote_service.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_search_fallback.dart';
import 'package:natinfo_flutter/shared/data_sources/default_source_adapters.dart';
import 'package:natinfo_flutter/shared/data_sources/dataset_ids.dart';
import 'package:natinfo_flutter/shared/data_sources/source_environment.dart';
import 'package:natinfo_flutter/shared/data_sources/source_adapter.dart';
import 'package:natinfo_flutter/shared/data_sources/source_registry.dart';
import 'package:natinfo_flutter/shared/data_sources/source_resolver.dart';
import 'package:natinfo_flutter/shared/data_sources/source_spec.dart';
import 'package:natinfo_flutter/shared/data_sources/source_preferences.dart';
import 'package:natinfo_flutter/shared/data_sources/source_logger.dart';
import 'package:http/http.dart' as http;
import 'package:shared_preferences/shared_preferences.dart';
import 'package:meta/meta.dart';

class NatinfRepository {
  NatinfRepository({
    SourceResolver? sourceResolver,
    List<SourceAdapter>? sourceAdapters,
    SourceRegistry? sourceRegistry,
    SourcePreferences? sourcePreferences,
    SourceLogger? sourceLogger,
    AuthRepository? authRepository,
    Uri? authBaseUri,
  }) : _providedResolver = sourceResolver,
       _providedAdapters = sourceAdapters,
       _providedRegistry = sourceRegistry,
       _providedSourcePrefs = sourcePreferences,
       _sourceLogger = sourceLogger ?? SourceLogger.shared,
       _providedAuthRepository = authRepository,
       _authBaseUri = authBaseUri;

  late final Swagger _apiClient;
  late final Swagger _categoriesApiClient;
  late final NatinfCacheService _cacheService;
  late final NatinfRemoteService _remoteService;
  late AuthRepository _authRepository;
  http.Client? _httpClient;
  SourceResolver? _providedResolver;
  SourceRegistry? _providedRegistry;
  final List<SourceAdapter>? _providedAdapters;
  SourcePreferences? _providedSourcePrefs;
  final SourceLogger _sourceLogger;
  late SourceResolver _resolver;
  late Map<SourceType, SourceAdapter> _adaptersByType;
  late SourceSpec _natinfSource;
  late SourceSpec _categoriesSource;
  late SourcePreferences _sourcePreferences;
  final AuthRepository? _providedAuthRepository;
  final Uri? _authBaseUri;

  Future<void> init() async {
    _cacheService = NatinfCacheService();
    await _cacheService.init();
    final environment = await SourceEnvironmentLoader.load(
      registry: _providedRegistry,
      resolver: _providedResolver,
      sourcePreferences: _providedSourcePrefs,
    );
    _resolver = environment.resolver;
    _adaptersByType = _buildAdapters(_providedAdapters);
    _sourcePreferences = environment.sourcePreferences;
    final allowNetwork = _sourcePreferences.isNetworkAllowed(DatasetIds.natinf);
    final allowNetworkCategories = _sourcePreferences.isNetworkAllowed(
      DatasetIds.natinfCategories,
    );
    _natinfSource = selectNatinfSource(
      resolver: _resolver,
      allowNetwork: allowNetwork,
      preferredSourceId: AppConfig.sourceNatinfOverride,
    );
    _categoriesSource = selectCategoriesSource(
      resolver: _resolver,
      allowNetwork: allowNetworkCategories,
      preferredSourceId: AppConfig.sourceNatinfOverride,
    );
    final adapter = _adaptersByType[_natinfSource.type];
    final categoriesAdapter = _adaptersByType[_categoriesSource.type];
    if (adapter == null) {
      throw UnsupportedError(
        'No adapter registered for source type: ${_natinfSource.type}',
      );
    }
    if (_natinfSource.type != SourceType.api) {
      throw UnsupportedError(
        'Unsupported NATINF source type: ${_natinfSource.type}',
      );
    }
    if (categoriesAdapter == null) {
      throw UnsupportedError(
        'No adapter registered for source type: ${_categoriesSource.type}',
      );
    }
    if (_categoriesSource.type != SourceType.api) {
      throw UnsupportedError(
        'Unsupported NATINF categories source type: ${_categoriesSource.type}',
      );
    }
    _sourceLogger.info(
      'NATINF dataset using source "${_natinfSource.id}" (${_natinfSource.type.name}), '
      'networkAllowed=$allowNetwork, preferred=${AppConfig.sourceNatinfOverride ?? 'default'}.',
    );
    _sourceLogger.info(
      'NATINF categories dataset using source "${_categoriesSource.id}" '
      '(${_categoriesSource.type.name}), networkAllowed=$allowNetworkCategories, '
      'preferred=${AppConfig.sourceNatinfOverride ?? 'default'}.',
    );
    _authRepository =
        _providedAuthRepository ??
        AuthRepository(
          baseUri: _authBaseUri ?? _natinfSource.uri,
          storage: SecureAuthStorage(),
        );
    await _authRepository.loadSession(forceOffline: AppConfig.forceOffline);
    _authRepository.addSessionListener(() {
      unawaited(_cacheService.clearDocsProUnauthorized());
    });
    _httpClient = _authRepository.authorizedClient;
    _apiClient = Swagger.create(
      baseUrl: _natinfSource.uri,
      httpClient: _httpClient,
    );
    _categoriesApiClient = Swagger.create(
      baseUrl: _categoriesSource.uri,
      httpClient: _httpClient,
    );
    _remoteService = NatinfRemoteService(
      apiClient: _apiClient,
      categoriesApiClient: _categoriesApiClient,
      cacheService: _cacheService,
      httpClient: _httpClient!,
      logger: _sourceLogger,
    );
  }

  Map<SourceType, SourceAdapter> _buildAdapters(List<SourceAdapter>? provided) {
    return DefaultSourceAdapters.buildMap(overrides: provided ?? const []);
  }

  Future<bool> hasLocalData() async {
    return _cacheService.hasLocalData();
  }

  bool isDocsProUnauthorized(String numero) =>
      _cacheService.isDocsProUnauthorized(numero);

  bool isDocsProAvailable(String numero) =>
      _cacheService.isDocsProAvailable(numero);

  Future<List<CategoryBrief>> getCachedRootCategories() {
    return _cacheService.getCachedRootCategories();
  }

  Future<Category?> getCachedCategory(int id) async {
    return _cacheService.getCachedCategory(id);
  }

  Future<Natinf?> getNatinfByNumero(String numero) {
    return _cacheService.getNatinfByNumero(numero);
  }

  /// Enregistre l'ouverture d'une NATINF dans l'historique local.
  Future<void> recordHistoryEntry(String numero) {
    return _cacheService.recordHistoryEntry(numero);
  }

  Future<List<CategoryBrief>> fetchRootCategories() async {
    return _remoteService.fetchRootCategories();
  }

  Future<Category> fetchCategory(int id) async {
    return _remoteService.fetchCategory(id);
  }

  Future<void> fetchInitialData({
    void Function(String)? onStatus,
    void Function(double)? onProgress,
    void Function(int)? onPhase,
  }) async {
    await _remoteService.fetchInitialData(
      onStatus: onStatus,
      onProgress: onProgress,
      onPhase: onPhase,
    );
  }

  Future<List<Natinf>> searchNatinf(
    String query, {
    void Function(String)? onStatus,
    bool forceOffline = false,
  }) async {
    final offlineMode = await resolveOfflineMode(forceOffline: forceOffline);
    if (offlineMode != OfflineMode.none) {
      final bool forced = offlineMode == OfflineMode.forced;
      onStatus?.call(
        forced
            ? "Mode hors ligne forcé : recherche immédiate en BD locale..."
            : "Aucune connexion Internet détectée (mode avion ?), "
                "recherche locale...",
      );
      final localResults = await fallbackToLocalSearch(
        query,
        localSearch: _cacheService.searchLocal,
        onStatus: onStatus,
        successStatus:
            forced
                ? "Résultats trouvés en BD locale (mode hors ligne forcé)."
                : "Résultats trouvés en BD locale (aucune connexion).",
      );
      if (localResults.isNotEmpty) {
        return localResults;
      }
      throw Exception(
        forced
            ? "Aucune NATINF trouvée en BD locale (mode hors ligne forcé)."
            : "Aucune NATINF trouvée en BD locale (aucune connexion Internet).",
      );
    }

    try {
      return await _remoteService.searchNatinf(query, onStatus: onStatus);
    } on TimeoutException catch (_) {
      onStatus?.call(
        "Timeout (10s), passage en mode hors ligne, recherche locale...",
      );
      final localResults = await fallbackToLocalSearch(
        query,
        localSearch: _cacheService.searchLocal,
        onStatus: onStatus,
        successStatus:
            "Résultats trouvés en BD locale (mode hors ligne activé en raison d'un timeout API).",
      );
      if (localResults.isNotEmpty) {
        return localResults;
      }
      rethrow;
    } catch (e) {
      onStatus?.call(
        "Erreur API, passage en mode hors ligne, recherche locale...",
      );
      final localResults = await fallbackToLocalSearch(
        query,
        localSearch: _cacheService.searchLocal,
        onStatus: onStatus,
        successStatus:
            "Résultats trouvés en BD (mode hors ligne activé à cause d'une erreur API: ${e.toString()}).",
      );
      if (localResults.isNotEmpty) {
        return localResults;
      }
      rethrow;
    }
  }

  Future<List<NatinfDocPro>> fetchDocPro(String numero) async {
    final cached = _cacheService.getDocsProDocs(numero);
    if (cached != null) {
      return cached;
    }
    final docs = await _remoteService.fetchDocPro(numero);
    await _cacheService.saveDocsProDocs(numero, docs);
    await _cacheService.updateDocsProInfo(
      numero,
      unauthorized: false,
      available: docs.isNotEmpty,
    );
    return docs;
  }

  @visibleForTesting
  SourceSpec selectNatinfSource({
    required SourceResolver resolver,
    required bool allowNetwork,
    String? preferredSourceId,
    String? customUrl,
  }) {
    if (customUrl != null && customUrl.isNotEmpty) {
      final datasetName = resolver.registry.nameForDataset('natinf');
      final parsedUri = Uri.parse(customUrl);
      final requiresNetwork =
          parsedUri.scheme == 'http' || parsedUri.scheme == 'https';
      if (!allowNetwork && requiresNetwork) {
        return resolver.resolve(
          'natinf',
          allowNetwork: allowNetwork,
          preferredSourceId: preferredSourceId,
        );
      }
      return SourceSpec(
        id: 'custom-natinf',
        dataset: 'natinf',
        name: datasetName,
        type: SourceType.api,
        uri: parsedUri,
        scope: 'natinf',
        priority: 1,
        requiredAtBuild: false,
        requiresNetwork: requiresNetwork,
        schemaVersion: 1,
      );
    }

    return resolver.resolve(
      'natinf',
      allowNetwork: allowNetwork,
      preferredSourceId: preferredSourceId,
    );
  }

  @visibleForTesting
  SourceSpec selectCategoriesSource({
    required SourceResolver resolver,
    required bool allowNetwork,
    String? preferredSourceId,
  }) {
    return resolver.resolve(
      'natinf-categories',
      allowNetwork: allowNetwork,
      preferredSourceId: preferredSourceId,
    );
  }
}
