import 'dart:async';
import 'dart:io';
import 'package:diacritic/diacritic.dart';
import 'package:natinfo_flutter/features/natinf/data/api/swagger.swagger.dart';
import 'package:natinfo_flutter/features/natinf/data/natinf_mappers.dart';
import 'package:natinfo_flutter/features/natinf/domain/entities/favourite_folder.dart';
import 'package:path/path.dart';
import 'package:path_provider/path_provider.dart';
import 'package:sembast/sembast_io.dart';

class _Scored {
  final Natinf natinf;
  final int relevance;
  final int qualificationPos;
  final int pk;

  _Scored({
    required this.natinf,
    required this.relevance,
    required this.qualificationPos,
    required this.pk,
  });
}

class DatabaseHelper {
  static final DatabaseHelper _singleton = DatabaseHelper._internal();
  factory DatabaseHelper() => _singleton;
  DatabaseHelper._internal();

  Database? _db;
  final _store = intMapStoreFactory.store('natinf');
  final _historyStore = intMapStoreFactory.store('historique');
  final _favouritesStore = intMapStoreFactory.store('favourites');
  final _favouriteFoldersStore = intMapStoreFactory.store('favourite_folders');
  final _favouriteFolderItemsStore = stringMapStoreFactory.store(
    'favourite_folder_items',
  );
  final _docsProStore = stringMapStoreFactory.store('docs_pro');
  final _referencesStore = stringMapStoreFactory.store('references');
  final _categoriesStore = intMapStoreFactory.store('categories');

  /// Initialise la base de données Sembast
  Future<void> initDb() async {
    final Directory dir = await getApplicationDocumentsDirectory();
    final dbPath = join(dir.path, 'natinfo.db');
    _db = await databaseFactoryIo.openDatabase(dbPath);
  }

  Database get _dbOrThrow {
    final db = _db;
    if (db == null) {
      throw Exception("La base de données n'est pas initialisée.");
    }
    return db;
  }

  Future<int> insertOrUpdateNatinf(Natinf natinf) async {
    final record = _store.record(natinf.id!);
    await record.put(_dbOrThrow, natinf.toJson());
    return natinf.id!;
  }

  Future<void> setDocsProInfo(
    String numero, {
    required bool unauthorized,
    required bool available,
  }) async {
    final db = _dbOrThrow;
    await _docsProStore.record(numero).put(db, {
      'unauthorized': unauthorized,
      'available': available,
    });
  }

  Future<Map<String, dynamic>?> getDocsProInfo(String numero) async {
    final db = _dbOrThrow;
    return await _docsProStore.record(numero).get(db) as Map<String, dynamic>?;
  }

  Future<Map<String, bool>> loadDocsProUnauthorizedMap() async {
    final db = _dbOrThrow;
    final records = await _docsProStore.find(db);
    final result = <String, bool>{};
    for (final rec in records) {
      final map = rec.value as Map<String, dynamic>;
      final unauth = map['unauthorized'] == true;
      result[rec.key as String] = unauth;
    }
    return result;
  }

  Future<Map<String, bool>> loadDocsProAvailableMap() async {
    final db = _dbOrThrow;
    final records = await _docsProStore.find(db);
    final result = <String, bool>{};
    for (final rec in records) {
      final map = rec.value as Map<String, dynamic>;
      final available = map['available'] == true;
      result[rec.key as String] = available;
    }
    return result;
  }

  Future<void> clearDocsPro() async {
    final db = _dbOrThrow;
    await _docsProStore.delete(db);
  }

  Future<void> setReferences(String numero, List<NatinfReference> refs) async {
    final db = _dbOrThrow;
    await _referencesStore.record(numero).put(db, {
      'list': refs.map((e) => e.toJson()).toList(),
    });
  }

  Future<List<NatinfReference>?> getReferences(String numero) async {
    final db = _dbOrThrow;
    final Map<String, dynamic>? map =
        await _referencesStore.record(numero).get(db) as Map<String, dynamic>?;
    if (map != null) {
      final data = map['list'];
      if (data is List) {
        final list =
            data
                .map(
                  (e) => NatinfReference.fromJson(
                    Map<String, dynamic>.from(e as Map),
                  ),
                )
                .toList();
        return list;
      }
    }
    return null;
  }

  Future<Map<String, List<NatinfReference>>> loadReferencesMap() async {
    final db = _dbOrThrow;
    final records = await _referencesStore.find(db);
    final result = <String, List<NatinfReference>>{};
    for (final rec in records) {
      final map = rec.value as Map<String, dynamic>?;
      if (map != null) {
        final data = map['list'];
        final list =
            data is List
                ? data
                    .map(
                      (e) => NatinfReference.fromJson(
                        Map<String, dynamic>.from(e as Map),
                      ),
                    )
                    .toList()
                : <NatinfReference>[];
        result[rec.key as String] = list;
      }
    }
    return result;
  }

  Future<int> insertOrUpdateCategory(Category category) async {
    final record = _categoriesStore.record(category.id!);
    await record.put(_dbOrThrow, category.toJson());
    return category.id!;
  }

  Future<Category?> getCategory(int id) async {
    final db = _dbOrThrow;
    final Map<String, dynamic>? map =
        await _categoriesStore.record(id).get(db) as Map<String, dynamic>?;
    if (map != null) {
      return Category.fromJson(Map<String, dynamic>.from(map));
    }
    return null;
  }

  Future<List<CategoryBrief>> getRootCategories() async {
    final db = _dbOrThrow;
    final finder = Finder(
      filter: Filter.equals('parent', null),
      sortOrders: [SortOrder('name')],
    );
    final records = await _categoriesStore.find(db, finder: finder);
    return records
        .map(
          (rec) => CategoryBrief.fromJson(
            Map<String, dynamic>.from(rec.value as Map),
          ),
        )
        .toList();
  }

  Future<void> clearCategories() async {
    final db = _dbOrThrow;
    await _categoriesStore.delete(db);
  }

  Future<int> countCategories() async {
    final db = _dbOrThrow;
    return _categoriesStore.count(db);
  }

  Future<Natinf?> getNatinfByNumero(String numero) async {
    final db = _dbOrThrow;
    final finder = Finder(filter: Filter.equals('numero_natinf', numero));
    final record = await _store.findFirst(db, finder: finder);
    if (record != null) {
      final map = Map<String, dynamic>.from(record.value as Map);
      var natinf = parseStoredNatinf(map);
      final refs = await getReferences(numero);
      if (refs != null && refs.isNotEmpty) {
        natinf = natinf.copyWith(references: refs);
      }
      return natinf;
    }
    return null;
  }

  Future<int> countItems() async {
    final db = _dbOrThrow;
    final count = await _store.count(db);
    return count;
  }

  Future<void> clearOfflineData() async {
    final db = _dbOrThrow;
    // Supprime les NATINF téléchargés
    await _store.delete(db);
    // Supprime également le cache des catégories associé
    await _categoriesStore.delete(db);
    print("Données locales supprimées (NATINF + catégories).");
  }

  Future<void> clearNatinfData() async {
    final db = _dbOrThrow;
    await _store.delete(db);
  }

  Future<bool> isEmpty() async {
    final db = _dbOrThrow;
    final count = await _store.count(db);
    return count == 0;
  }

  /// Recherche locale en essayant de reproduire la logique de l'API NATINFo
  Future<List<Natinf>> getNatinfByQuery(String query) async {
    final db = _dbOrThrow;
    String normalize(String s) =>
        removeDiacritics(
          s,
        ).toLowerCase().replaceAll(RegExp(r'[^\w]+'), ' ').trim();

    final normalizedQuery = normalize(query);
    final queryWords =
        normalizedQuery.split(' ').where((w) => w.isNotEmpty).toList();

    RegExp makePattern(String kw) {
      var p = RegExp.escape(kw);
      p = p.replaceAllMapped(
        RegExp(r'([A-Za-z])\\?\.?([0-9])'),
        (m) => '${m[1]}\\.?${m[2]}',
      );
      return RegExp(r'\b' + p, caseSensitive: false);
    }

    final List<RegExp> patterns = [];
    for (final word in queryWords) {
      patterns.add(makePattern(word));
    }

    final fullQueryRegex = RegExp(
      r'\b' + RegExp.escape(normalizedQuery) + r'\b',
      caseSensitive: false,
    );

    final records = await _store.find(db);
    final buffer = <_Scored>[];

    for (final record in records) {
      final map = Map<String, dynamic>.from(record.value as Map);
      final normalized = normalizeNatinfJson(map);

      final numero = normalize(normalized['numero_natinf']?.toString() ?? '');
      final nature = normalize(
        normalized['nature_infraction']?.toString() ?? '',
      );
      final qualification = normalize(
        normalized['qualification_infraction']?.toString() ?? '',
      );
      final definiePar = normalize(normalized['definie_par']?.toString() ?? '');
      final reprimeePar = normalize(
        normalized['reprimee_par']?.toString() ?? '',
      );

      final bool allMatch = patterns.every(
        (pat) =>
            pat.hasMatch(numero) ||
            pat.hasMatch(nature) ||
            pat.hasMatch(qualification) ||
            pat.hasMatch(definiePar) ||
            pat.hasMatch(reprimeePar),
      );

      if (!allMatch) continue;

      int relevance = 0;
      if (fullQueryRegex.hasMatch(numero)) {
        relevance = 100;
      } else if (fullQueryRegex.hasMatch(qualification)) {
        relevance = 90;
      } else if (fullQueryRegex.hasMatch(definiePar)) {
        relevance = 80;
      } else if (fullQueryRegex.hasMatch(reprimeePar)) {
        relevance = 70;
      }

      int qualificationPos = qualification.indexOf(normalizedQuery);
      if (qualificationPos == -1) qualificationPos = 100000;

      buffer.add(
        _Scored(
          natinf: parseStoredNatinf(normalized),
          relevance: relevance,
          qualificationPos: qualificationPos,
          pk: record.key,
        ),
      );
    }

    buffer.sort((a, b) {
      final rel = b.relevance.compareTo(a.relevance);
      if (rel != 0) return rel;

      final pos = a.qualificationPos.compareTo(b.qualificationPos);
      if (pos != 0) return pos;

      return a.pk.compareTo(b.pk);
    });

    return buffer.map((s) => s.natinf).toList();
  }

  /// Favoris

  Future<int> addFavourites(String numero) async {
    final db = _dbOrThrow;
    return await _favouritesStore.add(db, {
      'numero_natinf': numero,
      'added_at': DateTime.now().millisecondsSinceEpoch,
    });
  }

  Future<List<Map<String, dynamic>>> getFavourites() async {
    final db = _dbOrThrow;
    final finder = Finder(sortOrders: [SortOrder('added_at', false)]);
    final records = await _favouritesStore.find(db, finder: finder);
    return records
        .map((record) => record.value as Map<String, dynamic>)
        .toList();
  }

  Future<bool> isFavourite(String numero) async {
    final db = _dbOrThrow;
    final finder = Finder(filter: Filter.equals('numero_natinf', numero));
    final record = await _favouritesStore.findFirst(db, finder: finder);
    return record != null;
  }

  Future<void> removeFavourite(String numero) async {
    final db = _dbOrThrow;
    final finder = Finder(filter: Filter.equals('numero_natinf', numero));
    await _favouritesStore.delete(db, finder: finder);
    await removeNatinfFromAllFolders(numero);
  }

  Future<void> clearFavourites() async {
    final db = _dbOrThrow;
    await _favouritesStore.delete(db);
    await _favouriteFolderItemsStore.delete(db);
  }

  /// Crée un dossier de favoris et renvoie son identifiant.
  Future<int> createFavouriteFolder({
    required String name,
    int? parentId,
  }) async {
    final db = _dbOrThrow;
    return await _favouriteFoldersStore.add(db, {
      'name': name,
      'parent_id': parentId,
      'created_at': DateTime.now().millisecondsSinceEpoch,
    });
  }

  /// Renvoie l'ensemble des dossiers de favoris.
  Future<List<FavouriteFolder>> getAllFavouriteFolders() async {
    final db = _dbOrThrow;
    final records = await _favouriteFoldersStore.find(db);
    return records
        .map(
          (record) => FavouriteFolder.fromMap(
            id: record.key as int,
            map: Map<String, dynamic>.from(record.value as Map),
          ),
        )
        .toList();
  }

  /// Modifie le nom d'un dossier de favoris.
  Future<void> renameFavouriteFolder(int folderId, String newName) async {
    final db = _dbOrThrow;
    await _favouriteFoldersStore.record(folderId).update(db, {'name': newName});
  }

  /// Supprime un dossier ainsi que son sous-arborescence.
  Future<void> deleteFavouriteFolder(int folderId) async {
    final db = _dbOrThrow;
    final finder = Finder(filter: Filter.equals('parent_id', folderId));
    final children = await _favouriteFoldersStore.find(db, finder: finder);
    for (final child in children) {
      await deleteFavouriteFolder(child.key as int);
    }
    await _removeFolderContents(folderId);
    await _favouriteFoldersStore.record(folderId).delete(db);
  }

  /// Ajoute un NATINF dans un dossier donné.
  Future<void> addNatinfToFolder(int folderId, String numero) async {
    final db = _dbOrThrow;
    final key = '$folderId|$numero';
    await _favouriteFolderItemsStore.record(key).put(db, {
      'folder_id': folderId,
      'numero_natinf': numero,
      'added_at': DateTime.now().millisecondsSinceEpoch,
    });
  }

  /// Retire un NATINF d'un dossier.
  Future<void> removeNatinfFromFolder(int folderId, String numero) async {
    final db = _dbOrThrow;
    final key = '$folderId|$numero';
    await _favouriteFolderItemsStore.record(key).delete(db);
  }

  /// Retire un NATINF de tous les dossiers.
  Future<void> removeNatinfFromAllFolders(String numero) async {
    final db = _dbOrThrow;
    final finder = Finder(filter: Filter.equals('numero_natinf', numero));
    await _favouriteFolderItemsStore.delete(db, finder: finder);
  }

  /// Renvoie la liste des identifiants de dossiers contenant [numero].
  Future<Set<int>> getFolderIdsForNatinf(String numero) async {
    final db = _dbOrThrow;
    final finder = Finder(filter: Filter.equals('numero_natinf', numero));
    final records = await _favouriteFolderItemsStore.find(db, finder: finder);
    return records
        .map(
          (record) =>
              (record.value as Map<String, dynamic>)['folder_id'] as int,
        )
        .toSet();
  }

  /// Renvoie les NATINF présents dans chaque dossier.
  Future<Map<int, List<String>>> getFolderNatinfMap() async {
    final db = _dbOrThrow;
    final records = await _favouriteFolderItemsStore.find(db);
    final result = <int, List<String>>{};
    for (final record in records) {
      final map = record.value as Map<String, dynamic>;
      final folderId = map['folder_id'] as int;
      final numero = map['numero_natinf'] as String?;
      if (numero == null) continue;
      final list = result.putIfAbsent(folderId, () => <String>[]);
      if (!list.contains(numero)) {
        list.add(numero);
      }
    }
    for (final entry in result.entries) {
      entry.value.sort();
    }
    return result;
  }

  Future<void> _removeFolderContents(int folderId) async {
    final db = _dbOrThrow;
    final finder = Finder(filter: Filter.equals('folder_id', folderId));
    await _favouriteFolderItemsStore.delete(db, finder: finder);
  }

  /// Historique

  Future<int> addHistory(String numero) async {
    final db = _dbOrThrow;
    return await _historyStore.add(db, {
      'numero_natinf': numero,
      'opened_at': DateTime.now().millisecondsSinceEpoch,
    });
  }

  Future<List<Map<String, dynamic>>> getHistory() async {
    final db = _dbOrThrow;
    final finder = Finder(sortOrders: [SortOrder('opened_at', false)]);
    final records = await _historyStore.find(db, finder: finder);
    return records
        .map((record) => record.value as Map<String, dynamic>)
        .toList();
  }

  Future<void> clearHistory() async {
    final db = _dbOrThrow;
    await _historyStore.delete(db);
  }
}
