package net.osmand.plus.plugins.externalsensors;

import static net.osmand.plus.plugins.externalsensors.devices.sensors.DeviceChangeableProperty.NAME;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.AlertDialog;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothManager;
import android.bluetooth.BluetoothProfile;
import android.bluetooth.le.BluetoothLeScanner;
import android.bluetooth.le.ScanCallback;
import android.bluetooth.le.ScanFilter;
import android.bluetooth.le.ScanRecord;
import android.bluetooth.le.ScanResult;
import android.bluetooth.le.ScanSettings;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.ParcelUuid;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import net.osmand.PlatformUtil;
import net.osmand.plus.OsmandApplication;
import net.osmand.plus.R;
import net.osmand.plus.Version;
import net.osmand.plus.activities.ActivityResultListener;
import net.osmand.plus.activities.MapActivity;
import net.osmand.plus.plugins.externalsensors.DevicesSettingsCollection.DevicePreferencesListener;
import net.osmand.plus.plugins.externalsensors.DevicesSettingsCollection.DeviceSettings;
import net.osmand.plus.plugins.externalsensors.devices.AbstractDevice;
import net.osmand.plus.plugins.externalsensors.devices.AbstractDevice.DeviceListener;
import net.osmand.plus.plugins.externalsensors.devices.DeviceConnectionResult;
import net.osmand.plus.plugins.externalsensors.devices.ble.BLEAbstractDevice;
import net.osmand.plus.plugins.externalsensors.devices.ble.BLEBPICPDevice;
import net.osmand.plus.plugins.externalsensors.devices.ble.BLEBikeSCDDevice;
import net.osmand.plus.plugins.externalsensors.devices.ble.BLEHeartRateDevice;
import net.osmand.plus.plugins.externalsensors.devices.ble.BLEOBDDevice;
import net.osmand.plus.plugins.externalsensors.devices.ble.BLERunningSCDDevice;
import net.osmand.plus.plugins.externalsensors.devices.ble.BLETemperatureDevice;
import net.osmand.plus.plugins.externalsensors.devices.sensors.AbstractSensor;
import net.osmand.plus.plugins.externalsensors.devices.sensors.DeviceChangeableProperty;
import net.osmand.plus.plugins.externalsensors.devices.sensors.SensorData;
import net.osmand.plus.plugins.externalsensors.devices.sensors.SensorDataField;
import net.osmand.plus.settings.backend.preferences.CommonPreferenceProvider;
import net.osmand.plus.utils.AndroidUtils;
import net.osmand.plus.utils.BLEUtils;
import net.osmand.plus.utils.FormattedValue;
import net.osmand.util.Algorithms;

import org.apache.commons.logging.Log;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;


public abstract class DevicesHelper implements DeviceListener, DevicePreferencesListener {

	public static final int ENABLE_BLUETOOTH_REQUEST_CODE = 400;

	private static final Log LOG = PlatformUtil.getLog(DevicesHelper.class);

	private final static List<UUID> SUPPORTED_BLE_SERVICE_UUIDS = Arrays.asList(
			BLEBikeSCDDevice.getServiceUUID(),
			BLEHeartRateDevice.getServiceUUID(),
			BLERunningSCDDevice.getServiceUUID(),
			BLETemperatureDevice.getServiceUUID());

	private OsmandApplication app;
	protected DevicesSettingsCollection devicesSettingsCollection;
	protected final Map<String, AbstractDevice<?>> devices = new ConcurrentHashMap<>();

	private boolean antScanning;
	private boolean bleScanning;

	private Activity activity;
	private boolean installAntPluginAsked;
	protected BluetoothAdapter bluetoothAdapter;
	private BluetoothLeScanner bleScanner;

	protected DevicesHelper(@NonNull OsmandApplication app, @NonNull CommonPreferenceProvider<String> preferenceProvider) {
		init(app, preferenceProvider);
	}

	private void init(@NonNull OsmandApplication app, @NonNull CommonPreferenceProvider<String> preferenceProvider) {
		this.app = app;
		this.devicesSettingsCollection = new DevicesSettingsCollection(preferenceProvider);
	}

	public void setActivity(@Nullable Activity activity) {
		if (this.activity != null) {
			dropUnpairedDevices();
			deinitBLE();
			devicesSettingsCollection.removeListener(this);
		}
		this.activity = activity;
		if (activity != null) {
			initBLE();
			initDevices();
			devicesSettingsCollection.addListener(this);
		}
	}

	/*
	 */
	void initBLE() {
		BluetoothManager bluetoothManager = (BluetoothManager) activity.getSystemService(Context.BLUETOOTH_SERVICE);
		bluetoothAdapter = bluetoothManager.getAdapter();
		if (bluetoothAdapter == null) {
			app.showShortToastMessage(R.string.bluetooth_not_supported);
		} else {
			bleScanner = bluetoothAdapter.getBluetoothLeScanner();
		}
	}

	public void deinitBLE() {
		try {
			if (bluetoothAdapter != null) {
				if (bleScanner != null && bluetoothAdapter.isEnabled()) {
					try {
						bleScanner.stopScan(bleScanCallback);
					} catch (Throwable error) {
						LOG.error("Stop ble scan error. " + error);
					}
				}
				bluetoothAdapter.cancelDiscovery();
				bluetoothAdapter = null;
			}
			bleScanner = null;
		} catch (SecurityException error) {
			LOG.debug("No permission on disable BLE");
		}
	}

	private void initDevices() {
		for (String deviceId : devicesSettingsCollection.getDeviceIds()) {
			DeviceSettings deviceSettings = devicesSettingsCollection.getDeviceSettings(deviceId);
			if (deviceSettings != null && !devices.containsKey(deviceId)) {
				if (deviceSettings.getDeviceType() == null) {
					devicesSettingsCollection.removeDeviceSettings(deviceId);
				} else {
					AbstractDevice<?> device = createDevice(deviceSettings.getDeviceType(), deviceId, deviceSettings);
					if (device != null) {
						devices.put(deviceId, device);
						updateDeviceProperties(device);
					}
				}
			}
		}
		updateDevices(activity);
	}

	private void updateDeviceProperties(@NonNull AbstractDevice<?> device) {
		String deviceId = device.getDeviceId();
		DevicesSettingsCollection.DeviceSettings deviceSettings = devicesSettingsCollection.getDeviceSettings(deviceId);
		if (deviceSettings != null) {
			for (DeviceChangeableProperty property : deviceSettings.getParams().keySet()) {
				device.setChangeableProperty(property, deviceSettings.getParams().get(property));
			}
		}
	}

	@Nullable
	protected AbstractDevice<?> createDevice(@NonNull DeviceType deviceType, @NonNull String deviceId, @NonNull DeviceSettings deviceSettings) {
		return switch (deviceType) {
			case BLE_OBD ->
					bluetoothAdapter != null ? new BLEOBDDevice(bluetoothAdapter, deviceId) : null;
			case BLE_TEMPERATURE ->
					bluetoothAdapter != null ? new BLETemperatureDevice(bluetoothAdapter, deviceId) : null;
			case BLE_HEART_RATE ->
					bluetoothAdapter != null ? new BLEHeartRateDevice(bluetoothAdapter, deviceId) : null;
			case BLE_BLOOD_PRESSURE ->
					bluetoothAdapter != null ? new BLEBPICPDevice(bluetoothAdapter, deviceId) : null;
			case BLE_BICYCLE_SCD ->
					bluetoothAdapter != null ? new BLEBikeSCDDevice(bluetoothAdapter, deviceId) : null;
			case BLE_RUNNING_SCDS ->
					bluetoothAdapter != null ? new BLERunningSCDDevice(bluetoothAdapter, deviceId) : null;
			default -> null;
		};
	}

	public boolean isAntScanning() {
		return antScanning;
	}

	public boolean isBleScanning() {
		return bleScanning;
	}

	private final ScanCallback bleScanCallback = new ScanCallback() {
		@SuppressLint("MissingPermission")
		@Override
		public void onScanResult(int callbackType, ScanResult result) {
			super.onScanResult(callbackType, result);
			BluetoothDevice device = result.getDevice();
			ScanRecord record = result.getScanRecord();
			if (AndroidUtils.hasBLEPermission(activity)) {
				LOG.debug("BLE scan result " + device.getAddress() + "; name " + device.getName() + "; " + record.getServiceUuids());
				if (device.getName() != null) {
					addScanResult(result);
				}
			} else {
				LOG.error("Try to add ble device while no permission");
			}
		}

		@Override
		public void onBatchScanResults(List<ScanResult> results) {
			super.onBatchScanResults(results);
			for (ScanResult result : results) {
				addScanResult(result);
			}
		}

		@SuppressLint("MissingPermission")
		private void addScanResult(ScanResult result) {
			if (!AndroidUtils.hasBLEPermission(activity)) {
				LOG.error("Try to addScanResult while no permission");
				return;
			}
			ScanRecord scanRecord = result.getScanRecord();
			if (scanRecord != null) {
				String address = result.getDevice().getAddress();
				DevicesSettingsCollection.DeviceSettings settings = devicesSettingsCollection.getDeviceSettings(address);
				String deviceName;
				deviceName = settings == null ? result.getDevice().getName() : settings.getParams().get(NAME);
				List<ParcelUuid> uuids = scanRecord.getServiceUuids();
				if (uuids != null) {
					for (ParcelUuid uuid : uuids) {
						BLEAbstractDevice device = createBLEDevice(result, uuid, address, deviceName);
						if (device != null) {
							if (!devices.containsKey(device.getDeviceId())) {
								addFoundBLEDevice(device);
							}
							break;
						}
					}
				}
			}
		}

		@Override
		public void onScanFailed(int errorCode) {
			super.onScanFailed(errorCode);
			if (errorCode == SCAN_FAILED_ALREADY_STARTED) {
				scanBLEDevices(false);
				scanBLEDevices(true);
			}
			LOG.error("BLE scan failed. Error " + errorCode);
		}
	};

	@Nullable
	protected BLEAbstractDevice createBLEDevice(ScanResult result, ParcelUuid uuid, String address, String deviceName) {
		return BLEAbstractDevice.createDeviceByUUID(
				bluetoothAdapter, uuid.getUuid(), address, deviceName, result.getRssi());
	}

	protected abstract void addFoundBLEDevice(@NonNull BLEAbstractDevice device);

	private boolean isSupportedBleDevice(@NonNull ScanRecord scanRecord) {
		List<ParcelUuid> uuids = scanRecord.getServiceUuids();
		if (uuids != null) {
			for (ParcelUuid uuid : uuids) {
				if (SUPPORTED_BLE_SERVICE_UUIDS.contains(uuid.getUuid())) {
					return true;
				}
			}
		}
		return false;
	}

	public void connectDevice(@Nullable Activity activity, @NonNull AbstractDevice<?> device) {
		device.addListener(this);
		device.connect(app, activity);
	}

	public void disconnectDevice(@NonNull AbstractDevice<?> device) {
		disconnectDevice(device, true);
	}

	void disconnectDevice(@NonNull AbstractDevice<?> device, boolean notify) {
		device.removeListener(this);
		if (device.disconnect() && notify) {
			onDeviceDisconnect(device);
		}
	}

	void connectDevices(@Nullable Activity activity) {
		for (AbstractDevice<?> device : getAllDevices()) {
			if (isDeviceEnabled(device)) {
				connectDevice(activity, device);
			}
		}
	}

	public void disconnectDevices() {
		for (AbstractDevice<?> device : getAllDevices()) {
			disconnectDevice(device);
		}
	}

	void updateDevice(@Nullable Activity activity, @NonNull AbstractDevice<?> device) {
		if (isDeviceEnabled(device) && device.isDisconnected()) {
			connectDevice(activity, device);
		} else if (!isDeviceEnabled(device) && device.isConnected()) {
			disconnectDevice(device);
		}
	}

	void updateDevices(@Nullable Activity activity) {
		for (AbstractDevice<?> device : getAllDevices()) {
			updateDevice(activity, device);
		}
	}

	void dropUnpairedDevice(@NonNull AbstractDevice<?> device) {
		disconnectDevice(device);
		devices.remove(device.getDeviceId());
	}

	public void dropUnpairedDevices() {
		for (AbstractDevice<?> device : getUnpairedDevices()) {
			dropUnpairedDevice(device);
		}
	}

	boolean isAntDevice(@NonNull AbstractDevice<?> device) {
		return false;
	}

	boolean isBLEDevice(@NonNull AbstractDevice<?> device) {
		return device instanceof BLEAbstractDevice;
	}

	@NonNull
	public List<AbstractDevice<?>> getAllDevices() {
		return new ArrayList<>(devices.values());
	}

	@NonNull
	public List<AbstractDevice<?>> getPairedDevices() {
		List<AbstractDevice<?>> res = new ArrayList<>();
		for (AbstractDevice<?> device : getAllDevices()) {
			if (isDevicePaired(device)) {
				res.add(device);
			}
		}
		return res;
	}

	@Nullable
	public AbstractDevice<?> getPairedDeviceById(@NonNull String deviceId) {
		for (AbstractDevice<?> device : getAllDevices()) {
			if (isDevicePaired(device) && deviceId.equals(device.getDeviceId())) {
				return device;
			}
		}
		return null;
	}

	@NonNull
	public List<AbstractDevice<?>> getUnpairedDevices() {
		List<AbstractDevice<?>> res = new ArrayList<>();
		for (AbstractDevice<?> device : getAllDevices()) {
			if (!isDevicePaired(device)) {
				res.add(device);
			}
		}
		return res;
	}

	@Nullable
	public AbstractDevice<?> getAnyDevice(@NonNull String deviceId) {
		for (AbstractDevice<?> device : getAllDevices()) {
			if (Algorithms.stringsEqual(device.getDeviceId(), deviceId)) {
				return device;
			}
		}
		return null;
	}

	void askAntPluginInstall() {
		installAntPluginAsked = true;
	}

	@Override
	public void onDeviceConnecting(@NonNull AbstractDevice<?> device) {
	}

	@Override
	public void onDeviceConnect(@NonNull AbstractDevice<?> device, @NonNull DeviceConnectionResult result, @Nullable String error) {
		if (!Algorithms.isEmpty(error)) {
			LOG.error(device + " sensor connection error: " + error);
		}
		switch (result) {
			case SUCCESS:
				/*
				ScheduledFuture<?> reconnectingFuture = reconnectingDevices.remove(device.getDeviceId());
				if (reconnectingFuture != null) {
					reconnectingFuture.cancel(true);
				}
				*/
				LOG.debug(device + " sensor connected");
				if (antScanning && isAntDevice(device)) {
					devices.put(device.getDeviceId(), device);
				} else if (bleScanning && isBLEDevice(device)) {
					// skip
				} else {
					app.runInUIThread(() -> onDeviceConnectSucceed(device));
				}
				break;
			case DEPENDENCY_NOT_INSTALLED:
				if (isAntDevice(device)) {
					LOG.debug("ANT+ External Sensors plugin is not installed. Ask plugin install");
					askAntPluginInstall();
				}
				break;
			case SEARCH_TIMEOUT:
				if (!antScanning && isAntDevice(device) || !bleScanning && isBLEDevice(device)) {
					if (!isDeviceEnabled(device)) {
						updateDevice(activity, device);
					} else {
						LOG.debug("Reconnect " + device + " after timeout");
						connectDevice(activity, device);
					}
				}
				break;
			default:
				break;
		}
	}

	protected void onDeviceConnectSucceed(@NonNull AbstractDevice<?> device) {
		if (!isDeviceEnabled(device)) {
			updateDevice(activity, device);
		} else {
			app.showShortToastMessage(R.string.device_connected, getFormattedDevicePropertyValue(device, NAME));
		}
	}

	@Override
	public void onDeviceDisconnect(@NonNull AbstractDevice<?> device) {
		LOG.debug(device + " disconnected");
		app.showShortToastMessage(R.string.device_disconnected, getFormattedDevicePropertyValue(device, NAME));
		/*
		if (!device.isDisconnected() && !reconnectingDevices.containsKey(device.getDeviceId())) {
			tryToReconnectToDevice(device);
		}
		*/
	}

	/*
		private void tryToReconnectToDevice(@NonNull AbstractDevice<?> device) {
			ScheduledExecutorService scheduler = reconnectToDeviceScheduler;
			if (!device.isDisconnected() && !reconnectingDevices.containsKey(device.getDeviceId()) && scheduler != null) {
				ScheduledFuture<?> future = scheduler.schedule(() -> {
					if(activity != null && device.connect(app, activity)) {
						ScheduledFuture<?> checkFuture = scheduler.schedule(() -> {
							checkReconnectDeviceResult(device.getDeviceId());
						}, RECONNECT_DEVICE_TIMEOUT, TimeUnit.SECONDS);
						reconnectingDevices.put(device.getDeviceId(), checkFuture);
					}
				}, RECONNECT_DEVICE_DELAY, TimeUnit.SECONDS);
				reconnectingDevices.put(device.getDeviceId(), future);
			}
		}

		private void checkReconnectDeviceResult(@NonNull String deviceId) {
			Activity activity = this.activity;
			if (reconnectingDevices.remove(deviceId) != null && activity != null) {
				AbstractDevice<?> device = getAnyDevice(deviceId);
				if (device != null) {
					app.showShortToastMessage(R.string.failed_to_connect, device.getName());
				}
			}
		}
	*/
	@Override
	public void onSensorData(@NonNull AbstractSensor sensor, @NonNull SensorData data) {
		for (SensorDataField dataField : data.getDataFields()) {
			FormattedValue fmtValue = dataField.getFormattedValue(app);
			if (fmtValue != null) {
				LOG.debug("onSensorData '" + sensor.getDevice().getName() + "' <" + sensor.getName() + ">: "
						+ fmtValue.value + (!Algorithms.isEmpty(fmtValue.unit) ? " " + fmtValue.unit : ""));
			}
		}
	}

	public boolean isDevicePaired(@NonNull AbstractDevice<?> device) {
		DevicesSettingsCollection.DeviceSettings settings = devicesSettingsCollection.getDeviceSettings(device.getDeviceId());
		return settings != null;
	}

	public void setDevicePaired(@NonNull AbstractDevice<?> device, boolean paired) {
		String deviceId = device.getDeviceId();
		DeviceSettings settings = devicesSettingsCollection.getDeviceSettings(deviceId);
		if (!paired) {
			devicesSettingsCollection.setDeviceSettings(deviceId, null);
			dropUnpairedDevice(device);
		} else {
			if (settings == null) {
				if (!Algorithms.isEmpty(deviceId)) {
					settings = DevicesSettingsCollection.createDeviceSettings(deviceId, device, true);
					devicesSettingsCollection.setDeviceSettings(deviceId, settings);
					updateDeviceProperties(device);
					onDevicePaired(device);
				}
			}
			//connectDevice(activity, device);
		}
		app.runInUIThread(() -> {
			MapActivity mapActivity = getMapActivity();
			if (mapActivity != null) {
				mapActivity.updateApplicationModeSettings();
			}
		});
	}

	public void onDevicePaired(@NonNull AbstractDevice<?> device) {

	}

	public boolean isDeviceEnabled(@NonNull AbstractDevice<?> device) {
		DeviceSettings settings = devicesSettingsCollection.getDeviceSettings(device.getDeviceId());
		return settings != null && settings.getDeviceEnabled();
	}

	public void setDeviceEnabled(@NonNull AbstractDevice<?> device, boolean enabled) {
		String deviceId = device.getDeviceId();
		DeviceSettings settings = devicesSettingsCollection.getDeviceSettings(deviceId);
		if (settings == null) {
			if (!Algorithms.isEmpty(deviceId)) {
				settings = DevicesSettingsCollection.createDeviceSettings(deviceId, device, enabled);
				devicesSettingsCollection.setDeviceSettings(deviceId, settings);
			}
		} else {
			settings.setDeviceEnabled(enabled);
			devicesSettingsCollection.setDeviceSettings(deviceId, settings);
		}
	}

	@NonNull
	public String getFormattedDevicePropertyValue(@NonNull AbstractDevice<?> device, @NonNull DeviceChangeableProperty property) {
		DeviceSettings settings = devicesSettingsCollection.getDeviceSettings(device.getDeviceId());
		String value = settings != null ? settings.getParams().get(property) : null;
		return property.getFormattedValue(app, value);
	}

	public void setDeviceProperty(@NonNull AbstractDevice<?> device, @NonNull DeviceChangeableProperty property, @NonNull String value) {
		String normalizedValue = property.normalizeValue(app, value);
		String deviceId = device.getDeviceId();
		DeviceSettings settings = devicesSettingsCollection.getDeviceSettings(deviceId);
		if (settings == null) {
			if (!Algorithms.isEmpty(deviceId)) {
				settings = DevicesSettingsCollection.createDeviceSettings(deviceId, device, false);
			}
		}
		if (settings != null) {
			settings.setDeviceProperty(property, normalizedValue);
			device.setChangeableProperty(property, value);
			devicesSettingsCollection.setDeviceSettings(deviceId, settings);
		}
	}

	@Override
	public void onDeviceEnabled(@NonNull String deviceId) {
		app.runInUIThread(() -> updateDevices(activity));
	}

	@Override
	public void onDeviceDisabled(@NonNull String deviceId) {
		app.runInUIThread(() -> updateDevices(activity));
	}

	public void scanAntDevices(boolean enable) {
	}

	@SuppressLint("MissingPermission")
	public void scanBLEDevices(boolean enable) {
		if (!enable) {
			if (bleScanner != null) {
				if (AndroidUtils.hasBLEPermission(activity)
						&& bleScanning
						&& AndroidUtils.isBluetoothEnabled(activity)) {
					bleScanner.stopScan(bleScanCallback);
				}
				bleScanning = false;
			}
		} else {
			if (!AndroidUtils.requestBLEPermissions(activity)) {
				app.showShortToastMessage("Permissions not granted");
				return;
			}
			if (!requestBLE()) {
				app.showShortToastMessage("Bluetooth isnt available");
				return;
			}

			if (bleScanner == null) {
				initBLE();
			}

			if (bleScanner == null) {
				app.showShortToastMessage("Can't initialize ble");
				return;
			}

			ArrayList<ScanFilter> filters = new ArrayList<>();
			for (UUID serviceUUID : SUPPORTED_BLE_SERVICE_UUIDS) {
				ScanFilter filter = new ScanFilter.Builder()
						.setServiceUuid(new ParcelUuid(serviceUUID))
						.build();
				filters.add(filter);
			}
			ScanSettings scanSettings = new ScanSettings.Builder()
					.setScanMode(ScanSettings.SCAN_MODE_LOW_POWER)
					.setCallbackType(ScanSettings.CALLBACK_TYPE_ALL_MATCHES)
					.setMatchMode(ScanSettings.MATCH_MODE_AGGRESSIVE)
					.setNumOfMatches(ScanSettings.MATCH_NUM_ONE_ADVERTISEMENT)
					.setReportDelay(0L)
					.build();

			bleScanner.startScan(null, scanSettings, bleScanCallback);
			bleScanning = true;
		}
	}

	public boolean isBLEEnabled() {
		return activity != null && BLEUtils.INSTANCE.isBLEEnabled(activity);
	}

	public boolean requestBLE() {
		boolean bluetoothEnabled = true;
		if (activity.getPackageManager().hasSystemFeature(PackageManager.FEATURE_BLUETOOTH_LE)) {
			if (bluetoothAdapter == null || !bluetoothAdapter.isEnabled()) {
				bluetoothEnabled = false;
				MapActivity mapActivity = getMapActivity();
				if (mapActivity != null) {
					try {
						Intent intent = new Intent(BluetoothAdapter.ACTION_REQUEST_ENABLE);
						mapActivity.startActivityForResult(intent, ENABLE_BLUETOOTH_REQUEST_CODE);
						mapActivity.registerActivityResultListener(new ActivityResultListener(ENABLE_BLUETOOTH_REQUEST_CODE, (resultCode, resultData) -> {
							if (resultCode != Activity.RESULT_OK) {
								app.showShortToastMessage(R.string.no_bt_permission);
							}
						}));
					} catch (ActivityNotFoundException e) {
						app.showToastMessage(R.string.no_activity_for_intent);
					}
				}
			}
		} else {
			bluetoothEnabled = false;
//			app.showToastMessage("Bluetooth LE isnt supported on this device");
		}
		return bluetoothEnabled;
	}

	private MapActivity getMapActivity() {
		if (activity instanceof MapActivity) {
			return (MapActivity) activity;
		} else {
			return null;
		}
	}

	@SuppressLint("MissingPermission")
	public boolean isBLEDeviceConnected(@NonNull String address) {
		if (isBLEEnabled() && AndroidUtils.hasBLEPermission(activity)) {
			BluetoothManager bluetoothManager = (BluetoothManager) activity.getSystemService(Context.BLUETOOTH_SERVICE);
			BluetoothDevice device = bluetoothAdapter.getRemoteDevice(address);
			return bluetoothManager.getConnectionState(device, BluetoothProfile.GATT) == BluetoothProfile.STATE_CONNECTED;
		}
		return false;
	}
}