package net.osmand.plus.track.helpers;

import static net.osmand.plus.track.GpxSplitType.DISTANCE;
import static net.osmand.plus.track.GpxSplitType.NO_SPLIT;
import static net.osmand.plus.track.GpxSplitType.TIME;
import static net.osmand.plus.track.GpxSplitType.UPHILL_DOWNHILL;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import net.osmand.shared.gpx.GpxFile;
import net.osmand.shared.gpx.primitives.Track;
import net.osmand.plus.R;
import net.osmand.plus.track.helpers.GpxDisplayHelper.GpxSplitParams;
import net.osmand.plus.track.helpers.GpxSelectionHelper.GpxDisplayItemType;

public class TrackDisplayGroup extends GpxDisplayGroup {

	private final Track track;
	private final boolean isGeneralTrack;

	private double splitDistance = -1;
	private int splitTime = -1;
	private boolean uphillDownhill = false;

	public TrackDisplayGroup(@NonNull GpxFile gpxFile, @NonNull Track track, boolean isGeneralTrack) {
		this(gpxFile, track, isGeneralTrack, -1);
	}

	public TrackDisplayGroup(@NonNull GpxFile gpxFile, @NonNull Track track, boolean isGeneralTrack, int trackIndex) {
		super(gpxFile, trackIndex);
		this.track = track;
		this.isGeneralTrack = isGeneralTrack;
	}

	@Nullable
	public Track getTrack() {
		return track;
	}

	public boolean isGeneralTrack() {
		return isGeneralTrack;
	}

	@Override
	public void applyName(@NonNull Context context, @NonNull String name) {
		setGpxName(name);
		int trackIndex = getIndex();
		GpxFile gpxFile = getGpxFile();
		String trackIndexStr = trackIndex == -1 || gpxFile.getTracks().size() == 1 ? "" : String.valueOf(trackIndex + 1);
		setName(context.getString(R.string.gpx_selection_track, name, trackIndexStr));
	}

	public boolean isSplitDistance() {
		return splitDistance > 0;
	}

	public double getSplitDistance() {
		return splitDistance;
	}

	public boolean isSplitTime() {
		return splitTime > 0;
	}

	public boolean isSplitUphillDownhill() {
		return uphillDownhill;
	}

	public int getSplitTime() {
		return splitTime;
	}

	public void updateSplit(@NonNull GpxSplitParams splitParams) {
		clearDisplayItems();
		if (splitParams.splitType() == NO_SPLIT) {
			splitDistance = -1;
			splitTime = -1;
			uphillDownhill = false;
		} else if (splitParams.splitType() == DISTANCE) {
			splitDistance = splitParams.splitInterval();
			splitTime = -1;
			uphillDownhill = false;
		} else if (splitParams.splitType() == TIME) {
			splitDistance = -1;
			splitTime = (int) splitParams.splitInterval();
			uphillDownhill = false;
		} else if (splitParams.splitType() == UPHILL_DOWNHILL) {
			splitDistance = -1;
			splitTime = (int) splitParams.splitInterval();
			uphillDownhill = true;
		}
	}

	@Override
	@NonNull
	public GpxDisplayItemType getType() {
		return GpxDisplayItemType.TRACK_SEGMENT;
	}

	@Override
	@NonNull
	protected GpxDisplayGroup newInstance(@NonNull GpxFile gpxFile) {
		return new TrackDisplayGroup(gpxFile, track, isGeneralTrack, getIndex());
	}

}
