#!/usr/bin/env python3
# -*- coding: utf-8 -*-

from sys import argv, exit
from typing import AnyStr
from os import linesep, path
import zlib


def embed_resources_chunk(list_filepath: AnyStr, source_filepath: AnyStr, resource_index: str) -> bool:
    list_filepath = path.abspath(list_filepath)
    print("List file path:   %s" %(
        list_filepath,
    ))

    source_filepath = path.abspath(source_filepath)
    print("Source file path: %s" %(
        source_filepath,
    ))

    resource_index = int(resource_index)
    print("Resource index:   %s" %(
        resource_index,
    ))

    with open(list_filepath, "r") as list_file, open(source_filepath, "w") as source_file:
        source_file.write("// AUTOGENERATED FILE\n")
        source_file.write("#include \"resources-bundle.h\"\n")
        source_file.write("\n")
        source_file.write("extern \"C\" {\n")
        source_file.write("\n")

        for list_entry in list_file:
            list_entry = list_entry.strip()
            if not list_entry:
                continue

            resource_filepath, bundle_entrypath = list_entry.split(":")

            resource_size = path.getsize(resource_filepath)
            with open(resource_filepath, "rb") as resourceFile:
                resource_data = resourceFile.read()

            if not resource_filepath.endswith(".png"):
                resource_data = zlib.compress(resource_data, 9)
            resource_data_size = len(resource_data)

            source_file.write("static const char* const __CoreResourcesEmbeddedBundle__ResourceName_%d = \"%s\";\n" % (
                resource_index,
                bundle_entrypath,
            ))
            source_file.write("EMIT_GETTER(__CoreResourcesEmbeddedBundle__ResourceName_%d, )\n" % (
                resource_index,
            ))
            source_file.write("static const uint8_t __CoreResourcesEmbeddedBundle__ResourceData_%d[] = {\n" % (
                resource_index,
            ))

            # Write size header
            source_file.write("\t0x%02x, 0x%02x, 0x%02x, 0x%02x," % (
                (resource_size >> 24)&0xff,
                (resource_size >> 16)&0xff,
                (resource_size >>  8)&0xff,
                (resource_size >>  0)&0xff,
            ))

            # Write content
            for (byteIdx, byteValue) in enumerate(resource_data):
                if byteIdx % 16 == 0:
                    source_file.write("\n\t")
                source_file.write("0x%02x, " % (byteValue))
            source_file.write("\n")
            source_file.write("};\n")
            source_file.write("EMIT_GETTER(__CoreResourcesEmbeddedBundle__ResourceData_%d, )\n" % (
                resource_index,
            ))

            source_file.write("const size_t __CoreResourcesEmbeddedBundle__ResourceSize_%d = 4 + %d;\n" % (
                resource_index,
                resource_data_size,
            ))
            source_file.write("EMIT_GETTER(__CoreResourcesEmbeddedBundle__ResourceSize_%d, &)\n" % (
                resource_index,
            ))
            source_file.write("\n")

            print("Packed '%s'(%d bytes) as '%s'(4+%d bytes)..." % (
                resource_filepath,
                resource_size,
                bundle_entrypath,
                resource_data_size,
            ))

            resource_index += 1

        # Write footer of the file and close it
        source_file.write("} // extern \"C\"\n")
        source_file.write("\n")
        source_file.flush()
        source_file.close()

    return True


if __name__ == "__main__":
    exit(0 if embed_resources_chunk(argv[1], argv[2], argv[3]) else -1)
