package net.osmand.plus.routing;

import static net.osmand.router.RouteExporter.OSMAND_ROUTER_V2;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import net.osmand.plus.settings.enums.ReverseTrackStrategy;
import net.osmand.shared.gpx.GpxFile;
import net.osmand.shared.gpx.primitives.WptPt;
import net.osmand.Location;
import net.osmand.PlatformUtil;
import net.osmand.data.LatLon;
import net.osmand.data.LocationPoint;
import net.osmand.data.WptLocationPoint;
import net.osmand.plus.OsmandApplication;
import net.osmand.plus.R;
import net.osmand.plus.measurementtool.GpxApproximationParams;
import net.osmand.plus.settings.backend.OsmandSettings;
import net.osmand.router.RouteSegmentResult;
import net.osmand.util.Algorithms;
import net.osmand.util.MapUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public class GPXRouteParams {

	public static final String OSMAND_ROUTER = "OsmAndRouter";

	protected GpxFile gpxFile;
	protected List<LocationPoint> wpt;
	protected List<RouteSegmentResult> route;
	protected List<RouteDirectionInfo> directions;
	protected List<Location> points = new ArrayList<>();
	protected List<Location> segmentEndpoints = new ArrayList<>();
	protected List<WptPt> routePoints = new ArrayList<>();
	protected boolean reverse;
	protected ReverseTrackStrategy reverseStrategy;
	protected boolean passWholeRoute;
	protected boolean calculateOsmAndRoute;
	protected boolean connectPointsStraightly;
	protected boolean useIntermediatePointsRTE;
	protected boolean calculateOsmAndRouteParts;
	protected boolean calculatedRouteTimeSpeed;
	protected boolean leftSide;
	protected int selectedRoute;
	protected int selectedSegment;
	protected GpxApproximationParams approximationParams;

	boolean addMissingTurns = true;

	public List<Location> getPoints() {
		return points;
	}

	public List<RouteSegmentResult> getRoute() {
		return route;
	}

	public boolean hasOsmAndRoute() {
		return !Algorithms.isEmpty(route);
	}

	public List<LocationPoint> getWpt() {
		return wpt;
	}

	public LatLon getLastPoint() {
		if (!points.isEmpty()) {
			Location l = points.get(points.size() - 1);
			return new LatLon(l.getLatitude(), l.getLongitude());
		}
		return null;
	}

	public void prepareGPXFile(@NonNull GPXRouteParamsBuilder builder) {
		gpxFile = builder.file;
		reverse = builder.reverse;
		reverseStrategy = builder.reverseStrategy;
		passWholeRoute = builder.passWholeRoute;
		calculateOsmAndRouteParts = builder.calculateOsmAndRouteParts;
		calculatedRouteTimeSpeed = builder.calculatedRouteTimeSpeed;
		connectPointsStraightly = builder.connectPointsStraightly;
		useIntermediatePointsRTE = builder.useIntermediateRtePoints;
		builder.calculateOsmAndRoute = false; // Disabled temporary builder.calculateOsmAndRoute;

		leftSide = builder.leftSide;
		selectedRoute = builder.getSelectedRoute();
		selectedSegment = builder.getSelectedSegment();
		approximationParams = builder.getApproximationParams();

		prepareGPXFile(gpxFile);
	}

	private void prepareGPXFile(@NonNull GpxFile file) {
		if (!file.isPointsEmpty()) {
			wpt = new ArrayList<>(file.getPointsSize());
			for (WptPt w : file.getPointsList()) {
				wpt.add(new WptLocationPoint(w));
			}
		}

		if (OSMAND_ROUTER_V2.equals(file.getAuthor())) {
			route = RouteProvider.parseOsmAndGPXRoute(points, file, segmentEndpoints, selectedSegment, leftSide);
			if (selectedSegment == -1) {
				routePoints = file.getRoutePoints();
			} else {
				routePoints = file.getRoutePoints(selectedSegment);
			}
			if (reverse) {
				Collections.reverse(points);
				Collections.reverse(routePoints);
				Collections.reverse(segmentEndpoints);
			}
			addMissingTurns = route != null && route.isEmpty();
		} else if (file.isCloudmadeRouteFile() || OSMAND_ROUTER.equals(file.getAuthor())) {
			directions = RouteProvider.parseOsmAndGPXRoute(points, file, segmentEndpoints,
					OSMAND_ROUTER.equals(file.getAuthor()), leftSide, 10, selectedSegment);
			if (OSMAND_ROUTER.equals(file.getAuthor()) && file.hasRtePt()) {
				// For files generated by OSMAND_ROUTER use directions contained unaltered
				addMissingTurns = false;
			}
			if (reverse) {
				// clear directions all turns should be recalculated
				directions = null;
				Collections.reverse(points);
				Collections.reverse(segmentEndpoints);
				addMissingTurns = true;
			}
		} else {
			if (useIntermediatePointsRTE) {
				List<WptPt> rtePoints = selectedRoute == -1 ? file.getRoutePoints() : file.getRoutePoints(selectedRoute);
				for (WptPt pt : rtePoints) {
					points.add(RouteProvider.createLocation(pt));
				}
			} else {
				RouteProvider.collectSegmentPointsFromGpx(file, points, segmentEndpoints, selectedSegment);
			}
			if (reverse) {
				Collections.reverse(points);
				Collections.reverse(segmentEndpoints);
			}
		}
	}

	public static class GPXRouteParamsBuilder {
		private static final org.apache.commons.logging.Log log = PlatformUtil.getLog(GPXRouteParamsBuilder.class);

		boolean calculateOsmAndRoute;
		// parameters
		private final GpxFile file;
		private final boolean leftSide;
		private boolean reverse;
		private ReverseTrackStrategy reverseStrategy;
		private boolean passWholeRoute;
		private boolean calculateOsmAndRouteParts;
		private boolean calculatedRouteTimeSpeed;
		private boolean connectPointsStraightly;
		private boolean useIntermediateRtePoints;
		private int selectedSegment = -1;
		private int selectedRoute = -1;
		private GpxApproximationParams approximationParams;

		public GPXRouteParamsBuilder(@NonNull GpxFile file, @NonNull OsmandSettings settings) {
			this.file = file;
			this.leftSide = settings.DRIVING_REGION.get().leftHandDriving;
			updateIntermediateRtePoints();
		}

		public GPXRouteParamsBuilder(@NonNull GpxFile gpxFile, @NonNull GPXRouteParams params) {
			this.file = gpxFile;
			this.calculateOsmAndRoute = params.calculateOsmAndRoute;
			this.leftSide = params.leftSide;
			this.reverse = params.reverse;
			this.reverseStrategy = params.reverseStrategy;
			this.passWholeRoute = params.passWholeRoute;
			this.calculateOsmAndRouteParts = params.calculateOsmAndRouteParts;
			this.calculatedRouteTimeSpeed = params.calculatedRouteTimeSpeed;
			this.connectPointsStraightly = params.connectPointsStraightly;
			this.useIntermediateRtePoints = params.useIntermediatePointsRTE;
			this.selectedSegment = params.selectedSegment;
			this.selectedRoute = params.selectedRoute;
			this.approximationParams = params.approximationParams;

			updateIntermediateRtePoints();
		}

		private void updateIntermediateRtePoints() {
			if (selectedRoute == -1) {
				useIntermediateRtePoints = file.hasRtePt() && !file.hasTrkPt();
			} else {
				useIntermediateRtePoints = file.getRoutes().size() > selectedRoute && file.getRoutes().get(selectedRoute).getPoints().isEmpty();
			}
		}

		public boolean isReverse() {
			return reverse;
		}

		public boolean isCalculateOsmAndRouteParts() {
			return calculateOsmAndRouteParts;
		}

		public void setCalculateOsmAndRouteParts(boolean calculateOsmAndRouteParts) {
			this.calculateOsmAndRouteParts = calculateOsmAndRouteParts;
		}

		public boolean isCalculateOsmAndRoute() {
			return calculateOsmAndRoute;
		}

		public void setCalculateOsmAndRoute(boolean calculateOsmAndRoute) {
			this.calculateOsmAndRoute = calculateOsmAndRoute;
		}

		public boolean hasCalculatedRouteTimeSpeed() {
			return calculatedRouteTimeSpeed;
		}

		public void setCalculatedRouteTimeSpeed(boolean calculatedRouteTimeSpeed) {
			this.calculatedRouteTimeSpeed = calculatedRouteTimeSpeed;
		}

		public boolean shouldConnectPointsStraightly() {
			return connectPointsStraightly;
		}

		public void setConnectPointStraightly(boolean connectPointStraightly) {
			this.connectPointsStraightly = connectPointStraightly;
		}

		public int getSelectedSegment() {
			return selectedSegment;
		}

		public void setSelectedSegment(int selectedSegment) {
			this.selectedSegment = selectedSegment;
		}

		public int getSelectedRoute() {
			return selectedRoute;
		}

		public void setSelectedRoute(int selectedRoute) {
			this.selectedRoute = selectedRoute;
			updateIntermediateRtePoints();
		}

		public boolean shouldUseIntermediateRtePoints() {
			return useIntermediateRtePoints;
		}

		public void setPassWholeRoute(boolean passWholeRoute) {
			this.passWholeRoute = passWholeRoute;
		}

		public boolean isPassWholeRoute() {
			return passWholeRoute;
		}

		@Nullable
		public GpxApproximationParams getApproximationParams() {
			return approximationParams;
		}

		public void setApproximationParams(@Nullable GpxApproximationParams params) {
			this.approximationParams = params;
		}

		public GPXRouteParams build(@NonNull OsmandApplication app) {
			return build(app, null);
		}

		public GPXRouteParams build(@NonNull OsmandApplication app, @Nullable LatLon targetPoint) {
			GPXRouteParams gpxRouteParams = new GPXRouteParams();
			try {
				gpxRouteParams.prepareGPXFile(this);
				if (targetPoint != null && !targetPoint.equals(gpxRouteParams.getLastPoint())) {
					cutGpxTail(gpxRouteParams, targetPoint);
				}
			} catch (RuntimeException e) {
				log.error(e.getMessage(), e);
				app.showShortToastMessage(app.getString(R.string.gpx_parse_error) + " " + e.getMessage());
			}
			return gpxRouteParams;
		}

		private void cutGpxTail(GPXRouteParams gpxRouteParams, LatLon targetPoint) {
			int lastIdx = 0;
			double minDist = Float.MAX_VALUE;
			List<Location> locations = gpxRouteParams.points;
			for (int i = 0; i < locations.size(); i++) {
				Location point = locations.get(i);
				double distance = MapUtils.getDistance(point.getLatitude(), point.getLongitude(),
						targetPoint.getLatitude(), targetPoint.getLongitude());
				if (distance < minDist) {
					minDist = distance;
					lastIdx = i;
				}
			}
			if (!Algorithms.isEmpty(gpxRouteParams.segmentEndpoints)) {
				List<Location> removedLocations = gpxRouteParams.points.subList(lastIdx, gpxRouteParams.points.size());
				List<Location> endpoints = gpxRouteParams.segmentEndpoints;
				int firstRemovedIdx = 0;
				for (; firstRemovedIdx < endpoints.size(); firstRemovedIdx++) {
					Location point = endpoints.get(firstRemovedIdx);
					if (removedLocations.contains(point)) {
						break;
					}
				}
				if (firstRemovedIdx < endpoints.size()) {
					endpoints.subList(firstRemovedIdx, endpoints.size()).clear();
				}
			}
			if (!Algorithms.isEmpty(gpxRouteParams.routePoints)) {
				List<WptPt> routePoints = gpxRouteParams.routePoints;
				int firstRemovedIdx = 1;
				for (; firstRemovedIdx < routePoints.size(); firstRemovedIdx++) {
					WptPt routePoint = routePoints.get(firstRemovedIdx);
					if (routePoint.getTrkPtIndex() > lastIdx) {
						break;
					}
				}
				if (firstRemovedIdx < routePoints.size()) {
					routePoints.subList(firstRemovedIdx, routePoints.size()).clear();
				}
			}
			gpxRouteParams.points.subList(lastIdx, gpxRouteParams.points.size()).clear();
			if (!Algorithms.isEmpty(gpxRouteParams.route)) {
				gpxRouteParams.route.clear();
			}
		}

		public void setReverse(boolean reverse) {
			this.reverse = reverse;
		}

		public void setReverseStrategy(@NonNull ReverseTrackStrategy reverseStrategy) {
			this.reverseStrategy = reverseStrategy;
		}

		public GpxFile getFile() {
			return file;
		}

		public List<Location> getPoints(OsmandApplication app) {
			GPXRouteParams copy = build(app);
			return copy.getPoints();
		}

		@NonNull
		@Override
		public String toString() {
			return "GPXRouteParamsBuilder{" +
					"calculateOsmAndRoute=" + calculateOsmAndRoute +
					", file=" + file.getPath() +
					", leftSide=" + leftSide +
					", reverse=" + reverse +
					", reverseStrategy=" + reverseStrategy +
					", passWholeRoute=" + passWholeRoute +
					", calculateOsmAndRouteParts=" + calculateOsmAndRouteParts +
					", calculatedRouteTimeSpeed=" + calculatedRouteTimeSpeed +
					", connectPointsStraightly=" + connectPointsStraightly +
					", useIntermediateRtePoints=" + useIntermediateRtePoints +
					", selectedSegment=" + selectedSegment +
					", selectedRoute=" + selectedRoute +
					'}';
		}
	}
}