package net.osmand.plus.quickaction;


import android.content.Context;
import android.os.Bundle;
import android.view.KeyEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.EditText;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.StringRes;

import net.osmand.plus.OsmandApplication;
import net.osmand.plus.R;
import net.osmand.plus.activities.MapActivity;
import net.osmand.plus.views.layers.MapQuickActionLayer;
import net.osmand.util.Algorithms;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class QuickAction {

	public interface QuickActionSelectionListener {

		void onActionSelected(@NonNull QuickAction action, @Nullable KeyEvent event, boolean forceUpdate);
	}

	private static int SEQ;

	protected long id;
	private String name;
	private Map<String, String> params;
	private QuickActionType actionType;

	protected QuickAction() {
		this(MapButtonsHelper.TYPE_ADD_ITEMS);
	}

	public QuickAction(QuickActionType type) {
		this.id = System.currentTimeMillis() + (SEQ++);
		this.actionType = type;
	}

	public QuickAction(QuickAction quickAction) {
		this.actionType = quickAction.actionType;
		this.id = quickAction.id;
		this.name = quickAction.name;
		this.params = quickAction.params;
	}

	public int getNameRes() {
		return actionType == null ? 0 : actionType.getNameRes();
	}

	public int getActionNameRes() {
		return actionType == null ? 0 : actionType.getActionNameRes();
	}

	public int getIconRes() {
		return actionType == null ? 0 : actionType.getIconRes();
	}

	public int getIconRes(Context context) {
		return actionType == null ? 0 : actionType.getIconRes();
	}

	public long getId() {
		return id;
	}

	public int getType() {
		return actionType.getId();
	}

	public void setActionType(QuickActionType actionType) {
		this.actionType = actionType;
	}

	public boolean isActionEditable() {
		return actionType != null && actionType.isActionEditable();
	}

	public boolean isActionEnable(OsmandApplication app) {
		return true;
	}

	public String getExtendedName(@NonNull Context context) {
		return getExtendedName(context, true);
	}

	public String getExtendedName(@NonNull Context context, boolean useDash) {
		return getExtendedName(context, useDash ? R.string.ltr_or_rtl_combine_via_dash : R.string.ltr_or_rtl_combine_via_space);
	}

	public String getExtendedName(@NonNull Context context, @StringRes int combineId) {
		String name = getName(context);
		if (name.equals(getRawName()) || !shouldUseExtendedName()) {
			return name;
		}
		int actionNameRes = getActionNameRes();
		if (actionNameRes != 0 && !name.contains(context.getString(actionNameRes))) {
			String prefAction = context.getString(actionNameRes);
			return context.getString(combineId, prefAction, name);
		}
		return name;
	}

	public String getName(@NonNull Context context) {
		if (Algorithms.isEmpty(name) || !isActionEditable()) {
			return getDefaultName(context);
		} else {
			return name;
		}
	}

	public String getRawName() {
		return name;
	}

	protected boolean shouldUseExtendedName() {
		return actionType != null && actionType.shouldUseExtendedName();
	}

	@NonNull
	private String getDefaultName(@NonNull Context context) {
		return getNameRes() != 0 ? context.getString(getNameRes()) : "";
	}

	@NonNull
	public Map<String, String> getParams() {
		if (params == null) {
			params = new HashMap<>();
		}
		return params;
	}

	@NonNull
	public String getParameter(@NonNull String key, @NonNull String defValue) {
		String value = getParams().get(key);
		return (value == null || value.isEmpty()) ? defValue : value;
	}

	public void setParameter(@NonNull String key, @Nullable String value) {
		getParams().put(key, value);
	}

	public void setName(String name) {
		this.name = name;
	}

	public void setId(long id) {
		this.id = id;
	}

	public void setParams(Map<String, String> params) {
		this.params = params;
	}

	public boolean isActionWithSlash(@NonNull OsmandApplication app) {
		return false;
	}

	public String getActionText(@NonNull OsmandApplication app) {
		return getExtendedName(app, false);
	}

	public QuickActionType getActionType() {
		return actionType;
	}

	public void setAutoGeneratedTitle(EditText title) {
	}

	public boolean onKeyDown(@NonNull MapActivity mapActivity, int keyCode, KeyEvent event) {
		return true;
	}

	public boolean onKeyLongPress(@NonNull MapActivity mapActivity, int keyCode, KeyEvent event) {
		return true;
	}

	public boolean onKeyUp(@NonNull MapActivity mapActivity, int keyCode, KeyEvent event) {
		onActionSelected(mapActivity, event);
		return true;
	}

	public boolean onKeyMultiple(@NonNull MapActivity mapActivity, int keyCode, int count, KeyEvent event) {
		return true;
	}

	public void onActionSelected(@NonNull MapActivity mapActivity, @Nullable KeyEvent event){
		MapQuickActionLayer actionLayer = mapActivity.getMapLayers().getMapQuickActionLayer();
		actionLayer.onActionSelected(this, event, true);
	}

	public void execute(@NonNull MapActivity mapActivity, @Nullable Bundle params) {
	}

	public void drawUI(@NonNull ViewGroup parent, @NonNull MapActivity mapActivity, boolean nightMode) {
	}

	public boolean fillParams(@NonNull View root, @NonNull MapActivity mapActivity) {
		return true;
	}

	public boolean hasInstanceInList(List<QuickAction> active) {

		for (QuickAction action : active) {
			if (action.getType() == getType()) return true;
		}

		return false;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) return true;
		if (o == null) return false;

		if (o instanceof QuickAction action) {
			if (getType() != action.getType()) return false;
			return id == action.id;
		}
		return false;
	}

	@Override
	public int hashCode() {
		int result = getType();
		result = 31 * result + (int) (id ^ (id >>> 32));
		return result;
	}

	public boolean hasCustomName(Context context) {
		return !getName(context).equals(getDefaultName(context));
	}
}
