import org.jetbrains.kotlin.gradle.dsl.JvmTarget
import org.jetbrains.kotlin.gradle.dsl.KotlinVersion
import java.io.FileInputStream
import java.util.Properties

plugins {
    id("com.android.application")
    id("org.jetbrains.kotlin.android")
    id("org.jetbrains.kotlin.plugin.compose")
    id("org.jetbrains.kotlin.plugin.serialization")
    id("com.google.devtools.ksp")
    id("androidx.navigation.safeargs.kotlin")
    alias(libs.plugins.detekt.plugin)
}

// Google services plugins disabled - no longer using Google services
// val isFdroidBuild = gradle.startParameter.taskRequests.toString().contains("Fdroid")
// if (!isFdroidBuild) {
//     apply(plugin = libs.plugins.google.gms.google.services.get().pluginId)
//     apply(plugin = libs.plugins.google.firebase.crashlytics.get().pluginId)
// }

fun loadLocalProperties(): Properties = Properties().apply {
    val localPropsFile = rootProject.file("local.properties")
    if (localPropsFile.exists()) {
        FileInputStream(localPropsFile).use { load(it) }
    } else {
        setProperty("MIXPANELKEY", System.getenv("MIXPANEL_KEY") ?: "")
        setProperty("STOREFILE", System.getenv("STOREFILE") ?: "")
        setProperty("STOREPASSWORD", System.getenv("STOREPASSWORD") ?: "")
        setProperty("KEYALIAS", System.getenv("KEYALIAS") ?: "")
        setProperty("KEYPASSWORD", System.getenv("KEYPASSWORD") ?: "")
    }
}

kotlin {

    compilerOptions {

        jvmTarget.set(JvmTarget.JVM_17)
        languageVersion.set(KotlinVersion.KOTLIN_2_2)
    }
}

kotlin {
    compilerOptions {

        jvmTarget.set(JvmTarget.JVM_17)
        languageVersion.set(KotlinVersion.KOTLIN_2_2)
    }
}

android {

    compileSdk = 36

    compileOptions {
        sourceCompatibility = JavaVersion.VERSION_17
        targetCompatibility = JavaVersion.VERSION_17
    }

    defaultConfig {
        applicationId = "net.opendasharchive.openarchive"
        minSdk = 29
        targetSdk = 36
        versionCode = 30020
        versionName = "4.0.2"
        multiDexEnabled = true
        vectorDrawables.useSupportLibrary = true
        testInstrumentationRunner = "androidx.test.runner.AndroidJUnitRunner"
        val localProps = loadLocalProperties()
        resValue("string", "mixpanel_key", localProps.getProperty("MIXPANELKEY") ?: "")
    }

    base {
        archivesName.set("save-${defaultConfig.versionName}")
    }

    buildFeatures {
        viewBinding = true
        buildConfig = true
        compose = true
    }

    buildTypes {

        getByName("release") {
            isMinifyEnabled = false
            isShrinkResources = false
            proguardFiles(getDefaultProguardFile("proguard-android.txt"), "proguard-rules.pro")
        }

        getByName("debug") {
            isMinifyEnabled = false
        }
    }

    flavorDimensions += "env"

    productFlavors {

        create("dev") {
            dimension = "env"
            versionNameSuffix = "-dev"
            applicationIdSuffix = ".debug"
            buildConfigField("boolean", "INCLUDE_GOOGLE_SERVICES", "true")
            buildConfigField("boolean", "INCLUDE_MIXPANEL", "true")
        }

        create("staging") {
            dimension = "env"
            versionNameSuffix = "-staging"
            applicationIdSuffix = ".debug"
            buildConfigField("boolean", "INCLUDE_GOOGLE_SERVICES", "true")
            buildConfigField("boolean", "INCLUDE_MIXPANEL", "true")
        }

        create("prod") {
            dimension = "env"
            applicationIdSuffix = ".release"
            buildConfigField("boolean", "INCLUDE_GOOGLE_SERVICES", "true")
            buildConfigField("boolean", "INCLUDE_MIXPANEL", "true")
        }

        create("fdroid") {
            dimension = "env"
            versionNameSuffix = "-fdroid"
            applicationIdSuffix = ".release"
            resValue("string", "mixpanel_key", "")
            buildConfigField("boolean", "INCLUDE_GOOGLE_SERVICES", "false")
            buildConfigField("boolean", "INCLUDE_MIXPANEL", "false")
        }
    }

    // Exclude Google Drive related files for F-Droid build
    tasks.withType<org.jetbrains.kotlin.gradle.tasks.KotlinCompile>().configureEach {
        if (name.contains("Fdroid")) {
            exclude("**/gdrive/**", "**/DriveServiceHelper.kt")
        }
    }


    packaging {
        resources {
            excludes.addAll(
                listOf(
                    "META-INF/LICENSE.txt", "META-INF/NOTICE.txt", "META-INF/LICENSE",
                    "META-INF/NOTICE", "META-INF/DEPENDENCIES", "LICENSE.txt"
                )
            )
        }
//        jniLibs {
//            useLegacyPackaging = false
//            // Exclude the problematic C2PA native library for F-Droid builds
//            if (gradle.startParameter.taskRequests.toString().contains("Fdroid")) {
//                excludes += "**/libc2pa_c.so"
//            }
//        }
    }

    lint {
        abortOnError = false
    }

    testOptions {
        unitTests {
            isIncludeAndroidResources = true
        }
    }

    namespace = "net.opendasharchive.openarchive"

    configurations.all {
        resolutionStrategy {
            force("org.bouncycastle:bcprov-jdk15to18:1.72")
            exclude(group = "org.bouncycastle", module = "bcprov-jdk15on")
        }
    }
}

dependencies {

    // Core Kotlin and Coroutines
    implementation(libs.kotlinx.coroutines.android)
    implementation(libs.kotlinx.serialization.json)


    // AndroidX Libraries
    implementation(libs.androidx.appcompat)
    implementation(libs.androidx.recyclerview)
    implementation(libs.androidx.viewpager2)
    implementation(libs.androidx.recyclerview.selection)
    implementation(libs.androidx.constraintlayout)
    implementation(libs.androidx.constraintlayout.compose)
    implementation(libs.androidx.coordinatorlayout)
    implementation(libs.androidx.core.splashscreen)

    implementation(libs.androidx.lifecycle.viewmodel.ktx)
    implementation(libs.androidx.lifecycle.viewmodel.compose)
    implementation(libs.androidx.lifecycle.livedata)
    implementation(libs.androidx.lifecycle.runtime.compose)

    implementation(libs.androidx.preferences)
    implementation(libs.androidx.biometric)
    implementation(libs.androidx.work)
    implementation(libs.androidx.security.crypto)

    implementation(libs.androidx.fragment.ktx)
    implementation(libs.androidx.fragment.compose)

    // Compose Preferences
    implementation(libs.compose.preferences)

    // Material Design
    implementation(libs.google.material)

    // AndroidX SwipeRefreshLayout
    implementation(libs.androidx.swiperefresh)

    // Compose Libraries
    implementation(libs.androidx.core.ktx)
    implementation(libs.androidx.activity.ktx)
    implementation(libs.androidx.activity.compose)
    implementation(libs.androidx.compose.material3)
    implementation(libs.androidx.compose.ui)
    implementation(libs.androidx.compose.foundation)
    implementation(libs.androidx.compose.ui.tooling.preview)
    implementation(libs.androidx.compose.icons.extended)
    // Firebase Crashlytics - COMMENTED OUT (no longer using Google services)
    // "devImplementation"(libs.firebase.crashlytics)
    // "stagingImplementation"(libs.firebase.crashlytics)
    // "prodImplementation"(libs.firebase.crashlytics)
    debugImplementation(libs.androidx.compose.ui.tooling)

    implementation(libs.androidx.compose.runtime)
    implementation(libs.androidx.compose.runtime.livedata)

    // Navigation
    implementation(libs.androidx.navigation.compose)
    implementation(libs.androidx.navigation.ui)
    implementation(libs.androidx.navigation.fragment)
    implementation(libs.androidx.navigation.fragment.compose)

    // Preference
    implementation(libs.androidx.preferences)

    // Dependency Injection
    implementation(libs.koin.core)
    implementation(libs.koin.android)
    implementation(libs.koin.androidx.compose)
    implementation(libs.koin.androidx.navigation)
    implementation(libs.koin.compose)
    implementation(libs.koin.compose.viewmodel)
    implementation(libs.koin.compose.viewmodel.navigation)

    // Image Libraries
    implementation(libs.coil)
    implementation(libs.coil.compose)
    implementation(libs.coil.video)
    implementation(libs.coil.network)

    // Networking and Data
    // Networking
    implementation(libs.retrofit)
    implementation(libs.retrofit.gson)
    implementation(libs.gson)
    implementation(libs.okhttp)
    implementation(libs.okhttp.logging)
    implementation(libs.guardianproject.sardine)

    // Utility Libraries
    implementation(libs.timber)
    //implementation(libs.orhanobut.logger)
    //implementation(libs.abdularis.circularimageview)
    implementation(libs.dotsindicator)
    implementation(libs.permissionx)

    // Barcode Scanning
    //implementation("com.google.zxing:core:3.5.3")
    //implementation("com.journeyapps:zxing-android-embedded:4.3.0")

    // Security and Encryption
    implementation(libs.bouncycastle.bcpkix)
    implementation(libs.bouncycastle.bcprov)
    api(libs.bouncycastle.bcpg)

    // Google Play Services - COMMENTED OUT (no longer using Google services)
    // "devImplementation"(libs.google.auth)
    // "stagingImplementation"(libs.google.auth)
    // "prodImplementation"(libs.google.auth)
    // "devImplementation"(libs.google.play.asset.delivery.ktx)
    // "stagingImplementation"(libs.google.play.asset.delivery.ktx)
    // "prodImplementation"(libs.google.play.asset.delivery.ktx)
    // "devImplementation"(libs.google.play.feature.delivery)
    // "stagingImplementation"(libs.google.play.feature.delivery)
    // "prodImplementation"(libs.google.play.feature.delivery)
    // "devImplementation"(libs.google.play.feature.delivery.ktx)
    // "stagingImplementation"(libs.google.play.feature.delivery.ktx)
    // "prodImplementation"(libs.google.play.feature.delivery.ktx)
    // "devImplementation"(libs.google.play.review)
    // "stagingImplementation"(libs.google.play.review)
    // "prodImplementation"(libs.google.play.review)
    // "devImplementation"(libs.google.play.review.ktx)
    // "stagingImplementation"(libs.google.play.review.ktx)
    // "prodImplementation"(libs.google.play.review.ktx)
    // "devImplementation"(libs.google.play.app.update.ktx)
    // "stagingImplementation"(libs.google.play.app.update.ktx)
    // "prodImplementation"(libs.google.play.app.update.ktx)

    // Google Drive API - COMMENTED OUT (no longer using Google services)
    // "devImplementation"(libs.google.http.client.gson)
    // "stagingImplementation"(libs.google.http.client.gson)
    // "prodImplementation"(libs.google.http.client.gson)
    // "devImplementation"(libs.google.api.client.android)
    // "stagingImplementation"(libs.google.api.client.android)
    // "prodImplementation"(libs.google.api.client.android)
    // "devImplementation"(libs.google.drive.api)
    // "stagingImplementation"(libs.google.drive.api)
    // "prodImplementation"(libs.google.drive.api)

    // Tor Libraries
    implementation(libs.tor.android)
    implementation(libs.jtorctl)

    // ProofMode - COMMENTED OUT
    // implementation(libs.bitcoinj.core)
    //implementation("com.eclipsesource.j2v8:j2v8:6.2.1@aar")

    // ProofMode - built from source as local module for F-Droid compatibility - COMMENTED OUT
    // implementation(project(":proofmode:android-libproofmode")) {
    //     //transitive = false
    //     exclude(group = "org.bitcoinj")
    //     exclude(group = "com.google.protobuf")
    //     exclude(group = "org.slf4j")
    //     exclude(group = "net.jcip")
    //     exclude(group = "commons-cli")
    //     exclude(group = "org.json")
    //     exclude(group = "com.google.guava")
    //     exclude(group = "com.google.guava", module = "guava-jdk5")
    //     exclude(group = "com.google.code.findbugs", module = "annotations")
    //     exclude(group = "com.squareup.okio", module = "okio")
    //     // Exclude BouncyCastle to avoid duplicate classes - we'll use our own version
    //     exclude(group = "org.bouncycastle")
    //     // Exclude Google Play Services for F-Droid compatibility
    //     exclude(group = "com.google.android.gms")
    // }

    // Guava Conflicts
    implementation(libs.guava)
    implementation(libs.guava.listenablefuture)

    implementation(libs.satyan.sugar)

    // adding web dav support: https://github.com/thegrizzlylabs/sardine-android'
    implementation("com.github.guardianproject:sardine-android:89f7eae512")

    implementation("com.github.derlio:audio-waveform:v1.0.1")

    implementation(libs.clean.insights)
    implementation(libs.netcipher)

    // Mixpanel analytics - COMMENTED OUT (no longer using proprietary analytics)
    // "devImplementation"(libs.mixpanel)
    // "stagingImplementation"(libs.mixpanel)
    // "prodImplementation"(libs.mixpanel)

    // Tests
    testImplementation(libs.junit)
    testImplementation(libs.robolectric)
    androidTestImplementation(libs.androidx.test.junit)
    androidTestImplementation(libs.androidx.test.runner)
    testImplementation(libs.work.testing)

    // Detekt
    detektPlugins(libs.detekt.formatting)
    detektPlugins(libs.detekt.rules.authors)
    detektPlugins(libs.detekt.rules.libraries)
    detektPlugins(libs.detekt.compose)
    detektPlugins(libs.detekt.rules.compose)

//    debugImplementation("com.squareup.leakcanary:leakcanary-android:3.0-alpha-8")
}

configurations.all {
    exclude(group = "com.google.guava", module = "listenablefuture")
}

detekt {
    config.setFrom(file("$rootDir/config/detekt-config.yml"))
    baseline = file("$rootDir/config/baseline.xml")
    source.setFrom(
        files("$rootDir/app/src")
    )
    buildUponDefaultConfig = true
    allRules = false
    autoCorrect = false
    ignoreFailures = true
}