// Copyright 2024 - Nym Technologies SA <contact@nymtech.net>
// SPDX-License-Identifier: GPL-3.0-only

use std::{net::IpAddr, path::PathBuf};

use futures::{StreamExt, stream::BoxStream};
use tokio::{
    sync::{
        broadcast,
        mpsc::{self, UnboundedReceiver, UnboundedSender},
        oneshot,
    },
    task::JoinHandle,
};
use tokio_util::sync::CancellationToken;
use tonic::transport::Server;

use nym_vpn_lib_types::{
    EnableSocks5Request, EntryPoint, ExitPoint, ListGatewaysOptions, LookupGatewayFilters,
    TargetState, TunnelEvent,
};

use nym_vpn_proto::proto::{
    self,
    nym_vpn_service_server::{NymVpnService, NymVpnServiceServer},
};

use crate::service::{SetNetworkError, Socks5Error, VpnServiceCommand};

pub type Result<T> = std::result::Result<T, tonic::Status>;

pub struct CommandInterface {
    // Send commands to the VPN service
    vpn_command_tx: UnboundedSender<VpnServiceCommand>,

    // Broadcast tunnel events to our API endpoint listeners
    tunnel_event_rx: broadcast::Receiver<TunnelEvent>,
}

impl CommandInterface {
    fn new(
        vpn_command_tx: UnboundedSender<VpnServiceCommand>,
        tunnel_event_rx: broadcast::Receiver<TunnelEvent>,
    ) -> Self {
        Self {
            vpn_command_tx,
            tunnel_event_rx,
        }
    }

    async fn send_and_wait<R, F, O>(&self, command: F, opts: O) -> Result<R>
    where
        F: FnOnce(oneshot::Sender<R>, O) -> VpnServiceCommand,
    {
        let (tx, rx) = oneshot::channel();

        self.vpn_command_tx
            .send(command(tx, opts))
            .map_err(|_| tonic::Status::internal("Command channel is closed"))?;

        rx.await
            .map_err(|_| tonic::Status::internal("Response channel is closed"))
    }
}

#[tonic::async_trait]
impl NymVpnService for CommandInterface {
    async fn info(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::InfoResponse>> {
        let info = self.send_and_wait(VpnServiceCommand::Info, ()).await?;
        let response = proto::InfoResponse::from(info);

        Ok(tonic::Response::new(response))
    }

    async fn get_config(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetConfigResponse>> {
        let config = self.send_and_wait(VpnServiceCommand::GetConfig, ()).await?;

        let response = proto::GetConfigResponse {
            config: Some(proto::VpnServiceConfig::from(config)),
        };

        Ok(tonic::Response::new(response))
    }

    async fn set_entry_point(
        &self,
        request: tonic::Request<proto::EntryNode>,
    ) -> Result<tonic::Response<()>> {
        let entry_point = EntryPoint::try_from(request.into_inner())
            .map_err(|e| tonic::Status::invalid_argument(format!("Invalid entry point: {e}")))?;

        let _ = self
            .send_and_wait(VpnServiceCommand::SetEntryPoint, entry_point)
            .await
            .map_err(|e| tonic::Status::internal(format!("Failed to set VPN entry point: {e}")))?;

        Ok(tonic::Response::new(()))
    }

    async fn set_exit_point(
        &self,
        request: tonic::Request<proto::ExitNode>,
    ) -> Result<tonic::Response<()>> {
        let exit_point = ExitPoint::try_from(request.into_inner())
            .map_err(|e| tonic::Status::invalid_argument(format!("Invalid exit point: {e}")))?;

        let _ = self
            .send_and_wait(VpnServiceCommand::SetExitPoint, exit_point)
            .await
            .map_err(|e| tonic::Status::internal(format!("Failed to set VPN exit point: {e}")))?;

        Ok(tonic::Response::new(()))
    }

    async fn set_disable_ipv6(&self, request: tonic::Request<bool>) -> Result<tonic::Response<()>> {
        let disable_ipv6 = request.into_inner();

        let _ = self
            .send_and_wait(VpnServiceCommand::SetDisableIPv6, disable_ipv6)
            .await
            .map_err(|e| tonic::Status::internal(format!("Failed to set IPv6 config: {e}")))?;

        Ok(tonic::Response::new(()))
    }

    async fn set_enable_two_hop(
        &self,
        request: tonic::Request<bool>,
    ) -> Result<tonic::Response<()>> {
        let enable_two_hop = request.into_inner();

        let _ = self
            .send_and_wait(VpnServiceCommand::SetEnableTwoHop, enable_two_hop)
            .await
            .map_err(|e| tonic::Status::internal(format!("Failed to set two-hop config: {e}")))?;

        Ok(tonic::Response::new(()))
    }

    async fn set_enable_bridges(
        &self,
        request: tonic::Request<bool>,
    ) -> Result<tonic::Response<()>> {
        let enable_bridges = request.into_inner();

        let _ = self
            .send_and_wait(VpnServiceCommand::SetEnableBridges, enable_bridges)
            .await
            .map_err(|e| tonic::Status::internal(format!("Failed to set enable bridges: {e}")))?;

        Ok(tonic::Response::new(()))
    }

    async fn set_netstack(&self, request: tonic::Request<bool>) -> Result<tonic::Response<()>> {
        let netstack = request.into_inner();

        let _ = self
            .send_and_wait(VpnServiceCommand::SetNetstack, netstack)
            .await
            .map_err(|e| tonic::Status::internal(format!("Failed to set netstack config: {e}")))?;

        Ok(tonic::Response::new(()))
    }

    async fn set_allow_lan(&self, request: tonic::Request<bool>) -> Result<tonic::Response<()>> {
        let allow_lan = request.into_inner();

        let _ = self
            .send_and_wait(VpnServiceCommand::SetAllowLan, allow_lan)
            .await
            .map_err(|e| tonic::Status::internal(format!("Failed to set allow lan: {e}")))?;

        Ok(tonic::Response::new(()))
    }

    async fn set_residential_exit(
        &self,
        request: tonic::Request<bool>,
    ) -> Result<tonic::Response<()>> {
        let residential_exit = request.into_inner();

        let _ = self
            .send_and_wait(VpnServiceCommand::SetResidentialExit, residential_exit)
            .await
            .map_err(|e| {
                tonic::Status::internal(format!("Failed to set residential exit only: {e}"))
            })?;

        Ok(tonic::Response::new(()))
    }

    async fn set_enable_custom_dns(
        &self,
        request: tonic::Request<bool>,
    ) -> Result<tonic::Response<()>> {
        let enable_custom_dns = request.into_inner();

        let _ = self
            .send_and_wait(VpnServiceCommand::SetEnableCustomDns, enable_custom_dns)
            .await
            .map_err(|e| {
                tonic::Status::internal(format!("Failed to set enable custom DNS: {e}"))
            })?;

        Ok(tonic::Response::new(()))
    }

    async fn set_custom_dns(
        &self,
        request: tonic::Request<proto::IpAddrList>,
    ) -> Result<tonic::Response<()>> {
        let custom_dns: Vec<IpAddr> = request
            .into_inner()
            .try_into()
            .map_err(|e| tonic::Status::invalid_argument(format!("Invalid Custom DNS: {e}")))?;

        let _ = self
            .send_and_wait(VpnServiceCommand::SetCustomDns, custom_dns)
            .await
            .map_err(|e| tonic::Status::internal(format!("Failed to set custom DNS: {e}")))?;

        Ok(tonic::Response::new(()))
    }

    async fn set_network(&self, request: tonic::Request<String>) -> Result<tonic::Response<()>> {
        let network = request.into_inner();
        let status = self
            .send_and_wait(VpnServiceCommand::SetNetwork, network)
            .await?;

        status.map_err(|e| match e {
            SetNetworkError::NetworkNotFound(network_name) => {
                tonic::Status::not_found(format!("Network not found: {network_name}"))
            }
            e => tonic::Status::internal(e.to_string()),
        })?;

        Ok(tonic::Response::new(()))
    }

    async fn get_system_messages(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetSystemMessagesResponse>> {
        let messages = self
            .send_and_wait(VpnServiceCommand::GetSystemMessages, ())
            .await?;

        let messages = messages
            .into_iter()
            .map(proto::SystemMessage::from)
            .collect::<Vec<_>>();
        let response = proto::GetSystemMessagesResponse { messages };

        Ok(tonic::Response::new(response))
    }

    async fn get_network_compatibility(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetNetworkCompatibilityResponse>> {
        let network_compatibility = self
            .send_and_wait(VpnServiceCommand::GetNetworkCompatibility, ())
            .await?
            .map(proto::NetworkCompatibility::from);

        let response = proto::GetNetworkCompatibilityResponse {
            network_compatibility,
        };

        Ok(tonic::Response::new(response))
    }

    async fn get_feature_flags(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetFeatureFlagsResponse>> {
        let feature_flags = self
            .send_and_wait(VpnServiceCommand::GetFeatureFlags, ())
            .await?
            .ok_or(tonic::Status::not_found("Feature flags not found"))?;

        Ok(tonic::Response::new(feature_flags.into()))
    }

    async fn get_default_dns(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::IpAddrList>> {
        let dns_ips = self
            .send_and_wait(VpnServiceCommand::GetDefaultDns, ())
            .await?;
        let ipaddr_list = proto::IpAddrList::from(dns_ips);
        Ok(tonic::Response::new(ipaddr_list))
    }

    async fn connect_tunnel(&self, _request: tonic::Request<()>) -> Result<tonic::Response<bool>> {
        let accepted = self
            .send_and_wait(VpnServiceCommand::SetTargetState, TargetState::Secured)
            .await?;

        Ok(tonic::Response::new(accepted))
    }

    async fn reconnect_tunnel(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<bool>> {
        let accepted = self.send_and_wait(VpnServiceCommand::Reconnect, ()).await?;

        Ok(tonic::Response::new(accepted))
    }

    async fn disconnect_tunnel(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<bool>> {
        let accepted = self
            .send_and_wait(VpnServiceCommand::SetTargetState, TargetState::Unsecured)
            .await?;

        Ok(tonic::Response::new(accepted))
    }

    async fn get_tunnel_state(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::TunnelState>> {
        let tunnel_state = self
            .send_and_wait(VpnServiceCommand::GetTunnelState, ())
            .await
            .map(proto::TunnelState::from)?;

        Ok(tonic::Response::new(tunnel_state))
    }

    type ListenToEventsStream = BoxStream<'static, Result<proto::TunnelEvent>>;
    async fn listen_to_events(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<Self::ListenToEventsStream>> {
        let rx = self.tunnel_event_rx.resubscribe();
        let stream = tokio_stream::wrappers::BroadcastStream::new(rx).map(|event| {
            event
                .map(proto::TunnelEvent::from)
                .map_err(|_| tonic::Status::internal("Failed to receive tunnel event"))
        });
        Ok(tonic::Response::new(
            Box::pin(stream) as Self::ListenToEventsStream
        ))
    }

    async fn list_gateways(
        &self,
        request: tonic::Request<proto::ListGatewaysRequest>,
    ) -> Result<tonic::Response<proto::ListGatewaysResponse>> {
        let options = ListGatewaysOptions::try_from(request.into_inner())
            .map_err(|err| tonic::Status::invalid_argument(err.to_string()))?;

        let gateways = self
            .send_and_wait(VpnServiceCommand::ListGateways, options)
            .await?
            .map_err(|err| tonic::Status::internal(format!("Failed to list gateways: {err}")))?;

        let response = proto::ListGatewaysResponse {
            gateways: gateways
                .into_iter()
                .map(proto::GatewayResponse::from)
                .collect(),
        };
        Ok(tonic::Response::new(response))
    }

    async fn list_filtered_gateways(
        &self,
        request: tonic::Request<proto::LookupGatewayFilters>,
    ) -> Result<tonic::Response<proto::ListGatewaysResponse>> {
        let filters = LookupGatewayFilters::try_from(request.into_inner())
            .map_err(|err| tonic::Status::invalid_argument(err.to_string()))?;

        let gateways = self
            .send_and_wait(VpnServiceCommand::ListFilteredGateways, filters)
            .await?
            .map_err(|err| {
                tonic::Status::internal(format!("Failed to list filtered gateways: {err}"))
            })?;

        let response = proto::ListGatewaysResponse {
            gateways: gateways
                .into_iter()
                .map(proto::GatewayResponse::from)
                .collect(),
        };
        Ok(tonic::Response::new(response))
    }

    async fn store_account(
        &self,
        request: tonic::Request<proto::StoreAccountRequest>,
    ) -> Result<tonic::Response<proto::AccountCommandResponse>> {
        let store_request = nym_vpn_lib_types::StoreAccountRequest::try_from(request.into_inner())
            .map_err(|err| tonic::Status::invalid_argument(err.to_string()))?;

        let result = self
            .send_and_wait(VpnServiceCommand::StoreAccount, store_request)
            .await?;

        let response = proto::AccountCommandResponse {
            error: result.err().map(proto::AccountCommandError::from),
        };

        Ok(tonic::Response::new(response))
    }

    async fn account_balance(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::AccountBalanceResponse>> {
        let response = self
            .send_and_wait(VpnServiceCommand::DecentralisedBalance, ())
            .await?;

        let response = proto::AccountBalanceResponse::from(response);

        Ok(tonic::Response::new(response))
    }

    async fn decentralised_obtain_ticketbooks(
        &self,
        request: tonic::Request<proto::DecentralisedObtainTicketbooksRequest>,
    ) -> Result<tonic::Response<proto::AccountCommandResponse>> {
        let ticketbook_request =
            nym_vpn_lib_types::DecentralisedObtainTicketbooksRequest::from(request.into_inner());
        let result = self
            .send_and_wait(
                VpnServiceCommand::DecentralisedObtainTicketbooks,
                ticketbook_request,
            )
            .await?;

        let response = proto::AccountCommandResponse {
            error: result.err().map(proto::AccountCommandError::from),
        };

        Ok(tonic::Response::new(response))
    }

    async fn is_account_stored(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<bool>> {
        let is_stored = self
            .send_and_wait(VpnServiceCommand::IsAccountStored, ())
            .await?;

        Ok(tonic::Response::new(is_stored))
    }

    async fn forget_account(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::AccountCommandResponse>> {
        let result = self
            .send_and_wait(VpnServiceCommand::ForgetAccount, ())
            .await?;

        let response = proto::AccountCommandResponse {
            error: result.err().map(proto::AccountCommandError::from),
        };

        Ok(tonic::Response::new(response))
    }

    async fn rotate_keys(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::AccountCommandResponse>> {
        let result = self
            .send_and_wait(VpnServiceCommand::RotateKeys, ())
            .await?;

        let response = proto::AccountCommandResponse {
            error: result.err().map(proto::AccountCommandError::from),
        };

        Ok(tonic::Response::new(response))
    }

    async fn get_account_identity(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetAccountIdentityResponse>> {
        let account_identity = self
            .send_and_wait(VpnServiceCommand::GetAccountIdentity, ())
            .await?
            .map_err(|err| {
                tonic::Status::internal(format!("Failed to get account identity: {err}"))
            })?;

        Ok(tonic::Response::new(proto::GetAccountIdentityResponse {
            account_identity,
        }))
    }

    async fn get_account_links(
        &self,
        request: tonic::Request<proto::GetAccountLinksRequest>,
    ) -> Result<tonic::Response<proto::AccountManagement>> {
        let locale = request.into_inner().locale;

        let account_links = self
            .send_and_wait(VpnServiceCommand::GetAccountLinks, locale)
            .await?
            .map_err(|err| {
                tonic::Status::internal(format!("Failed to get account links: {err}"))
            })?;

        Ok(tonic::Response::new(proto::AccountManagement::from(
            account_links,
        )))
    }

    async fn get_account_state(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::AccountControllerState>> {
        let account_controller_state = self
            .send_and_wait(VpnServiceCommand::GetAccountState, ())
            .await
            .map(proto::AccountControllerState::from)?;

        Ok(tonic::Response::new(account_controller_state))
    }

    async fn refresh_account_state(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<()>> {
        self.send_and_wait(VpnServiceCommand::RefreshAccountState, ())
            .await?;

        Ok(tonic::Response::new(()))
    }

    async fn get_account_usage(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetAccountUsageResponse>> {
        let account_usage = self
            .send_and_wait(VpnServiceCommand::GetAccountUsage, ())
            .await?
            .map_err(|err| {
                tonic::Status::internal(format!("Failed to get account usage: {err}"))
            })?;

        Ok(tonic::Response::new(proto::GetAccountUsageResponse {
            account_usages: Some(proto::get_account_usage_response::AccountUsages::from(
                account_usage,
            )),
        }))
    }

    async fn reset_device_identity(
        &self,
        request: tonic::Request<proto::ResetDeviceIdentityRequest>,
    ) -> Result<tonic::Response<()>> {
        let seed: Option<[u8; 32]> = request
            .into_inner()
            .seed
            .map(|seed| {
                seed.as_slice()
                    .try_into()
                    .map_err(|_| tonic::Status::invalid_argument("Seed must be 32 bytes long"))
            })
            .transpose()?;

        self.send_and_wait(VpnServiceCommand::ResetDeviceIdentity, seed)
            .await?
            .map_err(|err| {
                tonic::Status::internal(format!("Failed to reset device identity: {err}"))
            })?;

        Ok(tonic::Response::new(()))
    }

    async fn get_device_identity(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetDeviceIdentityResponse>> {
        let device_identity = self
            .send_and_wait(VpnServiceCommand::GetDeviceIdentity, ())
            .await?
            .map_err(|err| {
                tonic::Status::internal(format!("Failed to get device identity: {err}"))
            })?;

        Ok(tonic::Response::new(proto::GetDeviceIdentityResponse {
            device_identity,
        }))
    }

    async fn get_devices(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetDevicesResponse>> {
        let devices = self
            .send_and_wait(VpnServiceCommand::GetDevices, ())
            .await?
            .map_err(|err| tonic::Status::internal(format!("Failed to get devices: {err}")))?;

        Ok(tonic::Response::new(proto::GetDevicesResponse {
            devices: Some(proto::get_devices_response::Devices::from(devices)),
        }))
    }

    async fn get_active_devices(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetDevicesResponse>> {
        let devices = self
            .send_and_wait(VpnServiceCommand::GetActiveDevices, ())
            .await?
            .map_err(|err| {
                tonic::Status::internal(format!("Failed to get active devices: {err}"))
            })?;

        Ok(tonic::Response::new(proto::GetDevicesResponse {
            devices: Some(proto::get_devices_response::Devices::from(devices)),
        }))
    }

    async fn get_available_tickets(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::AvailableTickets>> {
        let available_ticketbooks = self
            .send_and_wait(VpnServiceCommand::GetAvailableTickets, ())
            .await?
            .map_err(|err| {
                tonic::Status::internal(format!("Failed to get available tickets: {err}"))
            })?;

        let available_tickets = nym_vpn_lib_types::AvailableTickets::from(available_ticketbooks);
        let response = proto::AvailableTickets::from(available_tickets);

        Ok(tonic::Response::new(response))
    }

    async fn get_log_path(
        &self,
        _: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetLogPathResponse>> {
        let log_path = self
            .send_and_wait(VpnServiceCommand::GetLogPath, ())
            .await?
            .unwrap_or(crate::logging::default_log_path());

        Ok(tonic::Response::new(log_path.try_into().map_err(
            |err| tonic::Status::internal(format!("Failed to obtain log path: {err}")),
        )?))
    }

    async fn delete_log_file(&self, _request: tonic::Request<()>) -> Result<tonic::Response<()>> {
        self.send_and_wait(VpnServiceCommand::DeleteLogFile, ())
            .await?;

        Ok(tonic::Response::new(()))
    }

    async fn is_sentry_enabled(&self, _: tonic::Request<()>) -> Result<tonic::Response<bool>> {
        let result = self
            .send_and_wait(VpnServiceCommand::IsSentryEnabled, ())
            .await?;
        Ok(tonic::Response::new(result))
    }

    async fn enable_sentry(&self, _: tonic::Request<()>) -> Result<tonic::Response<()>> {
        self.send_and_wait(VpnServiceCommand::ToggleSentry, true)
            .await?
            .map_err(|err| {
                tracing::error!("Failed to enable sentry monitoring: {err}");
                tonic::Status::internal("failed to enable sentry")
            })?;
        Ok(tonic::Response::new(()))
    }

    async fn disable_sentry(&self, _: tonic::Request<()>) -> Result<tonic::Response<()>> {
        self.send_and_wait(VpnServiceCommand::ToggleSentry, false)
            .await?
            .map_err(|err| {
                tracing::error!("Failed to disable sentry monitoring: {err}");
                tonic::Status::internal("failed to disable sentry")
            })?;
        Ok(tonic::Response::new(()))
    }

    async fn network_stats_set_enabled(
        &self,
        request: tonic::Request<bool>,
    ) -> Result<tonic::Response<()>> {
        let enabled = request.into_inner();

        let _ = self
            .send_and_wait(VpnServiceCommand::EnableNetStats, enabled)
            .await
            .map_err(|e| {
                tonic::Status::internal(format!("Failed to enable/disable network statistics: {e}"))
            })?;

        Ok(tonic::Response::new(()))
    }

    async fn network_stats_allow_disconnected(
        &self,
        request: tonic::Request<bool>,
    ) -> std::result::Result<tonic::Response<()>, tonic::Status> {
        let allow_disconnected = request.into_inner();

        let _ = self
            .send_and_wait(
                VpnServiceCommand::AllowDisconnectedNetStats,
                allow_disconnected,
            )
            .await
            .map_err(|e| {
                tonic::Status::internal(format!(
                    "Failed to set network statistics allow_disconnected: {e}"
                ))
            })?;

        Ok(tonic::Response::new(()))
    }

    async fn network_stats_reset_seed(
        &self,
        request: tonic::Request<proto::NetworkStatsResetSeedRequest>,
    ) -> Result<tonic::Response<()>> {
        let seed = request.into_inner().seed;

        let _ = self
            .send_and_wait(VpnServiceCommand::ResetNetStatsSeed, seed)
            .await
            .map_err(|e| {
                tonic::Status::internal(format!("Failed to reset network statistics seed: {e}"))
            })?;

        Ok(tonic::Response::new(()))
    }

    async fn network_stats_get_seed(
        &self,
        _: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::NetworkStatisticsIdentity>> {
        let identity = self
            .send_and_wait(VpnServiceCommand::GetNetStatsSeed, ())
            .await?
            .map_err(|e| {
                tonic::Status::internal(format!("Failed to get network statistics identity: {e}"))
            })?;

        Ok(tonic::Response::new(identity.into()))
    }

    async fn enable_socks5(
        &self,
        request: tonic::Request<proto::EnableSocks5Request>,
    ) -> Result<tonic::Response<()>> {
        let req = request.into_inner();

        let enable_socks5_request: EnableSocks5Request = req.try_into().map_err(|e| {
            tonic::Status::invalid_argument(format!("Invalid Enable SOCKS5 Request: {e}"))
        })?;

        self.send_and_wait(VpnServiceCommand::EnableSocks5, enable_socks5_request)
            .await?
            .map_err(|err| {
                tracing::error!("Failed to enable SOCKS5 proxy: {err}");
                match err {
                    Socks5Error::GatewayNotSupported => tonic::Status::failed_precondition(
                        "Gateway does not support SOCKS5 network requester",
                    ),
                    Socks5Error::InvalidConfig(msg) => tonic::Status::failed_precondition(msg),
                    Socks5Error::LazySocks5Error(_) => {
                        tonic::Status::internal(format!("Failed to enable SOCKS5 proxy: {err}"))
                    }
                }
            })?;

        Ok(tonic::Response::new(()))
    }

    async fn disable_socks5(&self, _: tonic::Request<()>) -> Result<tonic::Response<()>> {
        self.send_and_wait(VpnServiceCommand::DisableSocks5, ())
            .await?
            .map_err(|err| {
                tracing::error!("Failed to disable SOCKS5 proxy: {err}");
                tonic::Status::internal(format!("Failed to disable SOCKS5 proxy: {err}"))
            })?;

        Ok(tonic::Response::new(()))
    }

    async fn get_socks5_status(
        &self,
        _: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::Socks5Status>> {
        let status = self
            .send_and_wait(VpnServiceCommand::GetSocks5Status, ())
            .await?
            .map_err(|err| {
                tracing::error!("Failed to get SOCKS5 status: {err}");
                tonic::Status::internal(format!("Failed to get SOCKS5 status: {err}"))
            })?;

        // Convert from lib type to proto type using From trait
        let proto_status = proto::Socks5Status::from(status);

        Ok(tonic::Response::new(proto_status))
    }

    async fn get_privy_derivation_message(
        &self,
        _: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::PrivyDerivationMessage>> {
        Ok(tonic::Response::new(proto::PrivyDerivationMessage {
            message: nym_vpn_lib::login::privy::message_to_sign(),
        }))
    }
}

pub async fn start_command_interface(
    tunnel_event_rx: broadcast::Receiver<TunnelEvent>,
    shutdown_token: CancellationToken,
) -> std::io::Result<(JoinHandle<()>, UnboundedReceiver<VpnServiceCommand>)> {
    tracing::debug!("Starting command interface");

    let socket_path = default_socket_path();
    let (vpn_command_tx, vpn_command_rx) = mpsc::unbounded_channel();

    // Remove previous socket file in case if the daemon crashed in the prior run and could not clean up the socket file.
    #[cfg(unix)]
    remove_previous_socket_file(&socket_path).await;
    tracing::info!("Starting socket listener on: {}", socket_path.display());

    // Wrap the unix socket or named pipe into a stream that can be used by tonic
    let incoming = nym_ipc::server::create_incoming(socket_path.clone())?;

    let server_handle = tokio::spawn(async move {
        let incoming_shutdown_token = shutdown_token.child_token();
        let socket_listener_handle = tokio::spawn(async move {
            let command_interface = CommandInterface::new(vpn_command_tx, tunnel_event_rx);

            let server = Server::builder().add_service(NymVpnServiceServer::new(command_interface));

            match server
                .serve_with_incoming_shutdown(incoming, incoming_shutdown_token.cancelled_owned())
                .await
            {
                Ok(()) => {
                    tracing::info!("Socket listener has finished");
                }
                Err(e) => {
                    tracing::error!("Socket listener exited with error: {}", e);
                }
            }
        });

        if let Err(e) = socket_listener_handle.await {
            tracing::error!("Failed to join on socket listener: {}", e);
        }

        tracing::info!("Command interface exiting");
    });

    Ok((server_handle, vpn_command_rx))
}

#[cfg(unix)]
async fn remove_previous_socket_file(socket_path: &std::path::Path) {
    match tokio::fs::remove_file(socket_path).await {
        Ok(_) => tracing::info!(
            "Removed previous command interface socket: {}",
            socket_path.display()
        ),
        Err(err) if err.kind() == std::io::ErrorKind::NotFound => {}
        Err(err) => {
            tracing::error!(
                "Failed to remove previous command interface socket: {:?}",
                err
            );
        }
    }
}

fn default_socket_path() -> PathBuf {
    #[cfg(unix)]
    {
        PathBuf::from("/var/run/nym-vpn.sock")
    }

    #[cfg(windows)]
    {
        PathBuf::from(r"\\.\pipe\nym-vpn")
    }
}
