// Copyright 2025 - Nym Technologies SA <contact@nymtech.net>
// SPDX-License-Identifier: GPL-3.0-only

use nym_vpn_lib_types::{AccountCommandError, RegisterAccountResponse};
use nym_vpn_store::account::StorableAccount;

use nym_validator_client::nyxd::Coin;
use nym_vpn_api_client::{
    ResolverOverrides,
    response::{NymVpnDevice, NymVpnUsage},
    types::Platform,
};
use tokio::sync::oneshot;

use crate::AvailableTicketbooks;

#[derive(Debug, strum::Display)]
pub enum AccountCommand {
    /// Generate a mnemonic and store it
    CreateAccount(ReturnSender<(), AccountCommandError>),

    /// Store the given account
    StoreAccount(ReturnSender<(), AccountCommandError>, StorableAccount),

    /// Register the given account (meant to take the stored mnemonic). DOES NOT STORE IT. This is only used my mobile for IAP at the moment
    RegisterAccount(
        ReturnSender<RegisterAccountResponse, AccountCommandError>,
        StorableAccount,
        Platform,
    ),

    /// Delete the stored account and every associated data
    ForgetAccount(ReturnSender<(), AccountCommandError>),

    /// Rotate the wireguard keys
    RotateKeys(ReturnSender<(), AccountCommandError>),

    /// Retrieve current, on-chain, balance of the account. Only applicable for decentralised accounts
    AccountBalance(ReturnSender<Vec<Coin>, AccountCommandError>),

    /// Attempt to obtain specified amount of ticketbooks (per type) for the decentralised account
    ObtainTicketbooks(ReturnSender<(), AccountCommandError>, u64),

    /// Reset the device identity, optionally take a seed for reproducibility
    ResetDeviceIdentity(ReturnSender<(), AccountCommandError>, Option<[u8; 32]>),

    /// Forces the AC to sync with the VPN API
    RefreshAccountState(ReturnSender<(), AccountCommandError>),

    /// Tells the AC it's firewalled off the VPN API, so it should stop/pause network communication
    VpnApiFirewallUp(ReturnSender<(), AccountCommandError>),

    /// Tells the AC free to go ahead
    VpnApiFirewallDown(ReturnSender<(), AccountCommandError>),

    /// Upgrade mode-related commands
    UpgradeMode(UpgradeModeCommand),

    /// Read-only commands
    Common(CommonCommand),
}

impl AccountCommand {
    pub fn return_error(self, error: AccountCommandError) {
        match self {
            AccountCommand::CreateAccount(return_sender) => return_sender.send(Err(error)),
            AccountCommand::StoreAccount(return_sender, _) => return_sender.send(Err(error)),
            AccountCommand::RegisterAccount(return_sender, _, _) => return_sender.send(Err(error)),
            AccountCommand::ForgetAccount(return_sender) => return_sender.send(Err(error)),
            AccountCommand::RotateKeys(return_sender) => return_sender.send(Err(error)),
            AccountCommand::AccountBalance(return_sender) => return_sender.send(Err(error)),
            AccountCommand::ObtainTicketbooks(return_sender, _) => return_sender.send(Err(error)),
            AccountCommand::ResetDeviceIdentity(return_sender, _) => return_sender.send(Err(error)),
            AccountCommand::RefreshAccountState(return_sender) => return_sender.send(Err(error)),
            AccountCommand::VpnApiFirewallUp(return_sender) => return_sender.send(Err(error)),
            AccountCommand::VpnApiFirewallDown(return_sender) => return_sender.send(Err(error)),
            AccountCommand::Common(common_command) => match common_command {
                CommonCommand::GetStoredAccount(return_sender) => return_sender.send(Err(error)),
                CommonCommand::GetAccountIdentity(return_sender) => return_sender.send(Err(error)),
                CommonCommand::GetDeviceIdentity(return_sender) => return_sender.send(Err(error)),
                CommonCommand::GetUsage(return_sender) => return_sender.send(Err(error)),
                CommonCommand::GetDevices(return_sender) => return_sender.send(Err(error)),
                CommonCommand::GetActiveDevices(return_sender) => return_sender.send(Err(error)),
                CommonCommand::GetAvailableTickets(return_sender) => return_sender.send(Err(error)),
                CommonCommand::SetResolverOverrides(return_sender, _) => {
                    return_sender.send(Err(error))
                }
            },
            AccountCommand::UpgradeMode(upgrade_mode_command) => match upgrade_mode_command {
                UpgradeModeCommand::GetUpgradeModeEnabled(return_sender) => {
                    return_sender.send(Err(error))
                }
                UpgradeModeCommand::DisableUpgradeMode(return_sender) => {
                    return_sender.send(Err(error))
                }
            },
        }
    }
}

/// These commands have no impact on the state. Handling can be grouped in some cases
#[derive(Debug, strum::Display)]
pub enum CommonCommand {
    /// Returns Some(account) if an account is stored, None otherwise
    GetStoredAccount(ReturnSender<Option<StorableAccount>, AccountCommandError>),

    /// Returns Some(address) if an account is stored, None otherwise
    GetAccountIdentity(ReturnSender<Option<String>, AccountCommandError>),

    /// Returns Some(id) if the current device has an identity (is registered), None otherwise
    GetDeviceIdentity(ReturnSender<Option<String>, AccountCommandError>),

    /// Returns the state of the account
    GetUsage(ReturnSender<Vec<NymVpnUsage>, AccountCommandError>),

    /// Get the list of devices registered to that account
    GetDevices(ReturnSender<Vec<NymVpnDevice>, AccountCommandError>),

    /// Get the list of active devices registered to that account
    GetActiveDevices(ReturnSender<Vec<NymVpnDevice>, AccountCommandError>),

    /// Returns a list of tickets available in storage
    GetAvailableTickets(ReturnSender<AvailableTicketbooks, AccountCommandError>),

    /// Override the VPN API client resolver to allow him to go through the firewall (with Domain Fronting)
    SetResolverOverrides(
        ReturnSender<(), AccountCommandError>,
        Option<ResolverOverrides>,
    ),
}

/// Commands relating to the upgrade mode
#[derive(Debug, strum::Display)]
pub enum UpgradeModeCommand {
    /// Returns flag indicating whether the VPN API has informed us about the upgrade mode.
    /// this is known implicitly via the [AccountController](crate::controller::AccountController) being in the [UpgradeModeState](crate::state_machine::upgrade_mode_state::UpgradeModeState)
    GetUpgradeModeEnabled(ReturnSender<bool, AccountCommandError>),

    /// Inform the [AccountController](crate::controller::AccountController) about the upgrade mode being over
    /// to allow it to attempt to resume zk-nym acquisition.
    DisableUpgradeMode(ReturnSender<(), AccountCommandError>),
}

#[derive(Debug)]
pub struct ReturnSender<T, E> {
    sender: oneshot::Sender<Result<T, E>>,
}

impl<T, E> ReturnSender<T, E>
where
    T: std::fmt::Debug,
    E: std::fmt::Debug,
{
    pub fn new() -> (Self, oneshot::Receiver<Result<T, E>>) {
        let (sender, receiver) = oneshot::channel();
        (Self { sender }, receiver)
    }

    pub fn send(self, response: Result<T, E>)
    where
        T: Send,
        E: Send,
    {
        self.sender
            .send(response)
            .inspect_err(|err| {
                tracing::error!("Failed to send response: {:#?}", err);
            })
            .ok();
    }
}
