package net.nymtech.nymvpn.ui.screens.main.components

import androidx.compose.foundation.isSystemInDarkTheme
import androidx.compose.material3.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import net.nymtech.nymvpn.ui.common.animations.Pulse
import net.nymtech.nymvpn.ui.common.labels.PillLabel
import net.nymtech.nymvpn.ui.model.ConnectionState
import net.nymtech.nymvpn.ui.model.StateMessage
import net.nymtech.nymvpn.ui.theme.CustomColors
import net.nymtech.nymvpn.ui.theme.Theme
import net.nymtech.nymvpn.util.extensions.isVpnAlwaysOn
import nym_vpn_lib_types.ErrorStateReason

@Composable
fun ConnectionStateDisplay(connectionState: ConnectionState, stateMessage: StateMessage, theme: Theme) {
	val context = LocalContext.current
	val text = connectionState.status.asString(context)

	val isDarkMode = isSystemInDarkTheme()

	fun determinePillColor(lightColor: Color, darkColor: Color): Color {
		return when (theme) {
			Theme.AUTOMATIC, Theme.DYNAMIC -> if (isDarkMode) darkColor else lightColor
			Theme.DARK_MODE -> darkColor
			Theme.LIGHT_MODE -> lightColor
		}
	}

	when (connectionState) {
		ConnectionState.Connected ->
			PillLabel(
				text = text,
				backgroundColor = CustomColors.statusGreen,
				textColor = MaterialTheme.colorScheme.tertiary,
			)

		ConnectionState.Disconnected -> {
			if (stateMessage is StateMessage.Error && stateMessage.reason == ErrorStateReason.InactiveSubscription && isVpnAlwaysOn(context)) {
				PillLabel(
					text = "Subscription expired",
					backgroundColor = CustomColors.error,
					textColor = determinePillColor(CustomColors.errorStatusPillLight, CustomColors.errorStatusPillDark),
				)
			} else {
				PillLabel(
					text = text,
					backgroundColor = determinePillColor(CustomColors.statusDefaultLight, CustomColors.statusDefaultDark),
					textColor = MaterialTheme.colorScheme.onSecondary,
				)
			}
		}

		is ConnectionState.Connecting ->
			PillLabel(
				text = text,
				backgroundColor = determinePillColor(CustomColors.statusDefaultLight, CustomColors.statusDefaultDark),
				textColor = MaterialTheme.colorScheme.onBackground,
				trailing = { Pulse() },
			)

		ConnectionState.Disconnecting ->
			PillLabel(
				text = text,
				backgroundColor = determinePillColor(CustomColors.statusDefaultLight, CustomColors.statusDefaultDark),
				textColor = MaterialTheme.colorScheme.onBackground,
				trailing = { Pulse() },
			)

		ConnectionState.Offline -> PillLabel(
			text = text,
			backgroundColor = determinePillColor(CustomColors.statusRedLight, CustomColors.statusRed),
			textColor = MaterialTheme.colorScheme.onSurface,
		)

		ConnectionState.WaitingForConnection -> PillLabel(
			text = text,
			backgroundColor = determinePillColor(CustomColors.statusDefaultLight, CustomColors.statusDefaultDark),
			textColor = MaterialTheme.colorScheme.onBackground,
			trailing = { Pulse(color = CustomColors.error) },
		)
	}
}
