package net.nymtech.nymvpn.ui.screens.account.payment

import android.app.Activity
import android.content.res.Configuration
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.Font
import androidx.compose.ui.text.font.FontFamily
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.hilt.navigation.compose.hiltViewModel
import kotlinx.coroutines.delay
import net.nymtech.nymvpn.R
import net.nymtech.nymvpn.ui.AppUiState
import net.nymtech.nymvpn.ui.Route
import net.nymtech.nymvpn.ui.common.animations.PulsingDotsWave
import net.nymtech.nymvpn.ui.common.navigation.LocalNavController
import net.nymtech.nymvpn.ui.common.snackbar.SnackbarController
import net.nymtech.nymvpn.ui.theme.CustomColors
import net.nymtech.nymvpn.ui.theme.NymVPNTheme
import net.nymtech.nymvpn.ui.theme.Theme
import net.nymtech.nymvpn.ui.theme.Typography
import net.nymtech.nymvpn.util.StringValue
import net.nymtech.nymvpn.util.extensions.navigateAndForget
import net.nymtech.nymvpn.util.extensions.replaceCurrentWith
import nym_vpn_lib_types.AccountControllerState
import timber.log.Timber

@Composable
fun PaymentScreen(appUiState: AppUiState, productId: String, viewModel: PaymentViewModel = hiltViewModel()) {
	val navController = LocalNavController.current
	val context = LocalContext.current
	val activity = context as? Activity
	val userId = appUiState.managerState.accountId

	var animationStart by rememberSaveable { mutableStateOf(false) }
	var latestEvent by remember { mutableStateOf<PaymentUiEvent?>(null) }
	val accountState by viewModel.accountState.collectAsState()

	LaunchedEffect(activity, productId) {
		activity?.let {
			viewModel.startPurchaseFlow(it, productId, userId)
		}
	}

	LaunchedEffect(Unit) {
		viewModel.events.collect { event ->
			latestEvent = event
			when (event) {
				is PaymentUiEvent.PaymentError, is PaymentUiEvent.UserCanceled -> {
					SnackbarController.showMessage(StringValue.StringResource(R.string.account_payment_error))
					navController.replaceCurrentWith(Route.SelectPlan)
				}
				is PaymentUiEvent.PaymentSuccess -> {
					animationStart = true
					viewModel.refreshAccountState()
				}
				is PaymentUiEvent.SubscriptionOwned -> {
					navController.replaceCurrentWith(Route.Main())
				}
				PaymentUiEvent.PaymentPending -> {
					animationStart = true
				}
			}
		}
	}

	PaymentScreen(
		start = animationStart,
		accountState = accountState,
		onAnimationEnd = {
			navController.navigateAndForget(Route.Main())
		},
	)
}

@Composable
fun PaymentScreen(accountState: AccountControllerState?, start: Boolean, onAnimationEnd: () -> Unit) {
	val backgroundColor =
		if (accountState == AccountControllerState.ReadyToConnect || accountState == AccountControllerState.RequestingZkNyms) {
			MaterialTheme.colorScheme.primary
		} else {
			MaterialTheme.colorScheme.surfaceContainer
		}

	var textData by remember {
		mutableStateOf(
			Pair(
				R.string.account_payment_processing,
				R.string.account_payment_verifying,
			),
		)
	}

	LaunchedEffect(start, accountState) {
		Timber.d("accountState UI: $accountState")
		if (!start || accountState == null) return@LaunchedEffect

		when (accountState) {
			is AccountControllerState.ReadyToConnect -> {
				textData = Pair(
					R.string.account_payment_welcome,
					R.string.account_payment_protected,
				)
				delay(1_000)
				onAnimationEnd()
			}
			is AccountControllerState.RequestingZkNyms -> {
				textData = Pair(
					R.string.account_payment_saving,
					R.string.account_payment_anonymous,
				)
			}
			is AccountControllerState.Syncing -> {
				textData = Pair(
					R.string.account_payment_retrieving,
					R.string.account_payment_credentials,
				)
			}
			else -> {
			}
		}
	}

	Column(
		modifier = Modifier
			.fillMaxSize()
			.background(MaterialTheme.colorScheme.background),
		horizontalAlignment = Alignment.CenterHorizontally,
	) {
		Row(
			modifier = Modifier
				.padding(horizontal = 16.dp)
				.fillMaxWidth()
				.height(4.dp),
			horizontalArrangement = Arrangement.spacedBy(4.dp),
		) {
			Box(
				modifier = Modifier
					.weight(1f)
					.fillMaxHeight()
					.background(
						MaterialTheme.colorScheme.primary,
						shape = RoundedCornerShape(size = 4.dp),
					),
			)
			Box(
				modifier = Modifier
					.weight(1f)
					.fillMaxHeight()
					.background(
						backgroundColor,
						shape = RoundedCornerShape(size = 4.dp),
					),
			)
		}

		Column(
			horizontalAlignment = Alignment.CenterHorizontally,
			modifier = Modifier.padding(top = 200.dp),
		) {
			Box(
				modifier = Modifier
					.size(56.dp)
					.background(
						color = CustomColors.iconBackground,
						shape = RoundedCornerShape(size = 8.dp),
					)
					.border(
						width = 1.dp,
						color = CustomColors.iconBorder,
						shape = RoundedCornerShape(size = 8.dp),
					),
			) {
				PulsingDotsWave(
					modifier = Modifier
						.align(Alignment.Center)
						.padding(8.dp),
				)
			}

			Text(
				text = stringResource(textData.first),
				style = Typography.titleMedium,
				textAlign = TextAlign.Center,
				color = MaterialTheme.colorScheme.onBackground,
				modifier = Modifier.padding(top = 16.dp, start = 40.dp, end = 40.dp),
				fontFamily = FontFamily(Font(R.font.lab_grotesque_regular)),
			)

			Text(
				text = stringResource(textData.second),
				style = Typography.bodyMedium,
				textAlign = TextAlign.Center,
				modifier = Modifier.padding(top = 16.dp, start = 40.dp, end = 40.dp),
				color = MaterialTheme.colorScheme.outline,
				fontFamily = FontFamily(Font(R.font.lab_grotesque_regular)),
			)
		}
	}
}

@Preview(uiMode = Configuration.UI_MODE_NIGHT_YES, showBackground = true)
@Composable
private fun PreviewPaymentScreen() {
	NymVPNTheme(Theme.default()) {
		PaymentScreen(accountState = AccountControllerState.RequestingZkNyms, start = true, onAnimationEnd = {})
	}
}
