// Copyright 2024 - Nym Technologies SA <contact@nymtech.net>
// SPDX-License-Identifier: GPL-3.0-only

use std::path::PathBuf;

use futures::{StreamExt, stream::BoxStream};
use tokio::{
    sync::{
        broadcast,
        mpsc::{self, UnboundedReceiver, UnboundedSender},
        oneshot,
    },
    task::JoinHandle,
};
use tokio_util::sync::CancellationToken;
use tonic::transport::Server;

use nym_vpn_lib_types::TunnelEvent;
use nym_vpn_proto::proto::{
    self,
    nym_vpn_service_server::{NymVpnService, NymVpnServiceServer},
};
use nym_vpnd_types::{ConnectArgs, ListCountriesOptions, ListGatewaysOptions};

use crate::service::{SetNetworkError, VpnServiceCommand};

pub type Result<T> = std::result::Result<T, tonic::Status>;

pub struct CommandInterface {
    // Send commands to the VPN service
    vpn_command_tx: UnboundedSender<VpnServiceCommand>,

    // Broadcast tunnel events to our API endpoint listeners
    tunnel_event_rx: broadcast::Receiver<TunnelEvent>,
}

impl CommandInterface {
    fn new(
        vpn_command_tx: UnboundedSender<VpnServiceCommand>,
        tunnel_event_rx: broadcast::Receiver<TunnelEvent>,
    ) -> Self {
        Self {
            vpn_command_tx,
            tunnel_event_rx,
        }
    }

    async fn send_and_wait<R, F, O>(&self, command: F, opts: O) -> Result<R>
    where
        F: FnOnce(oneshot::Sender<R>, O) -> VpnServiceCommand,
    {
        let (tx, rx) = oneshot::channel();

        self.vpn_command_tx
            .send(command(tx, opts))
            .map_err(|_| tonic::Status::internal("Command channel is closed"))?;

        rx.await
            .map_err(|_| tonic::Status::internal("Response channel is closed"))
    }
}

#[tonic::async_trait]
impl NymVpnService for CommandInterface {
    async fn info(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::InfoResponse>> {
        let info = self.send_and_wait(VpnServiceCommand::Info, ()).await?;
        let response = proto::InfoResponse::from(info);

        Ok(tonic::Response::new(response))
    }

    async fn set_network(&self, request: tonic::Request<String>) -> Result<tonic::Response<()>> {
        let network = request.into_inner();
        let status = self
            .send_and_wait(VpnServiceCommand::SetNetwork, network)
            .await?;

        status.map_err(|e| match e {
            SetNetworkError::NetworkNotFound(network_name) => {
                tonic::Status::not_found(format!("Network not found: {network_name}"))
            }
            e => tonic::Status::internal(e.to_string()),
        })?;

        Ok(tonic::Response::new(()))
    }

    async fn get_system_messages(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetSystemMessagesResponse>> {
        let messages = self
            .send_and_wait(VpnServiceCommand::GetSystemMessages, ())
            .await?;

        let messages = messages.into_current_iter().map(|m| m.into()).collect();
        let response = proto::GetSystemMessagesResponse { messages };

        Ok(tonic::Response::new(response))
    }

    async fn get_network_compatibility(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetNetworkCompatibilityResponse>> {
        let network_compatibility = self
            .send_and_wait(VpnServiceCommand::GetNetworkCompatibility, ())
            .await?
            .map(proto::NetworkCompatibility::from);

        let response = proto::GetNetworkCompatibilityResponse {
            network_compatibility,
        };

        Ok(tonic::Response::new(response))
    }

    async fn get_feature_flags(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetFeatureFlagsResponse>> {
        let feature_flags = self
            .send_and_wait(VpnServiceCommand::GetFeatureFlags, ())
            .await?
            .ok_or(tonic::Status::not_found("Feature flags not found"))?;

        Ok(tonic::Response::new(feature_flags.into()))
    }

    async fn connect_tunnel(
        &self,
        request: tonic::Request<proto::ConnectRequest>,
    ) -> Result<tonic::Response<()>> {
        let connect_args = ConnectArgs::try_from(request.into_inner())
            .map_err(|err| tonic::Status::invalid_argument(err.to_string()))?;

        self.send_and_wait(VpnServiceCommand::Connect, connect_args)
            .await?;

        Ok(tonic::Response::new(()))
    }

    async fn disconnect_tunnel(&self, _request: tonic::Request<()>) -> Result<tonic::Response<()>> {
        self.send_and_wait(VpnServiceCommand::Disconnect, ())
            .await?;

        Ok(tonic::Response::new(()))
    }

    async fn get_tunnel_state(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::TunnelState>> {
        let tunnel_state = self
            .send_and_wait(VpnServiceCommand::GetTunnelState, ())
            .await
            .map(proto::TunnelState::from)?;

        Ok(tonic::Response::new(tunnel_state))
    }

    type ListenToTunnelStateStream = BoxStream<'static, Result<proto::TunnelState>>;
    async fn listen_to_tunnel_state(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<Self::ListenToTunnelStateStream>> {
        let rx = self
            .send_and_wait(VpnServiceCommand::SubscribeToTunnelState, ())
            .await?;
        let stream = tokio_stream::wrappers::BroadcastStream::new(rx).map(|new_state| {
            new_state
                .map(proto::TunnelState::from)
                .map_err(|_| tonic::Status::internal("Failed to receive tunnel state"))
        });
        Ok(tonic::Response::new(
            Box::pin(stream) as Self::ListenToTunnelStateStream
        ))
    }

    type ListenToEventsStream = BoxStream<'static, Result<proto::TunnelEvent>>;
    async fn listen_to_events(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<Self::ListenToEventsStream>> {
        let rx = self.tunnel_event_rx.resubscribe();
        let stream = tokio_stream::wrappers::BroadcastStream::new(rx).map(|event| {
            event
                .map(proto::TunnelEvent::from)
                .map_err(|_| tonic::Status::internal("Failed to receive tunnel event"))
        });
        Ok(tonic::Response::new(
            Box::pin(stream) as Self::ListenToEventsStream
        ))
    }

    async fn list_gateways(
        &self,
        request: tonic::Request<proto::ListGatewaysRequest>,
    ) -> Result<tonic::Response<proto::ListGatewaysResponse>> {
        let options = ListGatewaysOptions::try_from(request.into_inner())
            .map_err(|err| tonic::Status::invalid_argument(err.to_string()))?;

        let gateways = self
            .send_and_wait(VpnServiceCommand::ListGateways, options)
            .await?
            .map_err(|err| tonic::Status::internal(format!("Failed to list gateways: {err}")))?;

        let response = proto::ListGatewaysResponse {
            gateways: gateways
                .into_iter()
                .map(proto::GatewayResponse::from)
                .collect(),
        };
        Ok(tonic::Response::new(response))
    }

    async fn list_countries(
        &self,
        request: tonic::Request<proto::ListCountriesRequest>,
    ) -> Result<tonic::Response<proto::ListCountriesResponse>> {
        let options = ListCountriesOptions::try_from(request.into_inner())
            .map_err(|err| tonic::Status::invalid_argument(err.to_string()))?;

        let countries = self
            .send_and_wait(VpnServiceCommand::ListCountries, options)
            .await?
            .map_err(|err| {
                tonic::Status::internal(format!("Failed to list entry countries: {err}"))
            })?;

        let response = proto::ListCountriesResponse {
            countries: countries.into_iter().map(proto::Location::from).collect(),
        };

        Ok(tonic::Response::new(response))
    }

    async fn store_account(
        &self,
        request: tonic::Request<proto::StoreAccountRequest>,
    ) -> Result<tonic::Response<proto::AccountCommandResponse>> {
        let store_request = nym_vpnd_types::StoreAccountRequest::from(request.into_inner());
        let result = self
            .send_and_wait(VpnServiceCommand::StoreAccount, store_request)
            .await?;

        let response = proto::AccountCommandResponse {
            error: result.err().map(proto::AccountCommandError::from),
        };

        Ok(tonic::Response::new(response))
    }

    async fn is_account_stored(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<bool>> {
        let is_stored = self
            .send_and_wait(VpnServiceCommand::IsAccountStored, ())
            .await?;

        Ok(tonic::Response::new(is_stored))
    }

    async fn forget_account(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::AccountCommandResponse>> {
        let result = self
            .send_and_wait(VpnServiceCommand::ForgetAccount, ())
            .await?;

        let response = proto::AccountCommandResponse {
            error: result.err().map(proto::AccountCommandError::from),
        };

        Ok(tonic::Response::new(response))
    }

    async fn get_account_identity(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetAccountIdentityResponse>> {
        let account_identity = self
            .send_and_wait(VpnServiceCommand::GetAccountIdentity, ())
            .await?
            .map_err(|err| {
                tonic::Status::internal(format!("Failed to get account identity: {err}"))
            })?;

        Ok(tonic::Response::new(proto::GetAccountIdentityResponse {
            account_identity,
        }))
    }

    async fn get_account_links(
        &self,
        request: tonic::Request<proto::GetAccountLinksRequest>,
    ) -> Result<tonic::Response<proto::AccountManagement>> {
        let locale = request.into_inner().locale;

        let account_links = self
            .send_and_wait(VpnServiceCommand::GetAccountLinks, locale)
            .await?
            .map_err(|err| {
                tonic::Status::internal(format!("Failed to get account links: {err}"))
            })?;

        Ok(tonic::Response::new(proto::AccountManagement::from(
            account_links,
        )))
    }

    async fn get_account_state(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::AccountControllerState>> {
        let account_controller_state = self
            .send_and_wait(VpnServiceCommand::GetAccountState, ())
            .await
            .map(proto::AccountControllerState::from)?;

        Ok(tonic::Response::new(account_controller_state))
    }

    type ListenToAccountStateStream = BoxStream<'static, Result<proto::AccountControllerState>>;
    async fn listen_to_account_state(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<Self::ListenToAccountStateStream>> {
        let rx = self
            .send_and_wait(VpnServiceCommand::SubscribeToAccountControllerState, ())
            .await?;
        let stream = tokio_stream::wrappers::WatchStream::new(rx)
            .map(|new_state| Ok(proto::AccountControllerState::from(new_state)));
        Ok(tonic::Response::new(
            Box::pin(stream) as Self::ListenToAccountStateStream
        ))
    }

    async fn refresh_account_state(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<()>> {
        self.send_and_wait(VpnServiceCommand::RefreshAccountState, ())
            .await?;

        Ok(tonic::Response::new(()))
    }

    async fn get_account_usage(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetAccountUsageResponse>> {
        let account_usage = self
            .send_and_wait(VpnServiceCommand::GetAccountUsage, ())
            .await?
            .map_err(|err| {
                tonic::Status::internal(format!("Failed to get account usage: {err}"))
            })?;

        Ok(tonic::Response::new(proto::GetAccountUsageResponse {
            account_usages: Some(proto::get_account_usage_response::AccountUsages::from(
                account_usage,
            )),
        }))
    }

    async fn reset_device_identity(
        &self,
        request: tonic::Request<proto::ResetDeviceIdentityRequest>,
    ) -> Result<tonic::Response<()>> {
        let seed: Option<[u8; 32]> = request
            .into_inner()
            .seed
            .map(|seed| {
                seed.as_slice()
                    .try_into()
                    .map_err(|_| tonic::Status::invalid_argument("Seed must be 32 bytes long"))
            })
            .transpose()?;

        self.send_and_wait(VpnServiceCommand::ResetDeviceIdentity, seed)
            .await?
            .map_err(|err| {
                tonic::Status::internal(format!("Failed to reset device identity: {err}"))
            })?;

        Ok(tonic::Response::new(()))
    }

    async fn get_device_identity(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetDeviceIdentityResponse>> {
        let device_identity = self
            .send_and_wait(VpnServiceCommand::GetDeviceIdentity, ())
            .await?
            .map_err(|err| {
                tonic::Status::internal(format!("Failed to get device identity: {err}"))
            })?;

        Ok(tonic::Response::new(proto::GetDeviceIdentityResponse {
            device_identity,
        }))
    }

    async fn get_devices(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetDevicesResponse>> {
        let devices = self
            .send_and_wait(VpnServiceCommand::GetDevices, ())
            .await?
            .map_err(|err| tonic::Status::internal(format!("Failed to get devices: {err}")))?;

        Ok(tonic::Response::new(proto::GetDevicesResponse {
            devices: Some(proto::get_devices_response::Devices::from(devices)),
        }))
    }

    async fn get_active_devices(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetDevicesResponse>> {
        let devices = self
            .send_and_wait(VpnServiceCommand::GetActiveDevices, ())
            .await?
            .map_err(|err| {
                tonic::Status::internal(format!("Failed to get active devices: {err}"))
            })?;

        Ok(tonic::Response::new(proto::GetDevicesResponse {
            devices: Some(proto::get_devices_response::Devices::from(devices)),
        }))
    }
    async fn get_available_tickets(
        &self,
        _request: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::AvailableTickets>> {
        let available_ticketbooks = self
            .send_and_wait(VpnServiceCommand::GetAvailableTickets, ())
            .await?
            .map_err(|err| {
                tonic::Status::internal(format!("Failed to get available tickets: {err}"))
            })?;

        let available_tickets = nym_vpn_lib_types::AvailableTickets::from(available_ticketbooks);
        let response = proto::AvailableTickets::from(available_tickets);

        Ok(tonic::Response::new(response))
    }

    async fn delete_log_file(&self, _request: tonic::Request<()>) -> Result<tonic::Response<()>> {
        self.send_and_wait(VpnServiceCommand::DeleteLogFile, ())
            .await?;

        Ok(tonic::Response::new(()))
    }

    async fn get_log_path(
        &self,
        _: tonic::Request<()>,
    ) -> Result<tonic::Response<proto::GetLogPathResponse>> {
        let log_path = self
            .send_and_wait(VpnServiceCommand::GetLogPath, ())
            .await?
            .unwrap_or(crate::logging::default_log_path());

        Ok(tonic::Response::new(log_path.try_into().map_err(
            |err| tonic::Status::internal(format!("Failed to obtain log path: {err}")),
        )?))
    }

    async fn is_sentry_enabled(&self, _: tonic::Request<()>) -> Result<tonic::Response<bool>> {
        let result = self
            .send_and_wait(VpnServiceCommand::IsSentryEnabled, ())
            .await?;
        Ok(tonic::Response::new(result))
    }

    async fn enable_sentry(&self, _: tonic::Request<()>) -> Result<tonic::Response<()>> {
        self.send_and_wait(VpnServiceCommand::ToggleSentry, true)
            .await?
            .map_err(|err| {
                tracing::error!("Failed to enable sentry monitoring: {err}");
                tonic::Status::internal("failed to enable sentry")
            })?;
        Ok(tonic::Response::new(()))
    }

    async fn disable_sentry(&self, _: tonic::Request<()>) -> Result<tonic::Response<()>> {
        self.send_and_wait(VpnServiceCommand::ToggleSentry, false)
            .await?
            .map_err(|err| {
                tracing::error!("Failed to disable sentry monitoring: {err}");
                tonic::Status::internal("failed to disable sentry")
            })?;
        Ok(tonic::Response::new(()))
    }

    async fn is_collect_network_stats_enabled(
        &self,
        _: tonic::Request<()>,
    ) -> std::result::Result<tonic::Response<bool>, tonic::Status> {
        let result = self
            .send_and_wait(VpnServiceCommand::IsCollectNetStatsEnabled, ())
            .await?;
        Ok(tonic::Response::new(result))
    }

    async fn enable_collect_network_stats(
        &self,
        _: tonic::Request<()>,
    ) -> std::result::Result<tonic::Response<()>, tonic::Status> {
        self.send_and_wait(VpnServiceCommand::ToggleCollectNetStats, true)
            .await?
            .map_err(|err| {
                tracing::error!("Failed to enable collect network stats: {err}");
                tonic::Status::internal("failed to enable collect network stats")
            })?;
        Ok(tonic::Response::new(()))
    }

    async fn disable_collect_network_stats(
        &self,
        _: tonic::Request<()>,
    ) -> std::result::Result<tonic::Response<()>, tonic::Status> {
        self.send_and_wait(VpnServiceCommand::ToggleCollectNetStats, false)
            .await?
            .map_err(|err| {
                tracing::error!("Failed to disable collect network stats: {err}");
                tonic::Status::internal("failed to disable collect network stats")
            })?;
        Ok(tonic::Response::new(()))
    }
}

pub async fn start_command_interface(
    tunnel_event_rx: broadcast::Receiver<TunnelEvent>,
    shutdown_token: CancellationToken,
) -> std::io::Result<(JoinHandle<()>, UnboundedReceiver<VpnServiceCommand>)> {
    tracing::debug!("Starting command interface");

    let socket_path = default_socket_path();
    let (vpn_command_tx, vpn_command_rx) = mpsc::unbounded_channel();

    // Remove previous socket file in case if the daemon crashed in the prior run and could not clean up the socket file.
    #[cfg(unix)]
    remove_previous_socket_file(&socket_path).await;
    tracing::info!("Starting socket listener on: {}", socket_path.display());

    // Wrap the unix socket or named pipe into a stream that can be used by tonic
    let incoming = nym_ipc::server::create_incoming(socket_path.clone())?;

    let server_handle = tokio::spawn(async move {
        let incoming_shutdown_token = shutdown_token.child_token();
        let socket_listener_handle = tokio::spawn(async move {
            let command_interface = CommandInterface::new(vpn_command_tx, tunnel_event_rx);

            let server = Server::builder().add_service(NymVpnServiceServer::new(command_interface));

            match server
                .serve_with_incoming_shutdown(incoming, incoming_shutdown_token.cancelled_owned())
                .await
            {
                Ok(()) => {
                    tracing::info!("Socket listener has finished");
                }
                Err(e) => {
                    tracing::error!("Socket listener exited with error: {}", e);
                }
            }
        });

        if let Err(e) = socket_listener_handle.await {
            tracing::error!("Failed to join on socket listener: {}", e);
        }

        tracing::info!("Command interface exiting");
    });

    Ok((server_handle, vpn_command_rx))
}

#[cfg(unix)]
async fn remove_previous_socket_file(socket_path: &std::path::Path) {
    match tokio::fs::remove_file(socket_path).await {
        Ok(_) => tracing::info!(
            "Removed previous command interface socket: {}",
            socket_path.display()
        ),
        Err(err) if err.kind() == std::io::ErrorKind::NotFound => {}
        Err(err) => {
            tracing::error!(
                "Failed to remove previous command interface socket: {:?}",
                err
            );
        }
    }
}

fn default_socket_path() -> PathBuf {
    #[cfg(unix)]
    {
        PathBuf::from("/var/run/nym-vpn.sock")
    }

    #[cfg(windows)]
    {
        PathBuf::from(r"\\.\pipe\nym-vpn")
    }
}
