// Copyright 2024 - Nym Technologies SA <contact@nymtech.net>
// SPDX-License-Identifier: GPL-3.0-only

use tokio::sync::Mutex;

use super::{MnemonicStorage, StoredMnemonic};

#[derive(Default)]
pub struct InMemoryMnemonicStorage {
    mnemonic: Mutex<Option<StoredMnemonic>>,
}

#[derive(Debug, thiserror::Error)]
pub enum InMemoryMnemonicStorageError {
    #[error("no mnemonic stored")]
    NoMnemonicStored,

    #[error("mnemonic already stored")]
    MnemonicAlreadyStored,
}

#[async_trait::async_trait]
impl MnemonicStorage for InMemoryMnemonicStorage {
    type StorageError = InMemoryMnemonicStorageError;

    async fn store_mnemonic(
        &self,
        mnemonic: bip39::Mnemonic,
    ) -> Result<(), InMemoryMnemonicStorageError> {
        let name = "default".to_string();
        let nonce = 0;
        let stored_mnemonic = StoredMnemonic {
            name,
            mnemonic,
            nonce,
        };
        let mut handle = self.mnemonic.lock().await;

        // Store the mnemonic if it's currently None
        if handle.is_none() {
            *handle = Some(stored_mnemonic);
            Ok(())
        } else {
            Err(InMemoryMnemonicStorageError::MnemonicAlreadyStored)
        }
    }

    async fn load_mnemonic(&self) -> Result<Option<bip39::Mnemonic>, InMemoryMnemonicStorageError> {
        Ok(self
            .mnemonic
            .lock()
            .await
            .as_ref()
            .map(|stored| stored.mnemonic.clone()))
    }

    async fn remove_mnemonic(&self) -> Result<(), InMemoryMnemonicStorageError> {
        let mut handle = self.mnemonic.lock().await;

        if handle.is_some() {
            *handle = None;
            Ok(())
        } else {
            Err(InMemoryMnemonicStorageError::NoMnemonicStored)
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[tokio::test]
    async fn store_and_load_mnemonic() {
        let mnemonic = "kiwi ketchup mix canvas curve ribbon congress method feel frozen act annual aunt comfort side joy mesh palace tennis cannon orange name tortoise piece";
        let mnemonic = bip39::Mnemonic::parse(mnemonic).unwrap();

        let storage = InMemoryMnemonicStorage::default();
        storage.store_mnemonic(mnemonic.clone()).await.unwrap();

        let loaded_mnemonic = storage.load_mnemonic().await.unwrap();
        assert_eq!(loaded_mnemonic, Some(mnemonic));
    }

    #[tokio::test]
    async fn load_non_existing_mnemonic_returns_none() {
        let storage = InMemoryMnemonicStorage::default();

        let loaded_mnemonic = storage.load_mnemonic().await.unwrap();
        assert_eq!(loaded_mnemonic, None);
    }
}
