// Copyright 2025 - Nym Technologies SA <contact@nymtech.net>
// SPDX-License-Identifier: GPL-3.0-only

use std::sync::Arc;

use nym_vpn_lib_types::{AccountCommandError, AvailableTickets, VpnApiError, VpnApiErrorResponse};

use crate::{conversions::ConversionError, proto};

impl TryFrom<proto::AccountCommandError> for AccountCommandError {
    type Error = ConversionError;

    fn try_from(value: proto::AccountCommandError) -> Result<Self, Self::Error> {
        let error_detail = value.error_detail.ok_or(ConversionError::NoValueSet(
            "StoreAccountError.error_detail",
        ))?;
        Ok(match error_detail {
            proto::account_command_error::ErrorDetail::StorageError(err) => Self::Storage(err),
            proto::account_command_error::ErrorDetail::Internal(err) => Self::Internal(err),
            proto::account_command_error::ErrorDetail::VpnApi(vpn_api) => {
                Self::VpnApi(vpn_api.try_into()?)
            }
            proto::account_command_error::ErrorDetail::UnexpectedResponse(err) => {
                Self::UnexpectedVpnApiResponse(err)
            }
            proto::account_command_error::ErrorDetail::NoAccountStored(_) => Self::NoAccountStored,
            proto::account_command_error::ErrorDetail::NoDeviceStored(_) => Self::NoDeviceStored,
            proto::account_command_error::ErrorDetail::ExistingAccount(_) => Self::ExistingAccount,
            proto::account_command_error::ErrorDetail::Offline(_) => Self::Offline,
            proto::account_command_error::ErrorDetail::InvalidMnemonic(message) => {
                Self::InvalidMnemonic(message)
            }
        })
    }
}

// We don't pass the source error across grpc, so on the recipient it will be empty. That's OK.
#[derive(Debug, Clone, PartialEq, Eq, thiserror::Error)]
#[error("(empty)")]
struct EmptyError;

impl TryFrom<proto::VpnApiError> for VpnApiError {
    type Error = ConversionError;

    fn try_from(value: proto::VpnApiError) -> Result<Self, Self::Error> {
        let error_detail = value
            .error_detail
            .ok_or(ConversionError::NoValueSet("VpnApiError.error_detail"))?;
        Ok(match error_detail {
            proto::vpn_api_error::ErrorDetail::Timeout(_) => Self::Timeout(Arc::new(EmptyError)),
            proto::vpn_api_error::ErrorDetail::StatusCode(code) => Self::StatusCode {
                code: code.try_into().map_err(ConversionError::generic)?,
                source: Arc::new(EmptyError),
            },
            proto::vpn_api_error::ErrorDetail::Response(vpn_api_error_response) => {
                Self::Response(vpn_api_error_response.into())
            }
        })
    }
}

impl From<proto::VpnApiErrorResponse> for VpnApiErrorResponse {
    fn from(value: proto::VpnApiErrorResponse) -> Self {
        Self {
            message: value.message,
            message_id: value.message_id,
            code_reference_id: value.code_reference_id,
        }
    }
}

impl From<AccountCommandError> for proto::AccountCommandError {
    fn from(value: AccountCommandError) -> Self {
        match value {
            AccountCommandError::Internal(err) => proto::AccountCommandError {
                error_detail: Some(proto::account_command_error::ErrorDetail::Internal(err)),
            },
            AccountCommandError::Storage(err) => proto::AccountCommandError {
                error_detail: Some(proto::account_command_error::ErrorDetail::StorageError(err)),
            },
            AccountCommandError::VpnApi(vpn_api_error) => proto::AccountCommandError {
                error_detail: Some(proto::account_command_error::ErrorDetail::VpnApi(
                    vpn_api_error.into(),
                )),
            },
            AccountCommandError::UnexpectedVpnApiResponse(err) => proto::AccountCommandError {
                error_detail: Some(
                    proto::account_command_error::ErrorDetail::UnexpectedResponse(err),
                ),
            },
            AccountCommandError::NoAccountStored => proto::AccountCommandError {
                error_detail: Some(proto::account_command_error::ErrorDetail::NoAccountStored(
                    true,
                )),
            },
            AccountCommandError::NoDeviceStored => proto::AccountCommandError {
                error_detail: Some(proto::account_command_error::ErrorDetail::NoDeviceStored(
                    true,
                )),
            },
            AccountCommandError::ExistingAccount => proto::AccountCommandError {
                error_detail: Some(proto::account_command_error::ErrorDetail::ExistingAccount(
                    true,
                )),
            },
            AccountCommandError::Offline => proto::AccountCommandError {
                error_detail: Some(proto::account_command_error::ErrorDetail::Offline(true)),
            },
            AccountCommandError::InvalidMnemonic(err) => proto::AccountCommandError {
                error_detail: Some(proto::account_command_error::ErrorDetail::InvalidMnemonic(
                    err,
                )),
            },
        }
    }
}

impl From<AvailableTickets> for proto::AvailableTickets {
    fn from(ticketbooks: AvailableTickets) -> Self {
        Self {
            mixnet_entry_tickets: ticketbooks.mixnet_entry_tickets,
            mixnet_entry_data: ticketbooks.mixnet_entry_data,
            mixnet_entry_data_si: ticketbooks.mixnet_entry_data_si,
            mixnet_exit_tickets: ticketbooks.mixnet_exit_tickets,
            mixnet_exit_data: ticketbooks.mixnet_exit_data,
            mixnet_exit_data_si: ticketbooks.mixnet_exit_data_si,
            vpn_entry_tickets: ticketbooks.vpn_entry_tickets,
            vpn_entry_data: ticketbooks.vpn_entry_data,
            vpn_entry_data_si: ticketbooks.vpn_entry_data_si,
            vpn_exit_tickets: ticketbooks.vpn_exit_tickets,
            vpn_exit_data: ticketbooks.vpn_exit_data,
            vpn_exit_data_si: ticketbooks.vpn_exit_data_si,
        }
    }
}

impl From<proto::AvailableTickets> for AvailableTickets {
    fn from(ticketbooks: proto::AvailableTickets) -> Self {
        Self {
            mixnet_entry_tickets: ticketbooks.mixnet_entry_tickets,
            mixnet_entry_data: ticketbooks.mixnet_entry_data,
            mixnet_entry_data_si: ticketbooks.mixnet_entry_data_si,
            mixnet_exit_tickets: ticketbooks.mixnet_exit_tickets,
            mixnet_exit_data: ticketbooks.mixnet_exit_data,
            mixnet_exit_data_si: ticketbooks.mixnet_exit_data_si,
            vpn_entry_tickets: ticketbooks.vpn_entry_tickets,
            vpn_entry_data: ticketbooks.vpn_entry_data,
            vpn_entry_data_si: ticketbooks.vpn_entry_data_si,
            vpn_exit_tickets: ticketbooks.vpn_exit_tickets,
            vpn_exit_data: ticketbooks.vpn_exit_data,
            vpn_exit_data_si: ticketbooks.vpn_exit_data_si,
        }
    }
}

impl From<VpnApiError> for proto::VpnApiError {
    fn from(value: VpnApiError) -> Self {
        let error_detail = match value {
            VpnApiError::Timeout(..) => proto::vpn_api_error::ErrorDetail::Timeout(true),
            VpnApiError::StatusCode { code, .. } => {
                proto::vpn_api_error::ErrorDetail::StatusCode(code.into())
            }
            VpnApiError::Response(vpn_api_error_response) => {
                proto::vpn_api_error::ErrorDetail::Response(vpn_api_error_response.into())
            }
        };
        Self {
            error_detail: Some(error_detail),
        }
    }
}

impl From<VpnApiErrorResponse> for proto::VpnApiErrorResponse {
    fn from(value: VpnApiErrorResponse) -> Self {
        Self {
            message: value.message,
            message_id: value.message_id,
            code_reference_id: value.code_reference_id,
        }
    }
}
